// sgi_dac.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <math.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>

#include "sgi_dac.h"
#include "application.h"
#include "statusaction.h"
#include "typeconvert.h"

#undef DEBUG_SGI

boolean SGIConverter::dontInterrupt = false;
int SGIConverter::inputDevice = AL_DEFAULT_INPUT;
int SGIConverter::outputDevice = AL_DEFAULT_OUTPUT;
int SGIConverter::inputDeviceCount = 0;
int SGIConverter::outputDeviceCount = 0;

SGIConverter::SGIConverter()
	: inPortConfig(nil), outPortConfig(nil), audio_port(nil) {
    if(initialize())
        catchSignals(true);
    else
        fail();
}

SGIConverter::~SGIConverter() {
    resetDefaults();
    if(inPortConfig)
        alFreeConfig(inPortConfig);
    if(outPortConfig)
        alFreeConfig(outPortConfig);
    if(audio_port)
        alClosePort(audio_port);
    catchSignals(false);
}

int
SGIConverter::inputDeviceToIndex(int deviceID) const {
    int value = -1;
    for (int i = 0; i < inputDeviceCount; i++) {
	if (deviceID == inputDevices[i].resourceID) {
	    value = i;
	    break;
	}
    }
    return value;
}

int
SGIConverter::outputDeviceToIndex(int deviceID) const {
    int value = -1;
    for (int i = 0; i < outputDeviceCount; i++) {
	if (deviceID == outputDevices[i].resourceID) {
	    value = i;
	    break;
	}
    }
    return value;
}

ChoiceValue
SGIConverter::inputDeviceToChoice(int deviceID) const {
    ChoiceValue value = 0;
    int ndx = inputDeviceToIndex(deviceID);
    if (ndx >= 0)
        value = inputDevices[ndx].choiceIndex;
    return value;
}

ChoiceValue
SGIConverter::outputDeviceToChoice(int deviceID) const {
    ChoiceValue value = 0;
    int ndx = outputDeviceToIndex(deviceID);
    if (ndx >= 0)
        value = outputDevices[ndx].choiceIndex;
    return value;
}

int
SGIConverter::inputChoiceToDevice(ChoiceValue index) const {
    int value = -1;
    for (int i = 0; i < inputDeviceCount; i++) {
	if (index == inputDevices[i].choiceIndex) {
	    value = inputDevices[i].resourceID;
	    break;
	}
    }
    return value;
}

int
SGIConverter::outputChoiceToDevice(ChoiceValue index) const {
    int value = -1;
    for (int i = 0; i < outputDeviceCount; i++) {
	if (index == outputDevices[i].choiceIndex) {
	    value = outputDevices[i].resourceID;
	    break;
	}
    }
    return value;
}

int
SGIConverter::displayError(const char *msg) {
    Application::alert(msg ? msg : "Error:", alGetErrorString(oserror()));
    return false;
}

int
SGIConverter::initialize() {
    BUG("SGIConverter::initialize()");
    alSetErrorHandler(nil);        // SGIConverter will handle errors
    inPortConfig = alNewConfig();
    outPortConfig = alNewConfig();
    if(!inPortConfig || !outPortConfig)
        return displayError("alNewConfig:");
    else
        return getDefaults();
}

long
SGIConverter::getParameter(long param) const {
    ALpv pv;
    pv.param = param;
    int stat = alGetParams(currentDevice(), &pv, 1);
#ifdef DEBUG_SGI
    if (stat != 0) fprintf(stderr, "alGetParams: %s\n", alGetErrorString(oserror()));
#endif
    return (long) pv.value.ll;
}

int
SGIConverter::setParameter(long param, long value) {
    pvarray[0].param = param;
    pvarray[0].value.i = value;
    return (alSetParams(currentDevice(), &pvarray[0], 1) == 0) ?
	    true : displayError("setParameter:");
}

char *strdup(const char *str) {
    char *s = new char[strlen(str) + 1];
    strcpy(s, str);
    return s;
}

int
SGIConverter::getDefaults() {
#ifdef AL_SYSTEM
#ifndef STRSIZE
#define STRSIZE 32
#endif
    // Find all the audio devices on the system as well as the default
    int         nValues;
    ALvalue     alValues[32];
    ALpv        qual;
    char        str[STRSIZE];   /* 32 is magic number from AL */

    // Do Inputs

    qual.param   = AL_TYPE;
    qual.value.i = AL_INPUT_DEVICE_TYPE;

    nValues = alQueryValues(AL_SYSTEM, AL_DEVICES, alValues, 16, &qual, 1);
    nValues++;          /* need and extra for "Default In" */

    if (nValues > 0) {
	ALpv pv[2];
        ALparamInfo paramInfo;

        inputDeviceCount = nValues;
        inputDevices[0].label = strdup("Default Input");
        inputDevices[0].resourceID = AL_DEFAULT_INPUT;
	inputDevices[0].choiceIndex = 1;
        alGetParamInfo(AL_DEFAULT_INPUT, AL_GAIN, &paramInfo); 
        inputDevices[0].maxGain = alFixedToDouble(paramInfo.max.ll); 
        inputDevices[0].minGain = alFixedToDouble(paramInfo.min.ll); 

        pv[0].param     = AL_LABEL;
        pv[0].value.ptr = str;
        pv[0].sizeIn    = STRSIZE;

        for (int i = 1; i < nValues && i < 8; i++) {
	    int res = alValues[i-1].i;
            if (alGetParams(res, pv, 1) != 1) {
                fprintf(stderr, "alGetParams failed\n");
		return false;
            }
            inputDevices[i].label = strdup(str);
            inputDevices[i].resourceID = res;
	    inputDevices[i].choiceIndex = 2 << (i - 1);

            alGetParamInfo(res, AL_GAIN, &paramInfo); 
            inputDevices[i].maxGain = alFixedToDouble(paramInfo.max.ll); 
            inputDevices[i].minGain = alFixedToDouble(paramInfo.min.ll); 
        }
    }

    // Do Outputs

    qual.param   = AL_TYPE;
    qual.value.i = AL_OUTPUT_DEVICE_TYPE;

    nValues = alQueryValues(AL_SYSTEM, AL_DEVICES, alValues, 16, &qual, 1);
    nValues++;          /* need and extra for "Default out" */
    if (nValues > 0) {
	ALpv pv[2];
        ALparamInfo paramInfo;
        outputDeviceCount = nValues;
        outputDevices[0].label = strdup("Default Output");
        outputDevices[0].resourceID = AL_DEFAULT_OUTPUT;
	outputDevices[0].choiceIndex = 1;
        alGetParamInfo(AL_DEFAULT_OUTPUT, AL_GAIN, &paramInfo); 
        outputDevices[0].maxGain = alFixedToDouble(paramInfo.max.ll); 
        outputDevices[0].minGain = alFixedToDouble(paramInfo.min.ll); 

        pv[0].param     = AL_LABEL;
        pv[0].value.ptr = str;
        pv[0].sizeIn    = STRSIZE;

        for (int i = 1; i < nValues && i < 8; i++) {
	    int res = alValues[i-1].i;
            if (alGetParams(res, pv, 1) != 1) {
                fprintf(stderr, "alGetParams failed\n");
		return false;
            }
            outputDevices[i].label = strdup(str);
            outputDevices[i].resourceID = res;
	    outputDevices[i].choiceIndex = 2 << (i - 1);

            alGetParamInfo(res, AL_GAIN, &paramInfo); 
            outputDevices[i].maxGain = alFixedToDouble(paramInfo.max.ll); 
            outputDevices[i].minGain = alFixedToDouble(paramInfo.min.ll); 
        }
    }
#endif
    return true;
}

void
SGIConverter::resetDefaults() {
}

boolean
SGIConverter::isPlayableFormat(DataType type) {
    return (type != UnsignedCharData
        && type != MuLawData && type != IntData);
}

int
SGIConverter::checkDataType(DataType type) {
    long sampwidth = AL_SAMPLE_16;
    long format;
    switch(type) {
    case FloatData: format = AL_SAMPFMT_FLOAT;  break;
    case DoubleData: format = AL_SAMPFMT_DOUBLE;  break;
    case ShortData:  format = AL_SAMPFMT_TWOSCOMP;  break;
    case SignedCharData:
        format = AL_SAMPFMT_TWOSCOMP;
        sampwidth = AL_SAMPLE_8;
        break;
    }
    if(alSetSampFmt(currentConfig(), format) < 0
       || alSetWidth(currentConfig(), sampwidth) < 0)
        return displayError("alSetSampFmt/alSetWidth:");
    else if(type >= FloatData)
        alSetFloatMax(currentConfig(), peakAmplitude());
    return true;
}

int
SGIConverter::checkSampleRate(int sampleRate) {
    pvarray[0].param = AL_RATE;
    pvarray[0].value.ll = alDoubleToFixed((double) sampleRate);
    pvarray[1].param = AL_MASTER_CLOCK;
    pvarray[1].value.i = AL_CRYSTAL_MCLK_TYPE;
    if (alSetParams(currentDevice(), pvarray, 2) < 0)
	return displayError("checkSampleRate: alSetParams:");
    return true;
}

int
SGIConverter::checkChannels(int chans) {
    if (alSetChannels(currentConfig(), chans) < 0)
	return displayError("checkChannels: alSetChannels:");
    return true;
}

// Needed to redefine this here to check port use before doing any config

int
SGIConverter::configure(Sound* sound, GoingTo wantTo) {
    // need to determine whether audio is in use (other than us).
    if(dontInterrupt && getParameter(AL_OUTPUT_COUNT) > 1) {
        Application::alert("Audio hardware currently in use.");
        return false;
    }
    return Super::configure(sound, wantTo);
}

int
SGIConverter::doConfigure() {
    BUG("SGIConverter::doConfigure()");
#ifdef DEBUG_SGI
    fprintf(stderr, "Queue size set to %ld.\n", bestQueueSize());
#endif
    if(alSetQueueSize(currentConfig(), bestQueueSize()) < 0)
        return displayError("alSetQueueSize:");
    if(audio_port) alClosePort(audio_port);
    if((audio_port = alOpenPort("mxv", willRecord() ? "r" : "w",
				currentConfig())) == 0) {
	return displayError("Failed to open audio port:");
    }
    if(alSetFillPoint(audio_port, bestQueueSize()/4) < 0)
        return displayError("alSetFillPoint:");
    return true;
}

// read and write 1/10 second's worth of sound, so queue is 5x
// that size, or 1/2 second

long
SGIConverter::bestQueueSize() {
    return round(sampleRate() / 2.0);
}

long
SGIConverter::writeSamps() {
    return channels() * (bestQueueSize() / 5);
}

long
SGIConverter::readSamps() {
    return channels() * (bestQueueSize() / 5);
}

inline long min(long x, long y) { return (x < y) ? x : y; }

int
SGIConverter::doConversion(StatusAction* askedToStop) {
    BUG("SGIConverter::doConversion()");
    int sampsize = sampleSize();
    int chans = channels();
    int nframes = dataSize() / (sampsize * chans); // total frames to write
    int frames = writeSamps() / chans;         // size of each write in frames
    frames = min(frames, nframes);
    int bufsize = frames * chans * sampsize;  // size of each write in bytes
#ifdef DEBUG_SGI
    fprintf(stderr, "Playing %ld frames = %ld bytes\n", nframes,
            nframes * chans * sampsize);
    fprintf(stderr, "Buffer is %ld frames = %ld bytes\n", frames, bufsize);
    fprintf(stderr, "Starting conversion...\n");
#endif
    lockMemory();
    addr databuffer = addr(pointerToData());
    Application::inform("Playing...");
    boolean stopped = false;
    while(nframes > 0) {
        if((*askedToStop)()) {
            stopped = true;
            break;
        }
        if(alGetFillable(audio_port) < frames)
            continue;
#ifdef DEBUG_SGI
        fprintf(stderr, "\twriting %d frames...\n", frames);
#endif
        if(alWriteFrames(audio_port, databuffer, frames) < 0)
            return displayError("Error in alWriteFrames():");
#ifdef DEBUG_SGI
        fprintf(stderr, "\t%d frames written\n", frames);
#endif
        databuffer += bufsize;
        nframes -= frames;
        frames = min(frames, nframes);
#ifdef DEBUG_SGI
        fprintf(stderr, "\tnframes now %d, frames now %d\n", nframes, frames);
#endif
    }
    if(!stopped) waitForStop(askedToStop);
    return stop();
}

int
SGIConverter::doRecording(StatusAction* askedToStop) {
    int sampsize = sampleSize();
    int chans = channels();
    int nframes = dataSize() / (sampsize * chans); // total frames to read
    int frames = readSamps() / chans;         // size of each read in frames
    frames = min(frames, nframes);
    int bufsize = frames * chans * sampsize;  // size of each read in bytes
#ifdef DEBUG_SGI
    fprintf(stderr, "Recording %ld frames = %ld bytes\n",
            nframes, nframes * chans * sampsize);
    fprintf(stderr, "Buffer is %ld frames = %ld bytes\n", frames, bufsize);
#endif
    lockMemory();
    char *databuffer = (char *) pointerToData();
    Application::inform("Recording...");
    while(nframes > 0) {
        if((*askedToStop)())
            return stop();
        if(alReadFrames(audio_port, databuffer, frames) < 0)
            return displayError("Error in alReadFrames():");
        databuffer += bufsize;
        nframes -= frames;
#ifdef DEBUG_SGI
        fprintf(stderr, "\t%d frames read, nframes= %d\n", frames, nframes);
#endif
        frames = min(frames, nframes);
    }
#ifdef DEBUG_SGI
        fprintf(stderr, "At bottom, calling stop()\n");
#endif
    stop();
    return true;
}

int
SGIConverter::setInputDevice(int device) {
    inputDevice = device;
#ifdef DEBUG_SGI
    fprintf(stderr, "setInputDevice: device = %d\n", device);
#endif
    return (alSetDevice(inPortConfig, device) == 0);
}

int
SGIConverter::setOutputDevice(int device) {
    outputDevice = device;
#ifdef DEBUG_SGI
    fprintf(stderr, "setOutputDevice: device = %d\n", device);
#endif
    return alSetDevice(outPortConfig, device);
}

int SGIConverter::currentDevice() const {
    return willRecord() ? currentInputDevice() : currentOutputDevice();
}

ALconfig SGIConverter::currentConfig() const {
    return willRecord() ? inPortConfig : outPortConfig;
}

int
SGIConverter::currentPlayLevel() const {
    ALfixed gain[8];
    ALpv x[4];
    /* Now get the current value of gain */
    x[0].param = AL_GAIN;
    x[0].value.ptr = gain;
    x[0].sizeIn = 8;           // we've provided an 8-channel vector
    x[1].param = AL_CHANNELS;
    if (alGetParams(currentOutputDevice(), x, 2) < 0) {
#ifdef DEBUG_SGI
	fprintf(stderr, "alGetParams: %s\n", alGetErrorString(oserror()));
#endif
        return 0;
    }
    double decibels = alFixedToDouble(gain[0]);
#ifdef DEBUG_SGI
    fprintf(stderr, "currentPlayLevel was %g decibels\n", decibels);
#endif
    double mindB =
        outputDevices[outputDeviceToIndex(currentOutputDevice())].minGain;
    double maxdB =
        outputDevices[outputDeviceToIndex(currentOutputDevice())].maxGain;
    if (maxdB > mindB) {
	double frac = (decibels - mindB) / (maxdB - mindB);
	return (100 * frac);
    }
    else
        return (100);
}

int
SGIConverter::currentRecordLevel() const {
    ALfixed gain[8];
    ALpv x[4];
    /* Now get the current value of gain */
    x[0].param = AL_GAIN;
    x[0].value.ptr = gain;
    x[0].sizeIn = 8;           // we've provided an 8-channel vector
    x[1].param = AL_CHANNELS;
    if (alGetParams(currentInputDevice(), x, 2) < 0) {
#ifdef DEBUG_SGI
	fprintf(stderr, "alGetParams: %s\n", alGetErrorString(oserror()));
#endif
        return 0;
    }
    double decibels = alFixedToDouble(gain[0]);
#ifdef DEBUG_SGI
    fprintf(stderr, "currentRecordLevel was %g decibels\n", decibels);
#endif
    double mindB =
        inputDevices[inputDeviceToIndex(currentInputDevice())].minGain;
    double maxdB =
        inputDevices[inputDeviceToIndex(currentInputDevice())].maxGain;
    if (maxdB > mindB) {
	double frac = (decibels - mindB) / (maxdB - mindB);
	return (100 * frac);
    }
    else
        return (100);
}

boolean
SGIConverter::setPlayLevel(int volume){
    double mindB =
        outputDevices[outputDeviceToIndex(currentOutputDevice())].minGain;
    double maxdB =
        outputDevices[outputDeviceToIndex(currentOutputDevice())].maxGain;
    if (maxdB <= mindB)
        return 0;
    double targetdB = volume/100.0 * (maxdB - mindB) + mindB;
    ALfixed gain[8];
    ALpv x;
    for(int i=0; i < 8; i++) gain[i] = alDoubleToFixed(targetdB);
    x.param = AL_GAIN;
    x.value.ptr = gain;
    x.sizeIn = 8;           // we've provided an 8-channel vector
    if (alSetParams(currentOutputDevice(), &x, 1) < 0) {
#ifdef DEBUG_SGI
	fprintf(stderr, "setPlayLevel: alSetParams: %s\n",
		alGetErrorString(oserror()));
#endif
        return 0;
    }
    return 1;
}

boolean
SGIConverter::setRecordLevel(int volume){
    double mindB =
        inputDevices[inputDeviceToIndex(currentInputDevice())].minGain;
    double maxdB =
        inputDevices[inputDeviceToIndex(currentInputDevice())].maxGain;
    if (maxdB <= mindB)
        return 0;

    double targetdB = volume/100.0 * (maxdB - mindB) + mindB;
    ALfixed gain[8];
    ALpv x;
    for(int i=0; i < 8; i++) gain[i] = alDoubleToFixed(targetdB);
    x.param = AL_GAIN;
    x.value.ptr = gain;
    x.sizeIn = 8;           // we've provided an 8-channel vector
    if (alSetParams(currentInputDevice(), &x, 1) < 0) {
#ifdef DEBUG_SGI
	fprintf(stderr, "setRecordLevel: alSetParams: %s\n",
		alGetErrorString(oserror()));
#endif
        return 0;
    }
    return 1;
}

boolean
SGIConverter::setSpeakerOutput(boolean isOn) {
    return setParameter(AL_MUTE, !isOn);
}

int
SGIConverter::pause() {
    return false;
}

int
SGIConverter::resume() {
    return false;
}

int
SGIConverter::stop() {
    BUG("SGIConverter::stop()");
    int status = true;
    if(running()) {
#ifdef DEBUG_SGI
        long buf[10];
        buf[0] = AL_ERROR_NUMBER;
        buf[2] = AL_ERROR_TYPE;
        buf[4] = AL_ERROR_LENGTH;
        buf[6] = AL_ERROR_LOCATION_LSP;
        buf[8] = AL_ERROR_LOCATION_MSP;
        if(audio_port) {
            ALgetstatus(audio_port, buf, 10);
            if(buf[1] > 0)
                fprintf(stderr, "Last audio port error:  %d at frame %d: %s\n",
                        buf[3], buf[5], alGetErrorString(buf[3]));
        }
#endif
        if(audio_port) alClosePort(audio_port);
        unlockMemory();
        resetDefaults();
        status = Super::stop();
    }
    return status;
}

int
SGIConverter::lockMemory() {
    // fix memory in RAM to avoid swapping
    return (mpin((void *) pointerToData(), unsigned(dataSize())) == 0);    
}

int
SGIConverter::unlockMemory() {
    // let it swap
    return (munpin((void *) pointerToData(), unsigned(dataSize())) == 0);    
}

int
SGIConverter::waitForStop(StatusAction* askedToStop) {
    BUG("SGIConverter::waitForStop()");
    while(alGetFilled(audio_port) > 0) {    
        if((*askedToStop)())
            break;
        sginap(1);        // allow audio buffer to drain
    }
    return true;
}

int
SGIConverter::sampleSize() {
    return type_to_sampsize(dataType());
}

#ifdef __GNUG__
typedef void (*SignalFun)(...);
#elif defined(sgi) 
#if defined(__sigargs)
typedef void (*SignalFun)(__sigargs);    // since this varies with OS version
#else
typedef void (*SignalFun)(...);
#endif /* sgi */
#else
typedef void (*SignalFun)(int);
#endif

void
SGIConverter::catchSignals(boolean flag) {
    if(flag) {
        signal(SIGHUP, SignalFun(signalExit));
        signal(SIGINT, SignalFun(signalExit));
        signal(SIGQUIT, SignalFun(signalQuit));
        signal(SIGBUS, SignalFun(signalQuit));
        signal(SIGSEGV, SignalFun(signalQuit));
        signal(SIGTERM, SignalFun(signalExit));
    }
    else {
        signal(SIGHUP, SIG_DFL);
        signal(SIGINT, SIG_DFL);
        signal(SIGQUIT, SIG_DFL);
        signal(SIGBUS, SIG_DFL);
        signal(SIGSEGV, SIG_DFL);
        signal(SIGTERM, SIG_DFL);
    }
}

void
SGIConverter::ignoreSignals(boolean flag) {
    if(flag) {
#ifdef SIGTSTP
        signal(SIGTSTP, SignalFun(signalIgnore));
#endif
    }
    else {
#ifdef SIGTSTP
        signal(SIGTSTP, SIG_DFL);
#endif
    }
}

// these are declared static to allow passing to signal()

void
SGIConverter::signalExit(...) {
    fprintf(stderr, "\nCaught signal.  Exiting...\n");
    fflush(stderr);
    Converter::destroyInstance();
    exit(0);
}

void
SGIConverter::signalIgnore(...) {
    fprintf(stderr, "\nCaught signal during play or record...ignoring.\n");
}

void
SGIConverter::signalQuit(...) {
    fprintf(stderr, "\nCaught deadly signal.  Aborting...\n");
    fflush(stderr);
    abort();
}
