// soundheader.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <sys/time.h>
#include "localdefs.h"
#include "application.h"
#include "comment.h"
#include "datafile.h"
#include "sound.h"
#include "soundheader.h"
#include "aiff_header.h"
#include "valuesetter.h"
#include "wavheader.h"
#include "typeconvert.h"

SoundHeader::Type SoundHeader::default_HeaderType = HEADER_DEFAULT_TYPE;

// create Sound Header class instance appropriate for the magic number found
// in the file.  Default case is to create a HybridHeader and let it handle
// all other possibilities (such as raw data)

SoundHeader *
SoundHeader::create(DataFile* file, DataType type, int srate, int chans, double peak, boolean reading) {
    Type headerType = defaultHeaderType();    // default is reset if magic read
    if(file && file->dataSize() > 0) {    // check for existing file
        switch(Header::readMagicNumber(file)) {
        case SF_MAGIC:
        case SF_SWAPMAGIC:
        case IRCAM_SUNMAGIC:
        case IRCAM_MIPSMAGIC:
        case IRCAM_NEXTMAGIC:
            headerType = Ircam;
            break;
        case FORM:
        case MROF:
#ifdef sgi
	case MPEG1_MAGIC:
#endif
            headerType = Aifc;
            break;
        case RIFF:
        case FFIR:
            headerType = Wave;
            break;
        case _SND:
        case DNS_:
            headerType = Hybrid;
        default:
            headerType = reading ? Hybrid : None;
            break;
        };
    }
    else if(file == nil)
        headerType = None;
    return create(headerType, type, srate, chans, peak);
}

SoundHeader *
SoundHeader::create(
    SoundHeader::Type htype, DataType type, int srate, int chans, double peak
) {
    SoundHeader* header = nil;
    switch(htype) {
    case Ircam:        // old 1024-byte BSD header
        header = new IrcamHeader(type, srate, chans, peak);
        break;
    case Aifc:        // AIF-C header for Macs and SGIs
        header = new AIFFSoundHeader(type, srate, chans, peak);
        break;
    case Wave:        // WAVE format used on Intel machines (MS Windows)
        header = new WAVSoundHeader(type, srate, chans);
        break;
    case Snd:        // 28 byte NeXT Snd or SPARC Au header
        header = new SndHeader(type, srate, chans);
        break;
    case None:
    case Hybrid:     // Lansky's BSD header with Snd header at the beginning
        header = new HybridHeader(type, srate, chans, peak);
    default:
        break;
    }
    if(htype == None) header->setRaw();    // writing headerless file
    return header;
}

SoundHeader::SoundHeader(
    DataType format, int rate, int chans, double peak, int magic)
        : Header(magic, format, chans), samprate(rate), peakamp(peak) {}

const char*
SoundHeader::magicError() {
    static char msg[80];
    sprintf(msg, "Unknown or invalid soundfile magic number:  %d", magic());
    return msg;
}

int
SoundHeader::checkHeader() {
    char msg[64];
    msg[0] = '\0';    // null for later check
    int retcode = 0;
    if(!isRaw() && !validChannels(nchans))
        sprintf(msg, "Illegal channel count for this header type: %d",
                nchans);
    else if(samprate < 1000 || samprate > 128000)
        sprintf(msg, "Invalid sound samp rate (%d)", samprate);
    else if(data_type == NoData)
        sprintf(msg, "Sound header: unsupported data type");
    else if(!isRaw() && !isValid(DataType(data_type)))    // dont check on raw read
        sprintf(msg, "Illegal sample format for this header type.");
    else
        retcode = 1;
    if(strlen(msg))
        Application::alert(msg);
    return retcode;
}

// this is the bytes/sec conversion needed to determine offsets in time
// value is rounded to the nearest frame bytes (chans*sampsize)

int
SoundHeader::secondsToBytes(double seconds) {
    return int(seconds * sampleRate()) * (nChans() * sampleSize());
}

//********

SndOrIrcamSoundHeader::SndOrIrcamSoundHeader(
    DataType format, int rate, int chans, double peak, int magic)
        : SoundHeader(format, rate, chans, peak, magic) {}

boolean
SndOrIrcamSoundHeader::isMagic() {
    return (magic() == _SND || magic() == DNS_ 
        || magic() == SF_MAGIC);
}

boolean
SndOrIrcamSoundHeader::magicIsSwapped() {
    return (magic() == DNS_
        || magic() == IRCAM_MIPSMAGIC
        || magic() == SF_SWAPMAGIC);
}

int
SndOrIrcamSoundHeader::readInfo(DataFile *file) {
    int status = false;
    switch(magic()) {
    case DNS_:
    case _SND:        // native or hybrid header
        status = readSndInfo(file);
        break;
    case SF_SWAPMAGIC:
    case SF_MAGIC:        // Ircam header
    case IRCAM_SUNMAGIC:
    case IRCAM_MIPSMAGIC:
    case IRCAM_NEXTMAGIC:
        status = readIrcamInfo(file);
        break;
    default:        // we should never get here, since checked earlier
        data_type = NoData;
        nchans = 0;
        status = false;
    }
    return status && file->good();
}

int
SndOrIrcamSoundHeader::readSndInfo(DataFile *file) {
    SndStruct sndStruct;
    int status = true;
    if(sndStruct.read(file)) {
        header_type = (sndStruct.data_loc == SIZEOF_BSD_HEADER) ? Hybrid : Snd;
        setDataOffset(sndStruct.data_loc);
        setDataSize(sndStruct.data_size);
        data_type = format_to_type(sndStruct.format);
        nchans = sndStruct.nchans;
        samprate = sndStruct.samp_rate;
        if((status = checkHeader()) == true) {
            char msg[64];
            const char* msg1 = nil;
            const char* msg2 = nil;
            // correct in event of missing or corrupt offset information
            int minOffset = (headerType() == Snd) ? 24 : SIZEOF_BSD_HEADER;
            if(dataOffset() < minOffset) {
                sprintf(msg, "Incorrect header data offset (%d < %d)",
                    dataOffset(), minOffset);
                msg1 = msg;
                setDataOffset(minOffset);
            }
            int trueSize = file->dataSize() - dataOffset();
            if(dataSize() != trueSize) {
                msg2 = "Incorrect data size in header -- adjusting.";
                setDataSize(trueSize);
            }
            if(msg1 || msg2)
                Application::alert("Warning:",
                    msg1? msg1 : msg2, msg1? msg2 : nil);
            if(headerType() == Hybrid) {
                // skip Snd header info block and read rest of header
                file->skip(4);
                IrcamStruct iStruct(0, 0, 0, min(28, dataOffset()));
                if(iStruct.read(file) && iStruct.isValid()) {
                    extractPeak(iStruct.data);
                    extractComment(iStruct.data);
                }
            }
            // else comment will be read from disk directly
        }
    }
    else {
        Application::alert("SoundHeader::readSndInfo:  read error.");
        status = false;
    }
    return status;
}

int
SndOrIrcamSoundHeader::readIrcamInfo(DataFile *file) {
    IrcamStruct iStruct;
    int status = true;
    if(iStruct.read(file)) {
        header_type = Ircam;
        setDataOffset(SIZEOF_BSD_HEADER);
        setDataSize(file->dataSize() - dataOffset());
        data_type = sampsize_to_type(iStruct.sf_packmode);
        nchans = iStruct.sf_chans;
        samprate = int(iStruct.sf_srate);
        if((status = checkHeader()) == true) {
            extractPeak(iStruct.data);
            extractComment(iStruct.data);
        }
    }
    else {
        Application::alert("SoundHeader::readIrcamInfo: read error.");
        status = false;
    }
    return status;
}

int
SndOrIrcamSoundHeader::readComment(DataFile *f) {
    // if not Snd, do nothing - comment extracted from header, not read directly
    return (headerType() == Snd) ? Header::readComment(f) : f->good();
}

int
SndOrIrcamSoundHeader::writeComment(DataFile *f) {
    // do nothing here;  comment written into header, not directly
    return f->good();
}

const char *
SndOrIrcamSoundHeader::getSFCode(const char *hd, int code) {
    const char *sfc = hd;
    const char *hdend = hd + diskHeaderCommentSize();
    while(sfc < hdend) {
        SFCode *sp = (SFCode *) sfc;
        if(sp->code == SF_END)
                break;
        // Catch possible wrap around on stack from bad header
        // or a zero struct size from bad header
        if(sp->bsize <= 0 || sfc + sp->bsize < hd)
                break;
        if(sp->code == code)
                return(sfc);
        sfc += sp->bsize;
    }
    return(NULL);
}

int
SndOrIrcamSoundHeader::extractPeak(const char *ptr) {
    const char *codeptr;
    peakamp = 0;
    if(ptr) {
        SFMaxamp sfm;
        if((codeptr = getSFCode(ptr, SF_MAXAMP)) != NULL) {
            bcopy(codeptr + sizeof(SFCode), (char *) &sfm,
                sizeof(SFMaxamp));
            for(int i = 0; i < nchans && i < MaxChannels; i++)
                if(sfm.value[i] > peakamp)
                    peakamp = sfm.value[i];
            return 1;
        }
        else
            Application::inform("No peak amp information found.", true);
    }
    return 0;
}

int
SndOrIrcamSoundHeader::extractComment(const char *ptr) {
    SFCode *sfc;
    const char *codeptr;
    if((codeptr = getSFCode(ptr, SF_COMMENT)) != NULL) {
        SFComment sfcm;
        sfc = (SFCode *) codeptr;
        bcopy(codeptr + sizeof(SFCode) , (char *) &sfcm, 
            int(sfc->bsize) - sizeof(SFCode));
        setComment(sfcm.comment);
        return true;
    }
    return false;
}

int
SndOrIrcamSoundHeader::putSFCode(const char *hd, const char *ptr,
        const SndOrIrcamSoundHeader::SFCode *codeptr) {
    static const int codesize = sizeof(SFCode);
    static SFCode endcode = {
        SF_END,
        codesize
    };
    char *sfc = (char *) hd;    // casting away const...
    int wasendcode = false;
        const char *hdend = hd + diskHeaderCommentSize();
    while(sfc < hdend) {
        SFCode *sp = (SFCode *) sfc;
        if(sp->code == SF_END) {
            wasendcode = true;
            break;
        }
        /* Catch possible wrap around on stack from bad header */
        if(sp->bsize <= 0 || sfc + sp->bsize < hd) {
            sp->code = SF_END;    // Force an end
            sp->bsize = codesize;
            break;
        }
        if(sp->code == codeptr->code)
            break;
        sfc += sp->bsize;
    }
    
    /* No space left */
    if(sfc + codeptr->bsize > hdend)
        return false;
#if 0
    if(!wasendcode)        // if not first code added to this header
        if(codeptr->bsize != sp->bsize) /* Size changed */
            return false;
#endif

    // copy code struct into header starting at end of last code
    bcopy((char *) codeptr, sfc, codesize);
    // copy actual data struct into header right after that
    bcopy(ptr, sfc + codesize, codeptr->bsize - codesize);
    // stick an end-code struct on to mark the end
    if(wasendcode) 
        bcopy((char *) &endcode, sfc + codeptr->bsize, codesize);
    return true;
}

int
SndOrIrcamSoundHeader::loadPeak(const char *ptr) {
    static SFMaxamp sfm;
    static SFCode ampcode = {
        SF_MAXAMP,
        sizeof(SFMaxamp) + sizeof(SFCode)
    };
    for(int i = 0; i < MaxChannels; i++) {
        sfm.value[i] = peakamp;
        sfm.samploc[i] = 0;
    }
    struct timeval tp;
    struct timezone tzp;
    gettimeofday(&tp,&tzp);
    sfm.timetag = tp.tv_sec;
    return putSFCode(ptr, (char *) &sfm, &ampcode);
}

int
SndOrIrcamSoundHeader::loadComment(const char *ptr) {
    static SFComment sfcom;
    static SFCode comcode = {
        SF_COMMENT,
        sizeof(SFComment) + sizeof(SFCode)
    };
    const Comment *com = getComment();
    if(com != nil) {
        strcpy(sfcom.comment, *com);
        return putSFCode(ptr, (char *) &sfcom, &comcode);
    }
    else return(true);    // nil comment is not failure        
}

// outline methods for nested classes

long
SndOrIrcamSoundHeader::SndStruct::readSize() {
    return sizeof(*this) - Super::readSize();
}

ValueSetterBase**
SndOrIrcamSoundHeader::SndStruct::valueSetters() {
    static ValueSetterBase* setterList[NumElements + 1];
    delete setterList[0];
    setterList[0] = newValueSetter(&magic);
    delete setterList[1];
    setterList[1] = newValueSetter(&data_loc);
    delete setterList[2];
    setterList[2] = newValueSetter(&data_size);
    delete setterList[3];
    setterList[3] = newValueSetter(&format);
    delete setterList[4];
    setterList[4] = newValueSetter(&samp_rate);
    delete setterList[5];
    setterList[5] = newValueSetter(&nchans);
    return setterList;
}

SndOrIrcamSoundHeader::IrcamStruct::IrcamStruct(
    float sr, int ch, int pm, int trim) :
        sf_magic(SF_MAGIC), sf_srate(sr), sf_chans(ch),
        sf_packmode(pm), data(nil), lengthTrim(trim) {
    createVariableBlock();
}

SndOrIrcamSoundHeader::IrcamStruct::~IrcamStruct() { delete [] data; }

ValueSetterBase**
SndOrIrcamSoundHeader::IrcamStruct::valueSetters() {
    static ValueSetterBase* setterList[NumElements + 1];
    delete setterList[0];
    setterList[0] = newValueSetter(&sf_magic);
    delete setterList[1];
    setterList[1] = newValueSetter(&sf_srate);
    delete setterList[2];
    setterList[2] = newValueSetter(&sf_chans);
    delete setterList[3];
    setterList[3] = newValueSetter(&sf_packmode);
    return setterList;
}

long
SndOrIrcamSoundHeader::IrcamStruct::variableBlockLength() {
    return readSize() - 16;        // 16 == 4 elements of struct
}

void
SndOrIrcamSoundHeader::IrcamStruct::createVariableBlock() {
    delete [] data;
    data = new char[variableBlockLength()];
    memset(data, 0, variableBlockLength());
}

//********

SndHeader::SndHeader(DataType format, int rate, int chans) 
        : SndOrIrcamSoundHeader(format, rate, chans, 0, _SND) {
    header_type = Snd;
    data_offset = diskHeaderSize();
}

SndHeader::SndHeader()
    : SndOrIrcamSoundHeader(
        ShortData,
        Sound::defaultSampleRate(),
        1, 0, _SND) {
    header_type = Snd;
    data_offset = diskHeaderSize();
}

boolean
SndHeader::isValid(DataType t) {
    return (t != UnsignedCharData && t != NoData);
}

int
SndHeader::writeInfo(DataFile *file) {
    // if writing Snd style, set offset based on comment length (if any)
    int offset = (headerType() == Snd) ?
        diskHeaderInfoSize() + commentLength() : diskHeaderSize();
    setDataOffset(offset);
    SndStruct sndStruct(dataOffset(), dataSize(),
        type_to_format(dataType()), sampleRate(), nChans());
    int status = sndStruct.write(file);
    if(!status) Application::alert("writeInfo:  file write error.");
    return status;
}

// new here:  snd/au format headers always written with minimum 4-byte comment
// even for .au files, for consistancy

int
SndHeader::writeComment(DataFile *file) {
    return Header::writeComment(file);
}

int
SndHeader::commentLength() {
    return max(Header::commentLength(), diskHeaderCommentSize());
}

//********

IrcamHeader::IrcamHeader(DataType format, int rate, int chans, double peak)
        : SndOrIrcamSoundHeader(format, rate, chans, peak, SF_MAGIC) {
    header_type = Ircam;
    data_offset = diskHeaderSize();
    peakamp = peak;
}

IrcamHeader::IrcamHeader()
    : SndOrIrcamSoundHeader(
        ShortData,            // could this be set?
        Sound::defaultSampleRate(),
        1, 0, SF_MAGIC) {
    header_type = Ircam;
    data_offset = diskHeaderSize();
}

boolean
IrcamHeader::isMagic() {
    return (magic() == IRCAM_MIPSMAGIC || magic() == IRCAM_SUNMAGIC 
        || magic() == IRCAM_NEXTMAGIC || magic() == SF_MAGIC);
}

int
IrcamHeader::writeInfo(DataFile *file) {
    IrcamStruct iStruct(sampleRate(), nChans(), sampleSize());
    if(!loadPeak(iStruct.data))
        Application::inform("Unable to load peak amp into header", true);
    if(!loadComment(iStruct.data))
        Application::inform("Unable to load comment into header", true);
    int status = iStruct.write(file);
    if(!status) Application::alert("writeInfo:  file write error.");
    return status;
}

//********

HybridHeader::HybridHeader(DataType format, int rate, int chans, double peak) 
        : SndHeader(format, rate, chans) {
    header_type = Hybrid;
    data_offset = diskHeaderSize();
    peakamp = peak;
}

HybridHeader::HybridHeader()
        : SndHeader(
            ShortData,
            Sound::defaultSampleRate(),
            1) {
    header_type = Hybrid;
    data_offset = diskHeaderSize();
    peakamp = 0;
}

boolean
HybridHeader::isValid(DataType t) {
    return (header_type == Snd) ? SndHeader::isValid(t) : (t == ShortData || t == FloatData);
}

int
HybridHeader::writeInfo(DataFile *file) {
    int status = false;
    if((status = SndHeader::writeInfo(file)) == true) {
        file->seek(SndHeader::diskHeaderSize()); // skip past local comment
        IrcamStruct iStruct(sampleRate(), nChans(), sampleSize(),
            SndHeader::diskHeaderSize());
        if(!loadPeak(iStruct.data))
            Application::inform("Unable to load peak amp into header", true);
        if(!loadComment(iStruct.data))
            Application::inform("Unable to load comment into header", true);
        // write out header minus the length of the local portion
        status = iStruct.write(file);
        if(!status) Application::alert("writeInfo:  file write error.");
    }
    return status;
}

int
HybridHeader::writeComment(DataFile *f) {
    return SndOrIrcamSoundHeader::writeComment(f);
}
