/* mkdecbootcd.c - (C) 2002 Karsten Merker <merker@debian.org>
 *
 * Version 0.1 - initial release, 2002/04/27
 * Version 0.2 - added manual mode to be able to use the program without
 *               loop-mounted CD-image, which requires root access
 *
 * This program generates a DECstation bootsector for an ISO-9660
 * CD image to allow the creation of bootable CDs for DECstations
 * 
 * This code is based on ideas from bootprep.c (C) 1999 by Harald
 * Koerfgen <hkoerfg@web.de> and on parts of delo, the DECstation ext2
 * bootloader (C) 2000 by Florian Lohoff <flo@rfc822.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Usage:
 * 1. Generate a raw bootfile from an ECOFF kernel by calling
 *    "objcopy -O binary vmlinux.ecoff bootfile".
 * 2. Get the value of kernel_entry from the kernel's System.map file.
 * 3. Master an ISO-9660 CD image containing bootfile.
 * 4. Mount the CD image via loopback mount
 *    (e.g. mount -o loop,ro /path/to/my/cdimage /mnt/)
 * 5. Run "mkdecbootcd bootfile cdimage kernel_entry" with
 *    - bootfile being the bootfile IN THE LOOPBACK-MOUNTED
 *      CD image (e.g. (/mnt/bootfile)
 *    - cdimage being the CD image file itself
 *      (e.g. /path/to/my/cdimage)
 *    - kernel_entry being the value of the kernel_entry label
 *      in the kernel's System.map. This can be either a decimal
 *      value, or (with a 0x prefix) a hexadecimal value
 * 6. umount the loopback-mounted CD image
 * 7. create a CD from the CD image
 */

#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <linux/fs.h>
#include <linux/hdreg.h>
#include <assert.h>
#include <string.h>
#include <errno.h>
#include <string.h>

#define PROGVERSION "0.2"
#undef DEBUG

#define BOOTBLOCK_SECTOR_SIZE 512 /* DECstation bootsector is always 512
				     Bytes regardless of the fs blocksize   */
#define MAX_MAPS       51         /* max. number of extents in a bootsector */ 
#define DEC_BOOT_MAGIC 0x02757a   /* Magic number checked by the DECstation
				     firmware on the boot device */
#define LOAD_ADDRESS 0x80040000   /* kernel load address */

typedef struct  extent {
        int     count, start;
} extent;


typedef struct  dec_bootblock {
        unsigned char   pad[8];
        int             magic;
        int             mode;
        int             loadAddr;
        int             execAddr;
        extent          bootmap[MAX_MAPS];
} dec_bootblock;

int writesector(char *disk, int startsector, int length, long kernel_entry) {
        dec_bootblock           *bb;
        int                     i,
                                fd;

        /* Malloc memory */
        if (!(bb=(struct dec_bootblock *) malloc(BOOTBLOCK_SECTOR_SIZE))) {
		printf("Could not allocate memory for bootsector\n");
		return(4);
	}

        /* Open disk */
        if ((fd=open(disk,O_RDWR)) < 0) {
                printf("Unable to open image %s\n",disk);                
                return(5);
        }

        /* Read sector - Contains the disk label (MSDOS/ULTRIX) */
        if(read(fd,bb,BOOTBLOCK_SECTOR_SIZE) != BOOTBLOCK_SECTOR_SIZE) {
                printf("Reading bootsector from image failed\n");
                close(fd);
                return(6);
        }

        /* Fill in our values we care on */
        bb->magic=DEC_BOOT_MAGIC;      /* DECstation BootBlock */
        bb->mode=0;                    /* single extent boot */
        bb->loadAddr=LOAD_ADDRESS;     /* Kernel start address in memory */
        bb->execAddr=kernel_entry;     /* Address of kernel_entry label */

        /* Copy file position and length to boot sector */

        bb->bootmap[0].start=startsector;
	bb->bootmap[0].count=length;

	/* fill the rest of the blocklist with zeros */ 

        for(i=1;i<MAX_MAPS-1;i++) {
                bb->bootmap[i].start=0x0;
                bb->bootmap[i].count=0x0;
        }

        /* And write back sector */
        assert(lseek(fd, 0, SEEK_SET) != -1);
	
	printf("Writing bootsector to image %s ... ", disk);
        if (write(fd, bb, BOOTBLOCK_SECTOR_SIZE) != BOOTBLOCK_SECTOR_SIZE) {
                printf("\nWrite of bootsector failed\n");
                close(fd);
                return(7);
        }
	printf("done.\n");
        close(fd);

        return(0);
}

usage()
{
	printf("Usage: mkdecbootcd bootfile cdimage kernel_entry\n");
	printf("       mkdecbootcd -m startsector count cdimage kernel_entry\n");
}


int main(int argc, char **argv) {
	int		fd,
			i,
			startblock,
			blocksize,
			bcount;
	struct stat	st;
	unsigned long	kernel_entry=0;
	unsigned long	cdstartsector=0;
	unsigned long	cdsectorcount=0;
	struct dec_bootblock cdbootblock;

	printf("mkdecbootcd Version %s - (C) 2002 by Karsten Merker <merker@debian.org>\n", PROGVERSION);

	if ((argc >=2) && (strcmp("-m", argv[1])==0)) {
		/* manual mode */
		if (argc!=6) {
			usage();
			return(0);
		}
		cdstartsector=strtoul(argv[2], NULL, 0);
		if (!cdstartsector) {
			printf("Invalid startsector value\n");
			return(8);
		}
		cdsectorcount=strtoul(argv[3], NULL, 0);
		if (!cdsectorcount) {
			printf("Invalid count value\n");
			return(9);
		}
		/* argv[4] is the cdimage name */
		kernel_entry=strtoul(argv[5], NULL, 0);
		if (!kernel_entry) {
			printf("Invalid kernel_entry value\n");
			return(3);
		}	
#ifdef DEBUG
		printf("startsector=%d, count=%d, kernel_entry=0x%x\n", 
			cdstartsector, cdsectorcount, kernel_entry);
#endif
			return(writesector(argv[4], cdstartsector*4, cdsectorcount*4, kernel_entry));

	} else {
		/* automatic mode */

		if (argc!=4) {
			usage();
			return(0);
		}

		kernel_entry=strtoul(argv[3], NULL, 0);
		if (!kernel_entry) {
			printf("Invalid kernel_entry value\n");
			return(3);
		}
#ifdef DEBUG
		printf("kernel_entry string: ***%s***\n", argv[3]);
		printf("kernel_entry number: ***0x%lx***\n", kernel_entry);
#endif

		if ((fd=open(argv[1], O_RDONLY))==-1) {
			printf("Could not open bootfile %s\n", argv[1]);
			return (1);
		}

		if (ioctl(fd, FIGETBSZ, &blocksize) != 0) {
			printf("Could not get blocksize for %s\n", argv[1]);
			return(2);
		}

		if (fstat(fd, &st)) {
			printf("Could not stat %s\n", argv[1]);	
		};

		bcount = (st.st_size + blocksize - 1) / blocksize;

		/* Get fist block */
		startblock=0;
		if (ioctl(fd, FIBMAP, &startblock)) {
			printf("FIBMAP ioctl failed - errno: %s\n",
					strerror(errno));
		}
		close(fd);
		printf("Creating bootsector for file %s\n", argv[1]);
	
#ifdef DEBUG
		printf("Size: %d, Blocks: 0x%x, Startblock: 0x%x, Blocksize: %d\n",
                st.st_size, bcount, startblock, blocksize);
		printf("Calling writesector with startblock=0x%x, bcount=0x%x\n",
		startblock*4, bcount*4);
#endif

		return(writesector(argv[2], startblock*4, bcount*4, kernel_entry));
	}
}
