(*
 *  C2caml : parses c headers and produces appropriate caml bindings for it
 *  Copyright (C) 1999  Sven LUTHER
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *)

(* $Id: lex.mll,v 1.1 1999/06/19 14:59:19 sven Exp $ *)
{
  open Var
  open Par
  exception Eof
  let char_list_of_string s = let rec frec n = function
      | "" -> []
      | s -> s.[0]::(frec (n-1) (String.sub s 1 n))
    in frec ((String.length s)-1) s
  let digit_of_char s = (int_of_char s) - 48
  let int_of_octal oct =
    let rec frec n = function
      | [] -> n
      | d::q -> frec (n * 8 + (digit_of_char d)) q
    in frec 0 (char_list_of_string oct)
  let int_of_hexa_char = function
    | 'a'|'A' -> 10
    | 'b'|'B' -> 11
    | 'c'|'C' -> 12
    | 'd'|'D' -> 13
    | 'e'|'E' -> 14
    | 'f'|'F' -> 15
    | d -> digit_of_char d
  let int_of_hexa hexa =
    let rec frec n = function
      | [] -> n
      | d::q -> frec (n * 16 + (int_of_hexa_char d)) q
    in frec 0 (char_list_of_string hexa)
}

rule cp = parse
  | [ '_' 'A'-'Z' 'a'-'z' ] [ '_' 'A'-'Z' 'a'-'z' '0'-'9' ]*
  {
    let s = Lexing.lexeme lexbuf in
    match s with
      | "auto" -> Keyword_auto
      | "register" -> Keyword_register
      | "static" -> Keyword_static
      | "extern" -> Keyword_extern
      | "typedef" -> Keyword_typedef
      | "void" -> Keyword_void
      | "char" -> Keyword_char
      | "short" -> Keyword_short
      | "int" -> Keyword_int
      | "long" -> Keyword_long
      | "float" -> Keyword_float
      | "double" -> Keyword_double
      | "signed" -> Keyword_signed
      | "unsigned" -> Keyword_unsigned
      | "const" -> Keyword_const
      | "volatile" -> Keyword_volatile
      | "struct" -> Keyword_struct
      | "union" -> Keyword_union
      | "enum" -> Keyword_enum
      | "case" -> Keyword_case
      | "default" -> Keyword_default
      | "if" -> Keyword_if
      | "else" -> Keyword_else
      | "switch" -> Keyword_switch
      | "while" -> Keyword_while
      | "do" -> Keyword_do
      | "for" -> Keyword_for
      | "goto" -> Keyword_goto
      | "continue" -> Keyword_continue
      | "break" -> Keyword_break
      | "return" -> Keyword_return
      | "sizeof" -> Keyword_sizeof
      | "__attribute__" -> Keyword_gnu_attribute
      | "__extension__" -> cp lexbuf
      | _ -> if Var.is_typedef s vl
        then Typedef_name s
        else if Var.is_enum_ident s vl
	then Enum_const 0
	else Ident s
  }
  | ['0'-'9'] ['0'-'9']*
  {
    let s = Lexing.lexeme lexbuf
    in Integer_const (int_of_string s)
  }
  | ['0'-'9'] ['0'-'9']* ['l' 'L']
  {
    let s = Lexing.lexeme lexbuf
    in Integer_const (int_of_string (String.sub s 0 ((String.length s) -1)))
  }
  | ['0'-'9'] ['0'-'9']* ['u' 'U']
  {
    let s = (Lexing.lexeme lexbuf)
    in Integer_const (int_of_string (String.sub s 0 ((String.length s) -1)))
  }
  | ['0'-'9'] ['0'-'9']* ['l' 'L'] ['l' 'L']
  {
    let s = Lexing.lexeme lexbuf
    in Integer_const (int_of_string (String.sub s 0 ((String.length s) -2)))
  }
  | ['0'-'9'] ['0'-'9']* ['u' 'U'] ['l' 'L'] ['l' 'L']
  {
    let s = Lexing.lexeme lexbuf
    in Integer_const (int_of_string (String.sub s 0 ((String.length s) -3)))
  }
  | '0' ['0'-'7']* 
  {
    let s = Lexing.lexeme lexbuf
    in Integer_const (int_of_octal s)
  }
  | '0' ['0'-'7']* ['l' 'L']
  {
    let s = Lexing.lexeme lexbuf
    in Integer_const (int_of_octal (String.sub s 0 ((String.length s) -1)))
  }
  | '0' ['0'-'7']* ['u' 'U']
  {
    let s = Lexing.lexeme lexbuf
    in Integer_const (int_of_octal (String.sub s 0 ((String.length s) -1)))
  }
  | ("0x"|"0X") ['0'-'9' 'a'-'f' 'A'-'F']* 
  {
    let s = Lexing.lexeme lexbuf
    in Integer_const (int_of_hexa (String.sub s 0 ((String.length s) - 1)))
  }
  | ("0x"|"0X") ['0'-'9' 'a'-'f' 'A'-'F']* ['l' 'L']
  {
    let s = Lexing.lexeme lexbuf
    in Integer_const (int_of_hexa (String.sub s 2 ((String.length s) - 3)))
  }
  | ("0x"|"0X") ['0'-'9' 'a'-'f' 'A'-'F']* ['u' 'U']
  {
    let s = Lexing.lexeme lexbuf
    in Integer_const (int_of_hexa (String.sub s 2 ((String.length s) - 3)))
  }
  | "'" _ "'"
  {
    let s = Lexing.lexeme lexbuf
    in Char_const (s.[1])
  }
  | "\"" (_*) "\""
  {
    let s = Lexing.lexeme lexbuf
    in String (String.sub s 1 ((String.length s) -2))
  }
  | "#" [^ '\n']* "\n"
  {
    cp lexbuf
  }
  | [' ' '\n' '\t' '']
  {
    cp lexbuf
  }
  | "/*" ( "*"+[^ '/'] | [^'*'] )* "*/"
  {
    cp lexbuf
  }
  | (","|"<<"|">>"|"{"|"}"|";"|"+"|"-"|"*"|"/"|"%"|"("|")"|"="|"..."|"["|"]"|
    ":"|"!="|"*="|"/="|"%="|"+="|"-="|"<<="|">>="|"&="|"^="|"|="|"?"|"||"|"&&"|
    "++"|"--"|"."| "->"|"|"|"~"|"!"|"^"|"&"|"=="|"<>"|"<"|">"|">="|"<=")
  {
    let s = Lexing.lexeme lexbuf in
    match s with
    | "," -> Symbol_comma
    | ";" -> Symbol_semicolon
    | "{" -> Symbol_open_bracket
    | "}" -> Symbol_close_bracket
    | "(" -> Symbol_open_paren
    | ")" -> Symbol_close_paren
    | "=" -> Symbol_affect
    | ":" -> Symbol_colon
    | "..." -> Symbol_dots
    | "[" -> Symbol_open_brace
    | "]" -> Symbol_close_brace
    | "*=" -> Symbol_mult_affect
    | "/=" -> Symbol_div_affect
    | "%=" -> Symbol_mod_affect
    | "+=" -> Symbol_add_affect
    | "-=" -> Symbol_sub_affect
    | "<<=" -> Symbol_left_affect
    | ">>=" -> Symbol_right_affect
    | "&=" -> Symbol_and_affect
    | "^=" -> Symbol_xor_affect
    | "|=" -> Symbol_or_affect
    | "?" -> Symbol_quest
    | "||" -> Symbol_logic_or
    | "&&" -> Symbol_logic_and
    | "|" -> Symbol_or
    | "^" -> Symbol_xor
    | "&" -> Symbol_and
    | "==" -> Symbol_equal
    | "!=" -> Symbol_not_equal
    | "<" -> Symbol_less
    | ">" -> Symbol_more
    | ">=" -> Symbol_more_or_equal
    | "<=" -> Symbol_less_or_equal
    | "<<" -> Symbol_left
    | ">>" -> Symbol_right
    | "+" -> Symbol_add
    | "-" -> Symbol_sub
    | "*" -> Symbol_mult
    | "/" -> Symbol_div
    | "%" -> Symbol_mod
    | "++" -> Symbol_add_add
    | "--" -> Symbol_sub_sub
    | "~" -> Symbol_neg
    | "!" -> Symbol_not
    | "." -> Symbol_dot
    | "->" -> Symbol_arrow
    | _ -> cp lexbuf
  }
  | eof { raise Eof }
  
{
  let getfilepart s =
    let l = String.length s
    in let rec f i =
      try if String.get s i = '/' then i+1 else f (i-1)
      with Invalid_argument _ -> 0
    in let a = f (l-1)
    in String.sub s a (l-a)
  let parse_prog s =
    try
      let working_file = "/tmp/c2caml_" ^ (getfilepart s)
      in let _ = Unix.system
        ("gcc `gtk-config --cflags` -E -o " ^ working_file ^ " " ^ s)
      in let lexbuf = Lexing.from_channel (open_in working_file)
      in Par.translation_unit cp lexbuf; Unix.unlink working_file
    with Eof -> print_newline ()
  let get n = List.nth !Var.vl n
}
