# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Count mass difference for given peaklist and match values.

# load modules
from nucleus import commfce


class mDiffCount:
    """Count mass difference for given peaklist and check for amino acids,
    dipeptides, modifications and/or selected value"""

    # ----
    def __init__(self, config):

        self.config = config
        self.ctrlData = {}
        self.peaklist = []
    # ----


    # ----
    def preAnalysePeaklist(self):
        """ Generate differences and find first match only. """

        # create list of dipeptide masses
        if self.ctrlData['checkdip']:
            self.countDipeptideMasses()

        # find first match of selected and create difference list
        return self.preCheckPeaklistDifferences()
    # ----


    # ----
    def countDipeptideMasses(self):
        """ Count dipeptide masses from list of available amino acids. """

        counted = []
        self.dipeptideMasses = {}
        massType = self.ctrlData['masstype']
        for aa1 in self.config.amino.keys():
            for aa2 in self.config.amino.keys():
                if aa2 not in counted:
                    self.dipeptideMasses[aa1 + aa2] = self.config.amino[aa1][massType] + self.config.amino[aa2][massType]
            counted += aa1
    # ----


    # ----
    def preCheckPeaklistDifferences(self):
        """ Find only first match for aech difference within mass range. """

        matchedDiffList = {}

        # get params
        massType = self.ctrlData['masstype']
        errorType = self.ctrlData['errortype']
        tolerance = self.ctrlData['tolerance']
        self.ctrlData['diffvalue'] = abs(self.ctrlData['diffvalue'])

        # find min/max mass for aa and dip
        if self.ctrlData['checkaa'] or self.ctrlData['checkdip']:
            massList = []
            for aa in self.config.amino:
                massList.append(self.config.amino[aa][massType])
            aaMin = min(massList) - commfce.countTolerance(min(massList), tolerance, errorType)
            aaMax = max(massList) + commfce.countTolerance(max(massList), tolerance, errorType)
            dipMin = min(massList)*2 - commfce.countTolerance(min(massList)*2, tolerance, errorType)
            dipMax = max(massList)*2 + commfce.countTolerance(max(massList)*2, tolerance, errorType)

        # find min/max mass for modifications
        if self.ctrlData['checkmod'] == 1:
            massList = []
            for mod in self.config.mod:
                massList.append(self.config.mod[mod][massType])
            modMin = min(massList) - commfce.countTolerance(min(massList), tolerance, errorType)
            modMax = max(massList) + commfce.countTolerance(max(massList), tolerance, errorType)
            del massList

        # walk in diff and check masses
        for x in range(len(self.peaklist)):
            for y in range(x, len(self.peaklist)):
                diff = abs(self.peaklist[x][0] - self.peaklist[y][0])

                # don't count for above maxdiff
                if self.ctrlData['usemaxdiff'] and diff > self.ctrlData['maxdiff']:
                    break

                matched = []

                # don't count for the same diff again
                if diff in matchedDiffList:
                    continue

                # check amino acids if set
                if self.ctrlData['checkaa']:
                    # check only if diff is in mass interval
                    if aaMin <= diff <= aaMax:
                        # walk in amino acid list
                        for aa in self.config.amino:
                            # count tolerance from aa mass and error
                            realtiveTolerance = commfce.countTolerance(self.config.amino[aa][massType], tolerance, errorType)
                            # if diff is same as aa mass +-limits
                            if (self.config.amino[aa][massType] - realtiveTolerance) <= diff <= (self.config.amino[aa][massType] + realtiveTolerance):
                                matched += ['aa']
                                break

                # check dipeptides if set
                if self.ctrlData['checkdip']:
                    # check only if diff is in mass interval
                    if (dipMin) <= diff <= (dipMax):
                        # walk in dipeptides list
                        for dip in self.dipeptideMasses.keys():
                            # count tolerance from dip mass and error
                            realtiveTolerance = commfce.countTolerance(self.dipeptideMasses[dip], tolerance, errorType)
                            # if diff is same as dip mass +-tolerance
                            if (self.dipeptideMasses[dip] - realtiveTolerance) <= diff <= (self.dipeptideMasses[dip] + realtiveTolerance):
                                matched += ['dip']
                                break

                # check modifications if set
                if self.ctrlData['checkmod']:
                    # check only if diff is in mass interval
                    if (modMin) <= diff <= (modMax):
                        # walk in modification list
                        for mod in self.config.mod:
                            # check allowed mod-group only
                            if self.config.mod[mod]['group'] in self.ctrlData['modgroups']:
                                # count tolerance from modification mass and error
                                realtiveTolerance = commfce.countTolerance(self.config.mod[mod][massType], tolerance, errorType)
                                # if diff is same as modification mass +-tolerance
                                if (self.config.mod[mod][massType] - realtiveTolerance) <= diff <= (self.config.mod[mod][massType] + realtiveTolerance):
                                    matched += ['mod']
                                    break

                # check specified value if set
                if self.ctrlData['checkdiff'] and self.ctrlData['diffvalue']:
                    # count tolerance from specified mass and error
                    realtiveTolerance = commfce.countTolerance(self.ctrlData['diffvalue'], tolerance, errorType)
                    # if diff is same as value mass +-tolerance
                    if (self.ctrlData['diffvalue'] - realtiveTolerance) <= diff <= (self.ctrlData['diffvalue'] + realtiveTolerance):
                        matched += ['val']

                # save matched masses
                if matched != []:
                    matchedDiffList[diff] = matched

        return matchedDiffList
    # ----


    # ----
    def checkSelectedDiff(self, diff, errorTypeShow):
        """ Check all matches for given difference. """

        matchedPeaks = []

        # get params
        massType = self.ctrlData['masstype']
        tolerance = self.ctrlData['tolerance']
        errorTypeCount = self.ctrlData['errortype']
        self.ctrlData['diffvalue'] = abs(self.ctrlData['diffvalue'])

        # check for amino acids
        if self.ctrlData['checkaa']:
            for aa in self.config.amino:
                # count tolerance from aa mass and error
                realtiveTolerance = commfce.countTolerance(self.config.amino[aa][massType], tolerance, errorTypeCount)
                # if diff is same as aa mass +-tolerance
                if (self.config.amino[aa][massType] - realtiveTolerance) <= diff <= (self.config.amino[aa][massType] + realtiveTolerance):
                    # return (matched, error, aa mass)
                    error = commfce.calcMassError(diff, self.config.amino[aa][massType], errorTypeShow)
                    matchedPeaks.append([aa, error, 'aa'])

        # check for dipeptides
        if self.ctrlData['checkdip']:
            for dip in self.dipeptideMasses:
                # count tolerance from dip mass and error
                realtiveTolerance = commfce.countTolerance(self.dipeptideMasses[dip], tolerance, errorTypeCount)
                # if diff is same as dip mass +-tolerance
                if (self.dipeptideMasses[dip] - realtiveTolerance) <= diff <= (self.dipeptideMasses[dip] + realtiveTolerance):
                    # return (matched, error, dip mass)
                    error = commfce.calcMassError(diff, self.dipeptideMasses[dip], errorTypeShow)
                    matchedPeaks.append([dip, error, 'dip'])

        # check for modifications
        if self.ctrlData['checkmod']:
            for mod in self.config.mod:
                # check allowed mod-group only
                if self.config.mod[mod]['group'] in self.ctrlData['modgroups']:
                    # count tolerance from modification mass and error
                    realtiveTolerance = commfce.countTolerance(self.config.mod[mod][massType], tolerance, errorTypeCount)
                    # if diff is same as modification mass +-tolerance
                    if (self.config.mod[mod][massType] - realtiveTolerance) <= diff <= (self.config.mod[mod][massType] + realtiveTolerance):
                        # return (matched, error, modification mass)
                        error = commfce.calcMassError(diff, self.config.mod[mod][massType], errorTypeShow)
                        matchedPeaks.append([mod, error, 'mod'])

        # check for given value
        if self.ctrlData['checkdiff'] and self.ctrlData['diffvalue']:
            # count tolerance from specified mass and error
            realtiveTolerance = commfce.countTolerance(self.ctrlData['diffvalue'], tolerance, errorTypeCount)
            if (self.ctrlData['diffvalue'] - realtiveTolerance) <= diff <= (self.ctrlData['diffvalue'] + realtiveTolerance):
                # return (matched, error, value)
                error = commfce.calcMassError(diff, self.ctrlData['diffvalue'], errorTypeShow)
                matchedPeaks.append([round(self.ctrlData['diffvalue'], self.config.cfg['common']['digits']), error, 'val'])

        return matchedPeaks
    # ----
