# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Export spectrum to image file.

# load libs
import wx
import os.path

# load modules
from nucleus import mwx


class mImageExporter:
    """ Export spectrum to image file. """

    # ----
    def __init__(self, parent):
        self.parent = parent
    # ----


    # ----
    def exportData(self, document, spectrum, lastDir=None):
        """ Show export dialog and get and save image data to file. """

        # rise image size dialog
        dlg = dlgImageSize(self.parent, width=700, height=450)
        if dlg.ShowModal() != wx.ID_OK:
            dlg.Destroy()
            return False
        else:
            imageSize = dlg.data
            dlg.Destroy()

        # get path from document
        path = document.getPath()

        # get file name
        if path:
            fileName = os.path.basename(path)
            fileName = os.path.splitext(fileName)
            fileName = fileName[0]
            if document.getScanID():
                fileName += '_scan' + document.getScanID()
            fileName = fileName
        else:
            fileName = 'spectrum'

        # get dir name
        if path:
            path = os.path.dirname(path)
        else:
            path = lastDir

        # available formats
        wildcard = "PNG image file (*.png)|*.png|" \
                    "BMP image file (*.bmp)|*.bmp"
        formats = [
                    wx.BITMAP_TYPE_PNG,
                    wx.BITMAP_TYPE_BMP,
                    ]

        if not wx.Platform == '__WXMSW__':
            fileName += '.png'

        # raise export dialog
        dlg = wx.FileDialog(self.parent, "Export Spectrum Image", lastDir, fileName, wildcard, wx.SAVE|wx.OVERWRITE_PROMPT)
        if dlg.ShowModal() == wx.ID_OK:
            path = dlg.GetPath()
            choice = dlg.GetFilterIndex()
            dlg.Destroy()
        else:
            dlg.Destroy()
            return False

        # get spectrum bitmap
        wx.BeginBusyCursor()
        bitmap = spectrum.getBitmap(imageSize[0], imageSize[1])
        wx.EndBusyCursor()

        # check data
        if not bitmap:
            dlg = wx.MessageDialog(self.parent, "Current document has no data to save!", "No Data", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False

        # save file
        try:
            bitmap.SaveFile(path, formats[choice])
        except IOError:
            dlg = wx.MessageDialog(self.parent, "File cannot be saved! Check permissions please.", "Save Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False
    # ----


class dlgImageSize(wx.Dialog):
    """ Set image size. """

    # ----
    def __init__(self, parent, width=None, height=None):
        wx.Dialog.__init__(self, parent, -1, "Image Size", style=wx.DEFAULT_DIALOG_STYLE)

        self.data = [width, height]

        # pack main frame
        mainBox = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainBox.Add(self.makeParamsBox(), 0, wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainBox.Add(self.makeButtonBox(), 0, wx.ALIGN_CENTER|wx.ALL, 10)
        else:
            mainBox.Add(self.makeParamsBox(), 0, wx.ALL, 5)
            mainBox.Add(self.makeButtonBox(), 0, wx.ALIGN_CENTER|wx.ALL, 5)

        # fit layout
        mainBox.Fit(self)
        self.SetSizer(mainBox)
        self.Centre()
    # ----


    # ----
    def makeParamsBox(self):
        """ Make box for crop parameters. """

        # make items
        width_label = wx.StaticText(self, -1, "Width (in pixels):")
        self.width_value = wx.TextCtrl(self, -1, str(self.data[0]), size=(100, -1), validator=mwx.txtValidator('digits'))
        height_label = wx.StaticText(self, -1, "Height (in pixels):")
        self.height_value = wx.TextCtrl(self, -1, str(self.data[1]), size=(100, -1), validator=mwx.txtValidator('digits'))

        # pack items
        grid = mwx.GridBagSizer()
        grid.Add(width_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.width_value, (0, 1))
        grid.Add(height_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.height_value, (1, 1))

        return grid
    # ----


    # ----
    def makeButtonBox(self):
        """ Make box for main buttons. """

        # make items
        OK_button = wx.Button(self, wx.ID_OK, "OK")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # pack items
        buttonBox = wx.BoxSizer(wx.HORIZONTAL)
        buttonBox.Add(OK_button, 0, wx.ALL, 5)
        buttonBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set events
        wx.EVT_BUTTON(self, wx.ID_OK, self.onDlgOK)

        # set defaults
        OK_button.SetDefault()

        return buttonBox
    # ----


    # ----
    def onDlgOK(self, evt):
        """ Get parameters. """

        # get params
        width = self.width_value.GetValue()
        height = self.height_value.GetValue()

        # test params
        try:
            width = float(width)
            height = float(height)
        except ValueError:
            return
        if not width or not height:
            return

        # check maximum value
        if width > 2500 or height > 2500:
            dlg = wx.MessageDialog(self, "Maximum value for width and height is 2500 pixels.", "Size Over Limit", wx.OK|wx.ICON_EXCLAMATION)
            dlg.ShowModal()
            dlg.Destroy()
            return

        # end dialog
        self.data[0] = width
        self.data[1] = height
        self.EndModal(wx.ID_OK)
    # ----
