# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Search sequence for sub-sequence or mass.

# load libs
import wx
import re
import string

# load modules
from nucleus import mwx


class dlgSearch(wx.Dialog):
    """Find sub-sequence by mass or sequence."""

    # ----
    def __init__(self, parent, byType, sequence, mSeqCount, config, massType, errorType, tolerance):

        # set title
        if byType == 'mass':
            title = "Search for Mass"
        else:
            title = "Search for Sequence"

        # make dialog
        wx.Dialog.__init__(self, parent, -1, title, style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)

        self.parent = parent
        self.mSeqCount = mSeqCount
        self.sequence = sequence
        self.byType = byType
        self.config = config
        self.massType = massType
        self.errorType = errorType
        self.tolerance = tolerance

        self.results = []

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(self.makeResultsBox(), 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(self.makeSearchBox(), 0, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(self.makeButtonBox(), 0, wx.ALL|wx.ALIGN_CENTER, 20)
        else:
            mainSizer.Add(self.makeResultsBox(), 1, wx.EXPAND|wx.LEFT|wx.RIGHT, 3)
            mainSizer.Add(self.makeSearchBox(), 0, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 3)
            mainSizer.Add(self.makeButtonBox(), 0, wx.ALL|wx.ALIGN_CENTER, 5)

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----


    # ----
    def makeResultsBox(self):
        """ Make results list. """

        # get labels
        if self.byType == 'mass':
            label1 = "Mass"
            label2 = "Error"
        else:
            label1 = "Mo. mass"
            label2 = "Av. mass"

        # make items
        self.results_list = mwx.ListCtrl(self, -1, size=(570, 250))
        self.results_list.InsertColumn(0, "#")
        self.results_list.InsertColumn(1, label1, wx.LIST_FORMAT_RIGHT)
        self.results_list.InsertColumn(2, label2, wx.LIST_FORMAT_RIGHT)
        self.results_list.InsertColumn(3, "Range", wx.LIST_FORMAT_CENTER)
        self.results_list.InsertColumn(4, "Sequence", wx.LIST_FORMAT_LEFT)
        self.results_list.SetToolTip(wx.ToolTip("Select peptide to highlight in sequence editor"))

        # pack items
        mainBox = wx.BoxSizer(wx.VERTICAL)
        mainBox.Add(self.results_list, 1, wx.EXPAND|wx.TOP, 5)

        # set columns width
        self.results_list.SetColumnWidth(0, 25)
        self.results_list.SetColumnWidth(1, 60)
        self.results_list.SetColumnWidth(2, 60)
        self.results_list.SetColumnWidth(3, 60)
        self.results_list.SetColumnWidth(4, wx.LIST_AUTOSIZE_USEHEADER)

        # set events
        self.results_list.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onPeptideSelected)

        return mainBox
    # ----


    # ----
    def makeSearchBox(self):
        """ Make search fields. """

        if self.byType == 'mass':
            return self.makeSearchMassBox()
        else:
            return self.makeSearchSequenceBox()
    # ----


    # ----
    def makeSearchSequenceBox(self):
        """ Make box for search params. """

        # get terminal modifications
        nTerm_choices = self.getTerminalModifications('n-term')
        cTerm_choices = self.getTerminalModifications('c-term')

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Search Parameters"), wx.HORIZONTAL)

        value_label = wx.StaticText(self, -1, "Sequence: ")
        self.value_value = wx.TextCtrl(self, -1, '', size=(150, -1), validator=mwx.txtValidator('aminoReg'))
        self.value_value.SetToolTip(wx.ToolTip("Use regular expression syntax for advanced searching"))

        nTerm_label = wx.StaticText(self, -1, "N-term: ")
        self.nTerm_combo = wx.ComboBox(self, -1, size=(85, -1), choices=nTerm_choices, style=wx.CB_READONLY)

        cTerm_label = wx.StaticText(self, -1, "C-term: ")
        self.cTerm_combo = wx.ComboBox(self, -1, size=(85, -1), choices=cTerm_choices, style=wx.CB_READONLY)

        charge_choices = ['M', '1+', '2+', '1-', '2-']
        charge_label = wx.StaticText(self, -1, "Charge: ")
        self.charge_combo = wx.ComboBox(self, -1, size=(50, -1), choices=charge_choices, style=wx.CB_READONLY)

        # pack items
        mainBox.Add(value_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.value_value, 1, wx.BOTTOM, 5)

        mainBox.Add(nTerm_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.nTerm_combo, 0, wx.BOTTOM, 5)

        mainBox.Add(cTerm_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.cTerm_combo, 0, wx.BOTTOM, 5)

        mainBox.Add(charge_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.charge_combo, 0, wx.RIGHT|wx.BOTTOM, 5)

        # set defaults
        self.nTerm_combo.Select(nTerm_choices.index('Hydrogen'))
        self.cTerm_combo.Select(cTerm_choices.index('Free acid'))
        self.charge_combo.Select(charge_choices.index(self.config.cfg['common']['charge']))

        return mainBox
    # ----


    # ----
    def makeSearchMassBox(self):
        """ Make box for search params. """

        # get terminal modifications
        nTerm_choices = self.getTerminalModifications('n-term')
        cTerm_choices = self.getTerminalModifications('c-term')

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Search Parameters"), wx.HORIZONTAL)

        value_label = wx.StaticText(self, -1, "Mass: ")
        self.value_value = wx.TextCtrl(self, -1, '', size=(70, -1), validator=mwx.txtValidator('float'))
        self.value_value.SetToolTip(wx.ToolTip("Check statusbar for mass type"))

        tolerance_label = wx.StaticText(self, -1, "Tolerance: ")
        self.tolerance_value = wx.TextCtrl(self, -1, '', size=(50, -1), validator=mwx.txtValidator('float'))
        self.tolerance_value.SetToolTip(wx.ToolTip("Check statusbar for units"))

        nTerm_label = wx.StaticText(self, -1, "N-term: ")
        self.nTerm_combo = wx.ComboBox(self, -1, size=(85, -1), choices=nTerm_choices, style=wx.CB_READONLY)

        cTerm_label = wx.StaticText(self, -1, "C-term: ")
        self.cTerm_combo = wx.ComboBox(self, -1, size=(85, -1), choices=cTerm_choices, style=wx.CB_READONLY)

        charge_choices = ['M', '1+', '2+', '1-', '2-']
        charge_label = wx.StaticText(self, -1, "Charge: ")
        self.charge_combo = wx.ComboBox(self, -1, size=(50, -1), choices=charge_choices, style=wx.CB_READONLY)

        # pack items
        mainBox.Add(value_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.value_value, 1, wx.BOTTOM, 5)

        mainBox.Add(tolerance_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.tolerance_value, 0, wx.BOTTOM, 5)

        mainBox.Add(nTerm_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.nTerm_combo, 0, wx.BOTTOM, 5)

        mainBox.Add(cTerm_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.cTerm_combo, 0, wx.BOTTOM, 5)

        mainBox.Add(charge_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.charge_combo, 0, wx.RIGHT|wx.BOTTOM, 5)

        # set defaults
        self.tolerance_value.SetValue(str(self.tolerance))
        self.nTerm_combo.Select(nTerm_choices.index('Hydrogen'))
        self.cTerm_combo.Select(cTerm_choices.index('Free acid'))
        self.charge_combo.Select(charge_choices.index(self.config.cfg['common']['charge']))

        return mainBox
    # ----


    # ----
    def makeButtonBox(self):
        """ Make main buttons. """

        # make buttons
        Search_button = wx.Button(self, -1, "Search")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Close")

        # pack items
        mainBox = wx.BoxSizer(wx.HORIZONTAL)
        mainBox.Add(Search_button, 0, wx.ALL, 5)
        mainBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set events
        Search_button.Bind(wx.EVT_BUTTON, self.onSearch)

        # set defaults
        Search_button.SetDefault()

        return mainBox
    # ----


    # ----
    def updateResults(self, peptides):
        """ Show current matched data in result-list. """

        # clear list
        self.results_list.DeleteAllItems()

        # paste results
        for x in range(len(peptides)):
            self.results_list.InsertStringItem(x, str(x+1))
            self.results_list.SetStringItem(x, 1, str(peptides[x][0]))
            self.results_list.SetStringItem(x, 2, str(peptides[x][1]))
            self.results_list.SetStringItem(x, 3, peptides[x][2])
            self.results_list.SetStringItem(x, 4, peptides[x][3])

        # set columns width
        if self.results_list.GetItemCount():
            autosize = wx.LIST_AUTOSIZE
        else:
            autosize = wx.LIST_AUTOSIZE_USEHEADER
        for col in range(5):
            self.results_list.SetColumnWidth(col, autosize)
        self.results_list.updateLastCol()
    # ----


    # ----
    def onPeptideSelected(self, evt):
        """ Highlight selected peptide in main sequence editor. """

        # get selection
        selected = evt.m_itemIndex

        # highlight selected peptide
        self.parent.sequence_value.SetSelection(self.results[selected][2], self.results[selected][3])
        self.parent.onSelectionChanged()
    # ----


    # ----
    def onSearch(self, evt):
        """ Start searching when 'search' button pressed. """

        # get comon params
        peptides = []
        value = self.value_value.GetValue()
        nTerm = self.nTerm_combo.GetValue()
        cTerm = self.cTerm_combo.GetValue()
        charge = self.getCharge(self.charge_combo.GetValue())

        if nTerm == 'Hydrogen':
            nTerm = None
        if cTerm == 'Free acid':
            cTerm = None

        # parse values
        if not value:
            return

        # select value
        self.value_value.SetSelection(-1, -1)

        # get specific mass params
        if self.byType == 'mass':
            self.tolerance = self.tolerance_value.GetValue()

            # parse values
            self.tolerance = self.tolerance.replace(',', '.')
            value = value.replace(',', '.')
            try:
                value = float(value)
                self.tolerance = float(self.tolerance)
            except ValueError:
                return

        # parse sequence expression
        else:
            value = string.upper(value)
            for char in value:
                if not char in 'ACDEFGHIKLMNPQRSTVWYacdefghiklmnpqrstvwy0123456789.^|*+?,()[]{}-=:':
                    return

            # check regular expression
            try:
                re.compile('^' + value + '$')
            except:
                message = "You have an error in the sequence expression."
                dlg = wx.MessageDialog(self, message, "Expression Error", wx.OK|wx.ICON_ERROR)
                dlg.ShowModal()
                dlg.Destroy()
                return

        # search and count sequence
        wx.BeginBusyCursor()
        if self.byType == 'mass':
            self.results = self.mSeqCount.searchForMass(self.sequence, value, self.massType, self.errorType, self.tolerance, charge, nTerm, cTerm)
            noMatchMessage = "No sequence found!\nCheck statusbar for tolerance units."
        else:
            self.results = self.mSeqCount.searchForSequence(self.sequence, value, charge, nTerm, cTerm)
            noMatchMessage = "No sequence found!\nCheck sequence expression."
        wx.EndBusyCursor()

        # format and update results
        if self.results:
            for peptide in self.results:
                mmass = round(peptide[0], self.config.cfg['common']['digits'])
                amass = round(peptide[1], self.config.cfg['common']['digits'])
                seqRange = '[%d-%d]' % (peptide[2]+1, peptide[3])
                before = ''
                after = ''
                if peptide[5]:
                    before = '..'+peptide[5]+') '
                if peptide[6]:
                    after = ' ('+peptide[6]+'..'
                sequence = before + peptide[4] + after
                peptides.append([mmass, amass, seqRange, sequence])

            # update results
            self.updateResults(peptides)

        else:
            dlg = wx.MessageDialog(self, noMatchMessage, "Info", wx.OK|wx.ICON_INFORMATION)
            dlg.ShowModal()
            dlg.Destroy()
    # ----


    # ----
    def getCharge(self, charge):
        """ Get charge values from charge-type. """

        if charge == 'M':
            return 0
        elif charge == '1+':
            return 1
        elif charge == '2+':
            return 2
        elif charge == '1-':
            return -1
        elif charge == '2-':
            return -2
    # ----


    # ----
    def getTerminalModifications(self, term):
        """ Get terminal modifications. """

        if term == 'n-term':
            choices = ['Hydrogen']
        else:
            choices = ['Free acid']
        for name in self.config.mod:
            if self.config.mod[name][term] == 1:
                choices.append(name)

        choices.sort()
        return choices
    # ----
