<?php
// $Horde: mnemo/lib/Mnemo.php,v 1.2.2.3 2002/12/30 12:34:05 jan Exp $

require_once MNEMO_BASE . '/lib/constants.php';

/**
 * Mnemo Base Class.
 *
 * @author Jon Parise <jon@horde.org>
 * @version $Revision: 1.2.2.3 $
 * @package mnemo
 */
class Mnemo {

    /**
     * Retrieves the current user's memo list from storage.
     * This function will also sort the resulting list, if requested.
     *
     * @param object Mnemo_Driver  $storage      The current storage object.
     * @param constant $criteria (optional) A set of flags specifying the set of
     *                           memos to return. (MEMO_ANY, MEMO_ADDED,
     *                           MEMO_MODIFIED, MEMO_DELETED)
     * @param constant $sortby   (optional) The field by which to sort.
     *                           (MNEMO_SORT_DESC, MNEMO_SORT_CATEGORY)
     * @param constant $sortdir  (optional) The direction by which to sort.
     *                           (MNEMO_SORT_ASC, MNEMO_SORT_DESC)
     *
     * @return array        Returns a list of the requested memos.
     *
     * @see Mnemo_Driver::listMemos()
     */
    function listMemos($storage, $criteria = MEMO_ANY, $sortby = MNEMO_SORT_DESC,
                       $sortdir = MNEMO_SORT_ASCEND)
    {
        /* Retrieve the memo list for storage. */
        $memos = $storage->listMemos($criteria);

        /* Sort the memo list. */
        $sort_functions = array(
            MNEMO_SORT_DESC => 'ByDesc',
            MNEMO_SORT_CATEGORY => 'ByCategory');

        /* Sort the array if we have a sort function defined for this field. */
        if (isset($sort_functions[$sortby])) {
            $prefix = ($sortdir == MNEMO_SORT_DESCEND) ? '_rsort' : '_sort';
            uasort($memos, array('Mnemo', $prefix . $sort_functions[$sortby]));
        }

        return $memos;
    }

    /**
     * Builds the HTML for a memo category widget.
     *
     * @param string  $name       The name of the widget.
     * @param integer $selected   (optional) The default category.
     * @param boolean $newheader  (optional) Include a new new category option.
     *
     * @return string       The HTML <select> widget.
     */
    function buildCategoryWidget($name, $selected = false, $newheader = false)
    {
        $html = "<select name=\"$name\">";

        $html .= '<option value="">'. _("Select Category") . "</option>\n";

        if ($newheader) {
            $html .= '<option value="">----' . "</option>\n";
            $html .= '<option value="*new*">' . _("New Category") . "</option>\n";
            $html .= '<option value="">----' . "</option>\n";
        }

        foreach (Mnemo::listCategories() as $id => $name) {
            $html .= "<option value=\"$id\"";
            $html .= ($id == $selected && $selected !== false) ? ' selected="selected">' : '>';
            $html .= $name . '</option>';
        }
        $html .= '</select>';

        return $html;
    }

    /**
     * List a user's categories
     *
     * @return array A list of categories.
     */
    function listCategories()
    {
        global $prefs;

        static $catString, $categories;

        if (!isset($categories) || $catString != $prefs->getValue('memo_categories')) {
            $catString = $prefs->getValue('memo_categories');
            $cats = explode('|', $catString);
            $categories = array(0 => _("Unfiled"));
            foreach ($cats as $cat) {
                list($key, $val) = explode(':', $cat);
                $categories[$key] = $val;
            }
        }

        return $categories;
    }

    /**
     * Add a new category
     *
     * @param string  $name     The name of the category to add.
     *
     * @return integer          A valid category id, 0 on failure or
     *                          the new category's id on success.
     *
     * @since Mnemo 1.1
     */
    function addCategory($name)
    {
        global $prefs;

        if ($prefs->isLocked('memo_categories') || empty($name)) {
            return 0;
        }

        $categories = Mnemo::listCategories();

        if (in_array($name, $categories)) {
            return 0;
        }

        $categories[] = $name;
        unset($categories[0]);

        $cats = array();
        $key = 0;
        foreach ($categories as $key => $cat) {
            $cat = array($key, $cat);
            $cats[] = implode(':', $cat);
        }

        $catString = implode('|', $cats);
        $prefs->setValue('memo_categories', $catString);
        $prefs->store();

        return $key;
    }

    /**
     * Delete a category
     *
     * @param integer   $categoryID The id of the category to remove.
     *
     * @return boolean              True on success, false on failure.
     *
     * @since Mnemo 1.1
     */
    function deleteCategory($categoryID)
    {
        global $prefs;
        $categories = Mnemo::listCategories();

        if ($prefs->isLocked('memo_categories') || !array_key_exists($categoryID, $categories)) {
            return false;
        }

        unset($categories[0]);
        unset($categories[$categoryID]);

        $cats = array();
        foreach ($categories as $key => $cat) {
            $cat = array($key, $cat);
            $cats[] = implode(':', $cat);
        }

        $catString = implode('|', $cats);
        $prefs->setValue('memo_categories', $catString);
        $prefs->store();

        return true;
    }

    /**
     * Rename a category
     *
     * @param integer   $categoryID The id of the category to remove.
     * @param string    $name       The new name of the category.
     *
     * @return boolean              True on success, false on failure.
     *
     * @since Mnemo 1.1
     */
    function renameCategory($categoryID, $name)
    {
        global $prefs;
        $categories = Mnemo::listCategories();

        if ($prefs->isLocked('memo_categories') || empty($name) ||
                        !array_key_exists($categoryID, $categories)) {
            return false;
        }

        unset($categories[0]);
        $categories[$categoryID] = $name;

        $cats = array();
        foreach ($categories as $key => $cat) {
            $cat = array($key, $cat);
            $cats[] = implode(':', $cat);
        }

        $catString = implode('|', $cats);
        $prefs->setValue('memo_categories', $catString);
        $prefs->store();

        return true;
    }


    /**
     * Returns the highlight colors for the categories
     *
     * @return array A list of colors, key matches listCategories keys.
     *
     * @since Mnemo 1.1
     */
    function categoryColors()
    {
        global $prefs;

        static $colorString, $colors;

        if (!isset($colors) || $colorString != $prefs->getValue('memo_colors')) {
            $colorString = $prefs->getValue('memo_colors');
            $cols = explode('|', $colorString);
            $colors = array(0 => "#ffffff");
            foreach ($cols as $col) {
                list($key, $val) = explode(':', $col);
                $colors[$key] = $val;
            }
        }
        return $colors;
    }

    /**
     * Returns the string matching the given category ID.
     *
     * @param integer $categoryID     The category ID to look up.
     *
     * @return string       The formatted category string.
     */
    function formatCategory($categoryID = 0)
    {
        $categories = Mnemo::listCategories();
        return isset($categories[$categoryID]) ?
            $categories[$categoryID] :
            $categories[0];
    }

    /**
     * Comparison function for sorting memos by description.
     *
     * @param array $a  Memo one.
     * @param array $b  Memo two.
     *
     * @return integer  1 if memo one is greater, -1 if memo two is greater; 0 if they are equal.
     */
    function _sortByDesc($a, $b)
    {
        return strcmp($a['desc'], $b['desc']);
    }

    /**
     * Comparison function for reverse sorting memos by description.
     *
     * @param array $a  Memo one.
     * @param array $b  Memo two.
     *
     * @return integer  -1 if memo one is greater, 1 if memo two is greater; 0 if they are equal.
     */
    function _rsortByDesc($a, $b)
    {
        return strcmp($b['desc'], $a['desc']);
    }

    /**
     * Comparison function for sorting memos by category.
     *
     * @param array $a  Memo one.
     * @param array $b  Memo two.
     *
     * @return integer  1 if memo one is greater, -1 if memo two is greater; 0 if they are equal.
     */
    function _sortByCategory($a, $b)
    {
        return strcmp($a['category'], $b['category']);
    }

    /**
     * Comparison function for reverse sorting memos by category.
     *
     * @param array $a  Memo one.
     * @param array $b  Memo two.
     *
     * @return integer  -1 if memo one is greater, 1 if memo two is greater; 0 if they are equal.
     */
    function _rsortByCategory($a, $b)
    {
        return strcmp($b['category'], $a['category']);
    }

    /**
     * Calculate the highlight version of a color.
     *
     * @param string $color     An html color eg #FFFFFF
     *
     * @ return string  A lighter html color
     *
     * @since Mnemo 1.1
     */
    function highlightColor($color)
    {
        $r = hexdec(substr($color,1,2));
        $g = hexdec(substr($color,3,2));
        $b = hexdec(substr($color,5,2));

        $factor = 0x11;
        if ($r >= $g && $r >= $b) {
            $g = $g / $r;
            $b = $b / $r;

            $r = $r + $factor;
            $g = floor($g * $r);
            $b = floor($b * $r);
        } elseif ($g >= $r && $g >= $b) {
            $r = $r / $g;
            $b = $b / $g;

            $g = $g + $factor;
            $r = floor($r * $g);
            $b = floor($b * $g);
        } else {
            $r = $r / $b;
            $g = $g / $b;

            $b = $b + $factor;
            $r = floor($r * $b);
            $g = floor($g * $b);
        }

        return "#" .  str_pad(dechex(min($r, 255)), 2, '0', STR_PAD_LEFT) . str_pad(dechex(min($g, 255)),2 ,'0' , STR_PAD_LEFT) . str_pad(dechex(min($b, 255)), 2, '0', STR_PAD_LEFT);
    }

    /**
     * Add a name=value pair to an URL, taking care of whether there
     * are existing parameters and whether to use ? or & as the glue.
     *
     * @access public
     *
     * @param string $url       The URL to modify
     * @param string $parameter The name=value pair to add.
     *
     * @return string The modified URL.
     *
     * @since Mnemo 1.1
     */
    function addParameter($url, $parameter)
    {
        if (!empty($parameter) && strstr($url, $parameter) === false) {
            if (substr($parameter, 0, 1) == '?') {
                $parameter = substr($parameter, 1);
            }

            $pos = strpos($url, '?');
            if ($pos !== false) {
                $url = substr_replace($url, $parameter . ini_get('arg_separator.output'),
                                      $pos + 1, 0);
            } else {
                $url .= '?' . $parameter;
            }
        }
        return $url;
    }
}
