#include <cstring>
#include <iostream>
#include <sstream>
#include <set>

#include "Exception.h"
#include "Tools.h"
#include "StringTools.h"
#include "MathTools.h"
#include "SinusTable.h"
#include "StopWatch.h"
#include "StringTokenizer.h"


//============================================================================
// Implementation of Tools.h
//============================================================================

//----------------------------------------------------------------------------
int myRand(int limit)
{
    return (int)(1.0 * limit * rand() / (RAND_MAX + 1.0));
}



//============================================================================
// Implementation of StringTools.h
//============================================================================

//----------------------------------------------------------------------------
bool STRING_TOOLS::toBool(const std::string &s)
{
    const char *str = s.c_str();

    bool value = false;

    if (!strcasecmp(str, "1") ||
        !strcasecmp(str, "true") ||
        !strcasecmp(str, "on") ||
        !strcasecmp(str, "yes"))
    {
        value = true;
    }
    else if (!strcasecmp(str, "0") ||
             !strcasecmp(str, "false") ||
             !strcasecmp(str, "off") ||
             !strcasecmp(str, "no"))
    {
        value = false;
    }
    else
    {
        throw Exception(
            std::string("Cannot convert '")
            .append(s).append("' to a boolean"));
    }

    return value;
}

//----------------------------------------------------------------------------
int STRING_TOOLS::toInt(const std::string &s)
{
    const char *str = s.c_str();
    char *endptr;

    int value = strtol(str, &endptr, 10);
    if (endptr == str)
    {
        throw Exception("Cannot convert empty string to an integer");
    }
    if (*endptr != '\0')
    {
        throw Exception(
            std::string("Cannot convert '")
            .append(s).append("' to an integer"));
    }

    return value;
}

//----------------------------------------------------------------------------
unsigned STRING_TOOLS::toUnsigned(const std::string &s)
{
    const char *str = s.c_str();
    char *endptr;

    unsigned value = strtoul(str, &endptr, 10);
    if (endptr == str)
    {
        throw Exception("Cannot convert empty string to an unsigned integer");
    }
    if (*endptr != '\0')
    {
        throw Exception(
            std::string("Cannot convert '")
            .append(s).append("' to an unsigned integer"));
    }

    return value;
}

//----------------------------------------------------------------------------
void STRING_TOOLS::toUnsignedVector(const std::string &s,
                                    std::vector<unsigned> &values)
{
    values.clear();

    StringTokenizer st(s, ",");
    std::string token;
    while (st.hasMoreTokens())
    {
        values.push_back(toUnsigned(st.nextToken(token)));
    }
}

//----------------------------------------------------------------------------
bool STRING_TOOLS::hasUnsignedInTokenString(const std::string &t,
                                            const unsigned value)
{
    StringTokenizer st(t, ",");
    while (st.countTokens() > 0)
    {
        std::string token;
        st.nextToken(token);
        if (toUnsigned(token) == value)  return true;
    }

    return false;
}

//----------------------------------------------------------------------------
bool STRING_TOOLS::popUnsignedFromTokenString(std::string &t,
                                              const unsigned value)
{
    typedef std::set<unsigned> UnsignedSet;

    StringTokenizer st(t, ",");
    UnsignedSet values;

    while (st.countTokens() > 0)
    {
        std::string token;
        st.nextToken(token);
        values.insert(toUnsigned(token));
    }

    UnsignedSet::iterator iter = values.find(value);
    if (iter == values.end())  return false;

    values.erase(iter);
    
    std::ostringstream os;
    for (iter = values.begin(); iter != values.end(); ++iter)
    {
        if (iter != values.begin())  os << ",";
        os << *iter;
    }

    t = os.str();
    return true;
}

//----------------------------------------------------------------------------
bool STRING_TOOLS::popFirstUnsignedFromTokenString(std::string &t,
                                                   const unsigned value)
{
    StringTokenizer st(t, ",");
    if (st.countTokens() == 0)  return false;

    std::string token;
    st.nextToken(token);
    if (toUnsigned(token) == value)
    {
        t = st.getTokenString();
        return true;
    }

    return false;
}

//----------------------------------------------------------------------------
std::string &STRING_TOOLS::convertNumberRanges(std::string &t)
{
    StringTokenizer st(t, ",");
    while (st.countTokens() > 0)
    {
        std::string token;
        st.nextToken(token);

        size_t index = token.find('-');
        if (index != std::string::npos)
        {
            std::ostringstream os;
            size_t first = toUnsigned(token.substr(0, index));
            size_t last = toUnsigned(token.substr(index+1));
            for (size_t i = first; i <= last; i++)
            {
                if (i != first)  os << ",";
                os << i;
            }
            size_t tokenIndex = t.find(token);
            assert(tokenIndex != std::string::npos);
            t.replace(tokenIndex, token.length(), os.str());
        }
    }

    return t;
}



//============================================================================
// Implementation of MathTools.h
//============================================================================

//----------------------------------------------------------------------------
MATH_TOOLS::Vector::Vector()
{
    m_x = 0.0;
    m_y = 0.0;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector::Vector(double x, double y)
{
    m_x = x;
    m_y = y;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector::~Vector()
{
}

//----------------------------------------------------------------------------
double MATH_TOOLS::Vector::getAbsValue() const
{
    return sqrt(m_x*m_x + m_y*m_y);
}

//----------------------------------------------------------------------------
double MATH_TOOLS::Vector::getAngle() const
{
    return sqrt(m_x*m_x + m_y*m_y);
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::setExponential(double absValue, double angle)
{
    m_x = absValue * SinusTable::sin((int)rint(angle));
    m_y = absValue * SinusTable::cos((int)rint(angle));
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::addExponential(double absValue, double angle)
{
    m_x += absValue * SinusTable::sin((int)rint(angle));
    m_y += absValue * SinusTable::cos((int)rint(angle));
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::operator+=(const Vector &other)
{
    m_x += other.m_x;
    m_y += other.m_y;
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::operator-=(const Vector &other)
{
    m_x -= other.m_x;
    m_y -= other.m_y;
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::operator*=(const double scale)
{
    m_x *= scale;
    m_y *= scale;
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::operator/=(const double scale)
{
    m_x /= scale;
    m_y /= scale;
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector MATH_TOOLS::operator+(const Vector &v1, const Vector &v2)
{
    return Vector(v1.m_x + v2.m_x, v1.m_y + v2.m_y);
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector MATH_TOOLS::operator-(const Vector &v1, const Vector &v2)
{
    return Vector(v1.m_x - v2.m_x, v1.m_y - v2.m_y);
}


//----------------------------------------------------------------------------
double MATH_TOOLS::getAngle(double x, double y)
{
    double angle = (atan2(y, -x) * 360.0 / (2.0*M_PI)) - 90.0;
    if (angle < 0.0)  angle += 360.0;
    return angle;
}



//============================================================================
// Implementation of SinusTable.h
//============================================================================

//----------------------------------------------------------------------------
SinusTable SinusTable::sm_instance;
double SinusTable::sm_sinTable[360];
double SinusTable::sm_cosTable[360];

//----------------------------------------------------------------------------
SinusTable::SinusTable()
{
    for (int i=0; i<360; i++)
    {
        double a = i * 2.0*M_PI/360.0;

        sm_sinTable[i] = ::sin(a);
        sm_cosTable[i] = ::cos(a);
    }
}

//----------------------------------------------------------------------------
SinusTable::~SinusTable()
{
}



//============================================================================
// Implementation of StopWatch.h
//============================================================================

//----------------------------------------------------------------------------
StopWatch::StopWatch(const char *name)
{
    m_name = name;

    reset();
}

//----------------------------------------------------------------------------
StopWatch::~StopWatch()
{
    m_name = NULL;
}

//----------------------------------------------------------------------------
void StopWatch::start()
{
    (void)gettimeofday(&m_start, NULL);
}

//----------------------------------------------------------------------------
void StopWatch::stop()
{
    m_count++;

    (void)gettimeofday(&m_stop, NULL);

    m_total.tv_sec += m_stop.tv_sec - m_start.tv_sec;
    m_total.tv_usec += m_stop.tv_usec - m_start.tv_usec;

    if (m_total.tv_usec < 0)
    {
        m_total.tv_sec--;
        m_total.tv_usec += 1000000;
    }
    else if (m_total.tv_usec > 1000000)
    {
        m_total.tv_sec++;
        m_total.tv_usec -= 1000000;
    }
}

//----------------------------------------------------------------------------
void StopWatch::reset()
{
    m_count = 0;

    memset(&m_start, 0, sizeof(m_start));
    memset(&m_stop, 0, sizeof(m_stop));
    memset(&m_total, 0, sizeof(m_total));
}

//----------------------------------------------------------------------------
int StopWatch::compare(const StopWatch &other) const
{
    if (m_total.tv_sec < other.m_total.tv_sec)
    {
        return -1;
    }
    if (m_total.tv_sec > other.m_total.tv_sec)
    {
        return 1;
    }
    if (m_total.tv_usec < other.m_total.tv_usec)
    {
        return -1;
    }
    if (m_total.tv_usec > other.m_total.tv_usec)
    {
        return 1;
    }

    return 0;
}

//----------------------------------------------------------------------------
std::ostream &operator<<(std::ostream &os, const StopWatch &s)
{
    return os << s.m_name << ": "
              << s.m_total.tv_sec << " sec, "
              << s.m_total.tv_usec << " usec "
              << "(Count = " << s.m_count << ")";
}



//============================================================================
// Implementation of StringTokenizer.h
//============================================================================

//----------------------------------------------------------------------------
StringTokenizer::StringTokenizer(const std::string &str,
                                 const std::string &sep)
        : m_string(str), m_sep(sep)
{
    if (m_string.empty())
    {
        m_numTokens = 0;
    }
    else
    {
        m_numTokens = 1;

        for (std::string::const_iterator iter = m_string.begin();
             iter != m_string.end(); ++iter)
        {
            if (m_sep.find(*iter) != std::string::npos)  ++m_numTokens;
        }
    }
}

//----------------------------------------------------------------------------
StringTokenizer::~StringTokenizer()
{
}

//----------------------------------------------------------------------------
std::string &StringTokenizer::nextToken(std::string &token)
{
    if (m_numTokens == 0)
    {
        token.clear();
    }
    else
    {
        --m_numTokens;

        size_t index = m_string.find_first_of(m_sep);
        if (index == std::string::npos)
        {
            token = m_string;
            m_string.clear();
        }
        else
        {
            token.assign(m_string, 0, index);
            m_string.erase(0, index+1);
        }
    }

    return token;
}
