#include <time.h>
#include <apr.h>
#include <apr_pools.h>
#include <apr_errno.h>
#include <libbtpeer/types/btp_session.h>
#include <libbtpeer/types/btp_torrent.h>

btp_session_config new_btp_session_config = {
    BTP_SESSION_FLAG_OWN_SOCKET,
    BTP_SESSION_MAX_FILES,
    BTP_SESSION_MAX_TORRENTS,
    BTP_SESSION_MAX_PEERS,
    "",
    "",
    0
};

apr_status_t btp_session_debug(
    btp_session* s, apr_status_t ret, const char* msg
) {
    if(s->c.f & BTP_SESSION_FLAG_DEBUG) {
        char err[BT_SHORT_STRING];

        if(msg)
            fprintf(stderr, msg);
        
        if(ret != APR_SUCCESS)
            fprintf(stderr, "%s\n", apr_strerror(ret, err, sizeof(err)));
    }
    
    return ret;
}

btp_session* btp_session_create(apr_pool_t* p, btp_session_config c) {
    btp_session* session;
    apr_pool_t* pool;
    apr_status_t ret;
    
    if((ret = apr_pool_create(&pool, p)) != APR_SUCCESS)
        return NULL;
    
    session = apr_pcalloc(pool, sizeof(btp_session));
    session->c = c;
    session->create_t = time(NULL);
    session->p = pool;
    session->fp = btp_file_pool_create(session->p, session->c.max_files);
    session->tl = bt_llist_new(session->p, session->c.max_torrents);
    
    if(session->c.f & BTP_SESSION_FLAG_OWN_SOCKET)
        if((ret = btp_session_socket_init(session)) != APR_SUCCESS) {
            btp_session_debug(
                session, ret, "btp_session_create: btp_session_socket_init: "
            );
            return NULL;
        }
    
    return session;
}

apr_status_t btp_session_socket_init(btp_session* s) {
    apr_socket_t* sock;
    apr_sockaddr_t* sa;
    apr_status_t ret;
    
    if(
        (ret = apr_socket_create(
            &sock, APR_INET, SOCK_STREAM, APR_PROTO_TCP, s->p
        ))
        != APR_SUCCESS
    )
        return btp_session_debug(
            s, ret, "btp_session_socket_init: apr_socket_create: "
        );
    
    if(
        (ret = apr_sockaddr_info_get(
            &sa, s->c.ip[0] ? s->c.ip : APR_ANYADDR, APR_UNSPEC, s->c.port,
            APR_IPV4_ADDR_OK, s->p))
        != APR_SUCCESS
    )
        return btp_session_debug(
            s, ret, "btp_session_socket_init: apr_sockaddr_info_get: "
        );
    
    if((ret = apr_socket_bind(sock, sa)) != APR_SUCCESS)
        return btp_session_debug(
            s, ret, "btp_session_socket_init: apr_socket_bind: "
        );
    
    if((ret = apr_socket_listen(sock, 5)) != APR_SUCCESS)
        return btp_session_debug(
            s, ret, "btp_session_socket_init: apr_socket_listen: "
        );
    
    s->s = sock;

    /*
    if(s->c.f & BTP_SESSION_FLAG_DEBUG)
        if((ret = apr_socket_opt_set(s->s, APR_SO_DEBUG, 1)) != APR_SUCCESS)
            return btp_session_debug(
                s, ret, "btp_session_socket_init: apr_socket_opt_set: "
            );
    */
    
    if((ret = apr_socket_addr_get(&sa, APR_LOCAL, s->s)) != APR_SUCCESS)
        return btp_session_debug(
            s, ret, "btp_session_socket_init: apr_socket_addr_get: "
        );
    
    s->c.port = sa->port;
    
    return APR_SUCCESS;
}

apr_status_t btp_session_have_torrent(
    btp_session* s, const char* dest, const char* name
) {
    bt_llist_e* e;
    
    for(e=s->tl->first; e; e=e->next) {
        btp_torrent* t = (btp_torrent*)e->d;
        if((!strcmp(dest, t->destination)) && (!strcmp(name, t->info->name)))
            return APR_SUCCESS;
    }
    
    return APR_ENOENT;
}

apr_status_t btp_session_add_torrent(btp_session* s, btp_torrent* t) {
    char* dest = t->destination[0] ? t->destination : s->c.dest;
    char path[BT_PATH_LEN];
    
    if(btp_session_have_torrent(s, dest, t->info->name) != APR_ENOENT)
        return APR_EEXIST;
    
    if(
        (snprintf(
            path, BT_PATH_LEN, "%s%s%s", dest, (dest[0] ? "/" : ""),
            t->info->name
        ))
        >= BT_PATH_LEN
    )
        return APR_ENAMETOOLONG;
    
    if(dest[0] && !t->destination[0])
        BT_STRCPY(t->destination, dest);
    
    t->port = s->c.port;
    if(s->c.ip[0]) BT_STRCPY(t->ip, s->c.ip);

    return bt_llist_add_last(s->tl, (void*)t, sizeof(btp_torrent));
}

apr_status_t btp_session_run_torrents(btp_session* s) {
    int i;
    apr_status_t ret = APR_SUCCESS;
    apr_status_t eret;
    int max =
        s->tl->n > BTP_SESSION_RUN_MAX_TORRENTS ?
        BTP_SESSION_RUN_MAX_TORRENTS : s->tl->n;
    
    
    for(i=0; (ret == APR_SUCCESS) && (i<max); i++) {
        btp_torrent* t = (btp_torrent*)s->tl->first->d;
        ret = btp_torrent_run(t);
        if(ret != APR_SUCCESS)
            btp_session_debug(
                s, ret, "btp_session_run_torrents: btp_torrent_run: "
            );
        
        if((eret = bt_llist_e_link_last(s->tl->first)) != APR_SUCCESS)
            return btp_session_debug(
                s, eret, "btp_session_run_torrents: bt_llist_e_link_last: "
            );
    }
    
    return ret;
}
