/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001  CodeFactory AB
 * Copyright (C) 2001  Richard Hult
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult <rhult@codefactory.se>
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <time.h>
#include <math.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <string.h>
#include <gal/e-table/e-table.h>
#include <gal/e-table/e-tree-table-adapter.h>
#include <gal/e-table/e-tree-memory.h>

#include "libmrproject/GNOME_MrProject.h"
#include "util/id-map.h"
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "util/time-utils.h"
#include "gantt-model.h"

#define DEBUG_TREE 0
#define DEBUG 0
#include "util/debug.h"

typedef struct {
	GNOME_MrProject_Task *task;
	ETreePath             path;
	GSList               *predecessors;
	GSList               *successors;
	GSList               *resources;
} TaskData;

typedef struct {
	GSList               *predecessors;
} Successors;

/* Private members. */
struct _GanttModelPriv {
	IdMap                *tdata_map;
	TaskData             *root_tdata;
	ETreePath             root_path;
};

enum {
	TASK_CHANGED,
	TASK_MOVED,
	TASK_DURATION_CHANGED,
	TASK_LINKED,
	TASK_UNLINKED,
	ALLOCATION_ASSIGNED,
	ALLOCATION_UNASSIGNED,
	ALLOCATED_RESOURCE_CHANGED,
	SCALE_CHANGED,
	LAST_SIGNAL
};

static void gantt_model_init                 (GanttModel      *model);
static void gantt_model_class_init           (GanttModelClass *klass);
static void gantt_model_destroy              (GtkObject       *object);
static void remove_all_resources             (GSList          *list);
static void gantt_model_unlink_task          (GanttModel      *model, 
					      GM_Id            id);
static gint my_e_tree_model_get_child_offset (ETreeModel      *etm,
					      ETreePath        node,
					      ETreePath        child);


GNOME_CLASS_BOILERPLATE (GanttModel, gantt_model, GtkObject, gtk_object);

static guint signals[LAST_SIGNAL] = { 0 };


static void
gantt_model_class_init (GanttModelClass *klass)
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass*) klass;
	object_class->destroy = gantt_model_destroy;

	klass->task_changed = NULL;
	signals[TASK_CHANGED] = 
		gtk_signal_new ("task_changed",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GanttModelClass, task_changed),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	signals[TASK_MOVED] = 
		gtk_signal_new ("task_moved",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__POINTER_INT_INT,
				GTK_TYPE_NONE, 3,
				GTK_TYPE_POINTER,
				GTK_TYPE_INT,
				GTK_TYPE_INT);

	signals[TASK_DURATION_CHANGED] = 
		gtk_signal_new ("task_duration_changed",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__POINTER_INT_INT,
				GTK_TYPE_NONE, 3,
				GTK_TYPE_POINTER,
				GTK_TYPE_INT,
				GTK_TYPE_INT);

	klass->scale_changed = NULL;
	signals[SCALE_CHANGED] = 
		gtk_signal_new ("scale_changed",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GanttModelClass, scale_changed),
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);

	klass->task_linked = NULL;
	signals[TASK_LINKED] = 
		gtk_signal_new ("task_linked",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GanttModelClass, task_linked),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	klass->task_unlinked = NULL;
	signals[TASK_UNLINKED] = 
		gtk_signal_new ("task_unlinked",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GanttModelClass, task_unlinked),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	klass->allocation_assigned = NULL;
	signals[ALLOCATION_ASSIGNED] = 
		gtk_signal_new ("allocation_assigned",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GanttModelClass, allocation_assigned),
				gtk_marshal_NONE__POINTER_INT,
				GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_INT);

	klass->allocation_unassigned = NULL;
	signals[ALLOCATION_UNASSIGNED] = 
		gtk_signal_new ("allocation_unassigned",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GanttModelClass, allocation_unassigned),
				gtk_marshal_NONE__INT_INT,
				GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);

	klass->allocated_resource_changed = NULL;
	signals[ALLOCATED_RESOURCE_CHANGED] =
		gtk_signal_new ("allocated_resource_changed",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GanttModelClass, 
						   allocated_resource_changed),
				gtk_marshal_NONE__INT_POINTER,
				GTK_TYPE_NONE,
				2, GTK_TYPE_INT, GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);	
}

static void
gantt_model_init (GanttModel *model)
{
	GanttModelPriv *priv;

	priv = g_new0 (GanttModelPriv, 1);
	model->priv = priv;
}

GanttModel *
gantt_model_new (ETreeModel           *etm,
		 ETreeTableAdapter    *etta,
		 ETree                *e_tree)
{
	GanttModel *model;

	model = gtk_type_new (TYPE_GANTT_MODEL);

	gtk_object_ref (GTK_OBJECT (etm));
	model->etm = etm;
	model->etta = E_TABLE_MODEL (etta);
	model->e_tree = e_tree;

	model->priv->root_path = e_tree_memory_node_insert (
		E_TREE_MEMORY (etm),
		NULL,
		0,
		NULL); /* We have no Task CORBA struct for the root node. */
	
	model->priv->tdata_map = id_map_new (0);
	model->priv->root_tdata = g_new0 (TaskData, 1);
	model->priv->root_tdata->path = model->priv->root_path;
	id_map_insert_id (model->priv->tdata_map, 0, model->priv->root_tdata);

	return model;
}

static void
gantt_model_destroy (GtkObject *object)
{
	GanttModel *model;

	model = GANTT_MODEL (object);
		
	gtk_object_unref (GTK_OBJECT (model->etm));
	g_free (model->priv);
	model->priv = NULL;

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

void
gantt_model_insert_task (GanttModel       *model,
			 GM_Id             parent_id,
			 GM_Id             sibling_id,
			 GM_TaskOrderType  type,
			 GM_Task          *task)
{
	TaskData  *tdata, *parent_tdata, *sibling_tdata;
	ETreePath  path;
	gint       offset;
	gboolean   success;
	
	g_return_if_fail (model != NULL);
	g_return_if_fail (IS_GANTT_MODEL (model));
	g_return_if_fail (task != NULL);
	g_return_if_fail (task->taskId != 0);

	parent_tdata = id_map_lookup (model->priv->tdata_map, parent_id);
	g_return_if_fail (parent_tdata);

	if (sibling_id > 0) {
		sibling_tdata = id_map_lookup (model->priv->tdata_map, sibling_id);
		g_return_if_fail (sibling_tdata);

		offset = my_e_tree_model_get_child_offset (model->etm,
							   parent_tdata->path,
							   sibling_tdata->path);

		if (offset != -1 && type == GNOME_MrProject_TASK_AFTER) {
			offset++;
		}
	}
	else {
		offset = -1;
	}

	/* Sigh. ETree won't put the row at its right place without the freeze/thaw.
	 * Should be removed when etree works.
	 */
	e_tree_memory_freeze (E_TREE_MEMORY (model->etm));
		
	path = e_tree_memory_node_insert (E_TREE_MEMORY (model->etm),
					  parent_tdata->path,
					  offset,
					  task);

	g_assert (path);

	/* See above. */
	e_tree_memory_thaw (E_TREE_MEMORY (model->etm));
	
	tdata = g_new0 (TaskData, 1);
	tdata->path = path;
	tdata->task = task;

	success = id_map_insert_id (model->priv->tdata_map, task->taskId, tdata);
	if (!success) {
		g_warning ("Already got task with Id %d.", task->taskId);
		tdata->path = NULL;
		tdata->task = NULL;
		g_free (tdata);
	}
}

static gboolean
remove_traverse (ETreeModel *etm, ETreePath path, gpointer data)
{
        GanttModel           *model;
        GNOME_MrProject_Task *task;
        TaskData             *tdata;

        model = GANTT_MODEL (data);
        task = e_tree_memory_node_get_data (E_TREE_MEMORY (etm), path);

        tdata = id_map_lookup (model->priv->tdata_map, task->taskId);
        if (tdata) {

		remove_all_resources (tdata->resources);
		tdata->resources = NULL;

		gantt_model_unlink_task (model, task->taskId);

                id_map_remove (model->priv->tdata_map, task->taskId);
		tdata->task = NULL;
		tdata->path = NULL;
                g_free (tdata);
        } else
                g_error ("Tried to remove nonexisting task.");

	/* remove the node too */
	e_tree_memory_node_remove (E_TREE_MEMORY (etm), path);

        return FALSE;
}

void
gantt_model_remove_task (GanttModel *model, GM_Id id)
{
	TaskData *tdata;
	
	g_return_if_fail (model != NULL);
	g_return_if_fail (IS_GANTT_MODEL (model));
	
	tdata = id_map_lookup (model->priv->tdata_map, id);
	g_assert (tdata);

	/* Remove the children. */
	e_tree_model_node_traverse (model->etm,
				    tdata->path,
				    remove_traverse, 
				    model);

	/* Remove the data of the subtree root as well. */
 	e_tree_memory_node_remove (E_TREE_MEMORY (model->etm), tdata->path);

	gantt_model_unlink_task (model, id);

	remove_all_resources (tdata->resources);
	id_map_remove (model->priv->tdata_map, id);
	tdata->task = NULL;
	tdata->path = NULL;
	g_free (tdata);
}

void 
gantt_model_remove_all_tasks (GanttModel *model)
{
	g_return_if_fail (IS_GANTT_MODEL (model));

	/* traverse the tree and kill all the children */
	e_tree_model_node_traverse (model->etm, model->priv->root_path, 
				    remove_traverse, model);

	/* FIXME: this should be done in the server. */

	e_tree_model_pre_change (model->etm);
	e_tree_model_node_data_changed (model->etm, model->priv->root_path);	
}

void
gantt_model_remove_tasks (GanttModel *model, GSList *ids)
{
	TaskData *tdata;
	GM_Id     id;
	GSList   *node;
	
	g_return_if_fail (model != NULL);
	g_return_if_fail (IS_GANTT_MODEL (model));

	/* FIXME: needed?? */
	e_tree_memory_freeze (E_TREE_MEMORY (model->etm));

	for (node = ids; node; node = node->next) {
		id = GPOINTER_TO_INT (node->data);
		tdata = id_map_lookup (model->priv->tdata_map, id);
		if (!tdata) {
			g_warning ("Trying to remove task that's not in the view.\n");
			continue;
		}

		/* Remove the children. */
		e_tree_model_node_traverse (model->etm,
					    tdata->path,
					    remove_traverse,
					    model);

		gantt_model_unlink_task (model, id);
		
		/* Remove the data of the subtree root as well. */
		e_tree_memory_node_remove (E_TREE_MEMORY (model->etm), tdata->path);
		id_map_remove (model->priv->tdata_map, id);
		tdata->task = NULL;
		tdata->path = NULL;
		g_free (tdata);
	}

	/* FIXME: needed? */
	e_tree_memory_thaw (E_TREE_MEMORY (model->etm));
}

/* Temp data structure used while reparenting a node in the ETree. */
typedef struct {
	GNode      *subtree_root_node;
	GHashTable *tdata_to_node_hash;
	GanttModel *model;
	ETreePath   new_parent_path;
	gint        new_position;
} ReparentData;

/* Phase one: build a GNode tree with the subtree that is to be
 * reparented.
 */
static gboolean
build_gnodes_traverse (ETreeModel *etm, ETreePath path, gpointer user_data)
{
	GNOME_MrProject_Task *task;
	TaskData             *tdata, *parent_tdata;
	GNode                *node, *parent_node;
	ETreePath             parent_path;
	ReparentData         *data;

	data = (ReparentData *) user_data;
	
	task = e_tree_memory_node_get_data (E_TREE_MEMORY (etm), path);
	g_assert (task);

	parent_path = e_tree_model_node_get_parent (etm, path);

	parent_tdata = id_map_lookup (data->model->priv->tdata_map, task->parentId);
	g_assert (parent_tdata);
	
	parent_node = g_hash_table_lookup (data->tdata_to_node_hash, parent_tdata);
	if (!parent_node) {
		parent_node = data->subtree_root_node;
	}

	tdata = id_map_lookup (data->model->priv->tdata_map, task->taskId);
	node = g_node_new (tdata);
	g_node_insert (parent_node, -1 /*offset*/, node);

	g_hash_table_insert (data->tdata_to_node_hash, tdata, node);
	
	return FALSE;
}

/* Phase two: take the GNode tree and insert the nodes into the ETreeModel
 * again.
 */
static gboolean
build_etree_traverse (GNode *node, gpointer user_data)
{
	ReparentData *data;
	TaskData     *tdata, *parent_tdata;
	GNode        *parent_node;
	ETreePath    *parent_path;
	gint          pos;

	data = (ReparentData *) user_data;
	tdata = node->data;

	parent_node = node->parent;
	if (!parent_node) {
		/* The subtree root. */
		parent_path = data->new_parent_path;
		pos = data->new_position;
	} else {
		parent_tdata = parent_node->data;
		parent_path = parent_tdata->path;
		pos = -1;
	}

	tdata->path = e_tree_memory_node_insert (E_TREE_MEMORY (data->model->etm),
					       parent_path,
					       pos,
					       tdata->task);

	return FALSE;
}

#if DEBUG_TREE
static gboolean
debug_tree (GNode *node, gpointer user_data)
{
	ReparentData *data;
	TaskData     *tdata;
	GNode        *parent_node;
	ETreePath     parent_path;
	gint          parent_id;
	gint          i, depth;

	data = user_data;
	tdata = node->data;

	depth = g_node_depth (node);
	
	for (i = 0; i < depth; i++) {
		g_print (".");
	}

	g_print ("%d\n", tdata->task->taskId);
	
	parent_node = node->parent;
	if (!parent_node) {
		parent_path = data->new_parent_path;
		parent_id = -1;
	} else {
		parent_path = ((TaskData *) parent_node->data)->path;
		parent_id = ((TaskData *) parent_node->data)->task->taskId;
	}

	return FALSE;
}
#endif

void
gantt_model_reparent_task (GanttModel *model,
			   GM_Id       parent_id,
			   GM_Id       task_id,
			   gint        position)
{
	ReparentData  data;
	TaskData     *tdata, *parent_tdata;

	g_return_if_fail (model != NULL);
	g_return_if_fail (IS_GANTT_MODEL (model));

	d(g_print ("reparent to %d pos %d\n", parent_id, position));
	
	e_tree_memory_freeze (E_TREE_MEMORY (model->etm));

	tdata = id_map_lookup (model->priv->tdata_map, task_id);
	g_assert (tdata);

	parent_tdata = id_map_lookup (model->priv->tdata_map, parent_id);
	g_assert (parent_tdata);

	data.tdata_to_node_hash = g_hash_table_new (g_direct_hash, 
						    g_direct_equal);
	data.subtree_root_node = g_node_new (tdata);
	data.model             = model;
	data.new_parent_path   = parent_tdata->path;
	data.new_position      = position;
	
	/* 1. Build a GNode tree from the subtree to reparent. */
	e_tree_model_node_traverse_preorder (model->etm,
					     tdata->path,
					     build_gnodes_traverse,
					     &data);

#if DEBUG_TREE
	/* Debug. */
	g_node_traverse (data.subtree_root_node,
			 G_PRE_ORDER,
			 G_TRAVERSE_ALL,
			 -1,
			 debug_tree,
			 &data);
#endif
	
	/* 2. Remove the subtree from the ETreeModel. */
	e_tree_memory_node_remove (E_TREE_MEMORY (model->etm), tdata->path);

	/* 3. Re-add the tree from the GNode tree. */
	g_node_traverse (data.subtree_root_node,
			 G_PRE_ORDER,
			 G_TRAVERSE_ALL,
			 -1,
			 build_etree_traverse,
			 &data);

	/* 4. Free stuff. */
	g_hash_table_destroy (data.tdata_to_node_hash);
	data.tdata_to_node_hash = NULL;
	g_node_destroy (data.subtree_root_node);
	data.subtree_root_node = NULL;

	/* 5. Update the task parent. */
	tdata->task->parentId = parent_id;
	
	e_tree_memory_thaw (E_TREE_MEMORY (model->etm));
} 

void
gantt_model_reposition_task (GanttModel       *model,
			     GM_Id             task_id,
			     GM_Id             sibling_id,
			     GM_TaskOrderType  type)
{
	TaskData *tdata;
	gint      position;

	g_return_if_fail (model != NULL);
	g_return_if_fail (IS_GANTT_MODEL (model));

	tdata = id_map_lookup (model->priv->tdata_map, task_id);
	g_assert (tdata);

	position = gantt_model_task_get_child_offset (model, sibling_id);

	/* Move == Reparent with the same parent, but different position. */
	gantt_model_reparent_task (model,
				   tdata->task->parentId,
				   task_id,
				   position);
}

void
gantt_model_link (GanttModel           *model,
		  const GM_Dependency *dependency)
{
	TaskData      *tdata, *pre_tdata;
	GM_Dependency *dependency_copy;
	
	tdata = id_map_lookup (model->priv->tdata_map, dependency->taskId);
	pre_tdata = id_map_lookup (model->priv->tdata_map, dependency->predecessorId);
	if (!tdata || !pre_tdata) {
		g_warning ("Eek don't have the tasks to link.");
		return;
	}

	dependency_copy = corba_util_dependency_duplicate (dependency);
	tdata->predecessors = g_slist_prepend (tdata->predecessors, dependency_copy);
	pre_tdata->successors = g_slist_prepend (pre_tdata->successors, dependency_copy);

	gtk_signal_emit (GTK_OBJECT (model), 
			 signals[TASK_LINKED],
			 dependency_copy);
}

static GSList *
remove_predecessor (GSList *list, GM_Id id)
{
	GSList *l;
	
	for (l = list; l; l = l->next) {
		GM_Dependency *dependency;

		dependency = l->data;

		if (dependency->predecessorId == id) {
			list = g_slist_remove_link (list, l);
			g_slist_free (l);
			break;
		}
	}
	return list;
}

static GSList *
remove_successor (GSList *list, GM_Id id)
{
	GSList *l;
	
	for (l = list; l; l = l->next) {
		GM_Dependency *dependency;

		dependency = l->data;

		if (dependency->taskId == id) {
			list = g_slist_remove_link (list, l);
			g_slist_free (l);
			break;
		}
	}
	return list;
}

void
gantt_model_unlink (GanttModel                 *model,
		    const GNOME_MrProject_Dependency *dependency)

{
	TaskData *tdata, *pre_tdata;
	
	tdata = id_map_lookup (model->priv->tdata_map, dependency->taskId);
	pre_tdata = id_map_lookup (model->priv->tdata_map, dependency->predecessorId);
	if (!tdata || !pre_tdata) {
		g_warning ("Eek don't have the tasks to unlink.");
		return;
	}

	/* Remove tasks from predecessor/successor lists. */
	tdata->predecessors = remove_predecessor (tdata->predecessors, dependency->predecessorId);
	pre_tdata->successors = remove_successor (pre_tdata->successors, dependency->taskId);

	gtk_signal_emit (GTK_OBJECT (model), 
			 signals[TASK_UNLINKED],
			 dependency);

	/* FIXME: must CORBA_free the dependency here. */
}

/* unlinks one task, that is: it removes all links to any other tasks */
static void
gantt_model_unlink_task (GanttModel *model, GM_Id id)
{
	GSList *node;
	TaskData *pre_tdata, *tdata;

	g_return_if_fail (IS_GANTT_MODEL (model));
	
	if ((tdata = id_map_lookup (model->priv->tdata_map, id)) == NULL) {
		g_warning ("No such task(%d) in tdata_map", id);
		return;
	}


	/* remove successors */
	for (node = tdata->successors; node; node = node->next) {
		GNOME_MrProject_Dependency *dep;
		
		dep = (GNOME_MrProject_Dependency *)node->data;
		g_assert (dep != NULL); /* just in case */

		if (dep->taskId == tdata->task->taskId) {
			g_warning ("big_T never expected this to occurs" 
				   ", beat him up with a stick");
			g_assert_not_reached();
		}
		else if (dep->predecessorId == tdata->task->taskId) {
			pre_tdata = id_map_lookup (model->priv->tdata_map, dep->taskId);
			pre_tdata->predecessors = remove_predecessor (pre_tdata->predecessors, 
								      dep->predecessorId);
		}
		else {
			g_warning ("Arghh... No such task to unlink\n");
		}

		gtk_signal_emit (GTK_OBJECT (model),
				 signals[TASK_UNLINKED],
				 dep);
	}
	g_slist_free (tdata->successors);
	tdata->successors = NULL;

	/* remove predecessors */
	for (node = tdata->predecessors; node; node = node->next) {
		GNOME_MrProject_Dependency *dep;

		dep = (GNOME_MrProject_Dependency *)node->data;
		g_assert (dep != NULL); /* just in case */

		if (dep->taskId == tdata->task->taskId) {
			pre_tdata = id_map_lookup (model->priv->tdata_map, 
						   dep->predecessorId);
			pre_tdata->successors = remove_successor (pre_tdata->successors, 
								  dep->taskId);

		}
		else if (dep->predecessorId == tdata->task->taskId) {
			g_warning ("big_T never expected this to occurs"
				   ", beat him up with a stick");
			g_assert_not_reached();
		}
		else {
			g_warning ("Arghh... No such task to unlink\n");
		}
		gtk_signal_emit (GTK_OBJECT (model),
				 signals[TASK_UNLINKED],
				 dep); 

	}
	g_slist_free (tdata->predecessors);
	tdata->predecessors = NULL; 
}





static void
remove_all_resources (GSList *list)
{
	GSList                   *node;
	GNOME_MrProject_Resource *resource;

	for (node = list; node; node = node->next) {

		resource = (GNOME_MrProject_Resource *)node->data;
		g_assert(resource != NULL); /* just in case */

		CORBA_free (resource);
		node->data = NULL;
	}
	if (list) {
		g_slist_free (list);
	}
}

static GSList *
remove_resource (GSList *list, GM_Id id)
{
	GSList *l;
	
	for (l = list; l; l = l->next) {
		GNOME_MrProject_Resource *resource;

		resource = l->data;
		if (resource->resourceId == id) {
			list = g_slist_remove_link (list, l);
			l->data = NULL;
			g_slist_free (l);
			CORBA_free (resource);
			break;
		}
	}
	return list;
}
	
void
gantt_model_assign_allocation (GanttModel        *model,
			       const GM_Resource *resource,
			       GM_Id              task_id)
{
	TaskData    *tdata;
	GM_Resource *resource_copy;
		
	tdata = id_map_lookup (model->priv->tdata_map, task_id);
	if (!tdata) {
		g_warning ("Eek don't have the task to assign to (%d).", (gint) task_id);
		return;
	}

	resource_copy = corba_util_resource_duplicate (resource);
	tdata->resources = g_slist_append (tdata->resources, resource_copy);

	gtk_signal_emit (GTK_OBJECT (model), 
			 signals[ALLOCATION_ASSIGNED],
			 resource_copy,
			 task_id);
}

void
gantt_model_unassign_allocation (GanttModel *model,
				 GM_Id       resource_id,
				 GM_Id       task_id)
{
	TaskData *tdata;

	g_return_if_fail (model != NULL);
	g_return_if_fail (IS_GANTT_MODEL (model));
	
	tdata = id_map_lookup (model->priv->tdata_map, task_id);
	if (!tdata) {
		g_warning ("Eek don't have the task to unassign from (%d).", (gint) task_id);
		return;
	}

	tdata->resources = remove_resource (tdata->resources, resource_id);
	
	gtk_signal_emit (GTK_OBJECT (model), 
			 signals[ALLOCATION_UNASSIGNED],
			 resource_id,
			 task_id);
}

void
gantt_model_update_allocated_resource (GanttModel  *model,
				       GM_Id        task_id,
				       GM_Resource *resource) 
{
	TaskData    *tdata;
	GSList      *node;
	GM_Resource *tmp_resource;
	
	g_return_if_fail (model != NULL);
	g_return_if_fail (IS_GANTT_MODEL (model));
	
	tdata = id_map_lookup (model->priv->tdata_map, task_id);
	
	if (!tdata) {
		g_warning ("Eek don't have the task to update resource for.");
		return;
	}
	
	for (node = tdata->resources; node; node = node->next) {
		tmp_resource = (GM_Resource *) node->data;
		
		if (tmp_resource->resourceId == resource->resourceId) {
			/* FIX: Should be able update all fields. */
			CORBA_free (tmp_resource->name);
			tmp_resource->name = CORBA_string_dup (resource->name);
			
			gtk_signal_emit (GTK_OBJECT (model),
					 signals[ALLOCATED_RESOURCE_CHANGED],
					 task_id,
					 resource);
		}
	}
}

GNOME_MrProject_Task *
gantt_model_get_task (GanttModel *model, GM_Id id)
{
	TaskData *tdata;
	
	g_return_val_if_fail (model != NULL, NULL);
	g_return_val_if_fail (IS_GANTT_MODEL (model), NULL);

	tdata = id_map_lookup (model->priv->tdata_map, id);

	if (tdata) {
		return tdata->task;
	} else {
		return NULL;
	}
}

GNOME_MrProject_Task *
gantt_model_get_task_at_row (GanttModel *model, gint row)
{
	ETreePath path;
	
	g_return_val_if_fail (model != NULL, NULL);
	g_return_val_if_fail (IS_GANTT_MODEL (model), NULL);

	path = e_tree_node_at_row (model->e_tree, row);
	if (!path) {
		return NULL;
	}
	
	return e_tree_memory_node_get_data (E_TREE_MEMORY (model->etm), path);
}

GNOME_MrProject_Task *
gantt_model_get_task_at_path (GanttModel *model, ETreePath path)
{
	g_return_val_if_fail (model != NULL, NULL);
	g_return_val_if_fail (IS_GANTT_MODEL (model), NULL);
	g_return_val_if_fail (path != NULL, NULL);

	return e_tree_memory_node_get_data (E_TREE_MEMORY (model->etm), path);
}

void
gantt_model_task_changed (GanttModel *model, GM_Id id)
{
	GM_Task *task;
	
	g_return_if_fail (model != NULL);
	g_return_if_fail (IS_GANTT_MODEL (model));
	g_return_if_fail (id != 0);

	task = gantt_model_get_task (model, id);
	if (task) {
		gtk_signal_emit (GTK_OBJECT (model), signals[TASK_CHANGED], task);
	}
}

void
gantt_model_task_moved (GanttModel *model, GM_Id id, time_t start, GM_TaskEdge edge)
{
	GM_Task *task;
	
	g_return_if_fail (model != NULL);
	g_return_if_fail (IS_GANTT_MODEL (model));
	g_return_if_fail (id != 0);

	task = gantt_model_get_task (model, id);
	if (task) {
		gtk_signal_emit (GTK_OBJECT (model), signals[TASK_MOVED], task, start, edge);
	}
}

void
gantt_model_task_duration_changed (GanttModel *model,
				   GM_Id       id,
				   GM_Time     duration,
				   GM_TaskEdge edge)
{
	GM_Task *task;
	
	g_return_if_fail (model != NULL);
	g_return_if_fail (IS_GANTT_MODEL (model));
	g_return_if_fail (id != 0);

	task = gantt_model_get_task (model, id);
	if (task) {
		gtk_signal_emit (GTK_OBJECT (model),
				 signals[TASK_DURATION_CHANGED],
				 task,
				 (gint) duration,
				 (gint) edge);
	}
}

void
gantt_model_update_task (GanttModel *model,
			 GM_Id       id,
			 GM_Task    *task)
{
	TaskData *tdata;
	
	g_return_if_fail (model != NULL);
	g_return_if_fail (IS_GANTT_MODEL (model));
	g_return_if_fail (id != 0);

	tdata = id_map_lookup (model->priv->tdata_map, id);
	if (!tdata) {
		g_warning ("Updated nonexistant task?");
		return;
	}

	corba_util_task_update (tdata->task, task, TASK_CHANGE_ALL);

	e_tree_model_pre_change (model->etm);
	e_tree_model_node_data_changed (model->etm, tdata->path);
}

gboolean
gantt_model_task_is_leaf (GanttModel *model, GM_Id id)
{
	TaskData *tdata;
       	
	g_return_val_if_fail (model != NULL, TRUE);
	g_return_val_if_fail (IS_GANTT_MODEL (model), TRUE);

	tdata = id_map_lookup (model->priv->tdata_map, id);
	if (!tdata) {
		return TRUE;
	}

	return (tdata->task->type == GNOME_MrProject_TASK_NORMAL);
}

GM_Id
gantt_model_get_task_above (GanttModel *model, GM_Id id)
{
	TaskData  *tdata;
	ETreePath  parent_path, prev_path;
	GM_Task   *task;
	gint       row, prev_row;
	gint	   depth, prev_depth;
	gint	   i;
	
	g_return_val_if_fail (model != NULL, -1);
	g_return_val_if_fail (IS_GANTT_MODEL (model), -1);
	g_return_val_if_fail (id > 0, -1);

	tdata = id_map_lookup (model->priv->tdata_map, id);
	if (!tdata) {
		return -1;
	}

	/* Get the row just above the row id the task. */
	row = e_tree_row_of_node (model->e_tree, tdata->path);
	prev_row = e_tree_get_prev_row (model->e_tree, row);
	if (prev_row == -1) {
		g_warning ("Error: There is no row above the selected one.");
		return -1;
	}

	prev_path = e_tree_node_at_row (model->e_tree, prev_row);
	task = e_tree_memory_node_get_data (E_TREE_MEMORY (model->etm), prev_path);

	/* Find the ancestor of the row above the selected row
	 * that has the same depth as the selected row.
	 */
	depth = e_tree_model_node_depth (model->etm, tdata->path);
	prev_depth = e_tree_model_node_depth (model->etm, prev_path);

	parent_path = prev_path;
	for (i = prev_depth; i > depth; i--) {
		parent_path = e_tree_model_node_get_parent (model->etm, parent_path);
	}

        task = e_tree_memory_node_get_data (E_TREE_MEMORY (model->etm), parent_path);
	return task->taskId;
}

/* Get the previous sibling, view-wise. */
GM_Id
gantt_model_get_prev_sibling (GanttModel *model, GM_Id id)
{
	TaskData  *tdata;
	ETreePath  prev_path;
	GM_Task   *task;
	gint       row, prev_row;
	
	g_return_val_if_fail (model != NULL, -1);
	g_return_val_if_fail (IS_GANTT_MODEL (model), -1);
	g_return_val_if_fail (id > 0, -1);

	tdata = id_map_lookup (model->priv->tdata_map, id);
	if (!tdata) {
		return -1;
	}

	row = e_tree_row_of_node (model->e_tree, tdata->path);

	/* Search upwards to find a node at the same level 
	 * as ourselves - i.e. a sibling.
	 */
	for (; row >= 0; row--) {
		prev_row = e_tree_get_prev_row (model->e_tree, row);

		if (prev_row == -1) {
			return -1;
		}
		
		prev_path = e_tree_node_at_row (model->e_tree, prev_row);
		if (!prev_path) {
			return -1;
		}
		
		task = e_tree_memory_node_get_data (E_TREE_MEMORY (model->etm), prev_path);

		/* Sibling? */
		if (tdata->task->parentId == task->parentId) {
			return task->taskId;
		}
	}

	/* If we get here no sibling was found */
	return -1;
}

/* Get the next sibling, view-wise. */
GM_Id
gantt_model_get_next_sibling (GanttModel *model, GM_Id id)
{
	TaskData  *tdata;
	ETreePath  next_path;
	GM_Task   *task;
	gint       row, next_row;
	
	g_return_val_if_fail (model != NULL, -1);
	g_return_val_if_fail (IS_GANTT_MODEL (model), -1);
	g_return_val_if_fail (id > 0, -1);

	tdata = id_map_lookup (model->priv->tdata_map, id);
	if (!tdata) {
		return -1;
	}

	row = e_tree_row_of_node (model->e_tree, tdata->path);

	/* Search downwards to find a node at the same level 
	 * as ourselves - i.e. a sibling. 
	 */
	while (row != -1) {
		next_row = e_tree_get_next_row (model->e_tree, row);

		g_print ("row %d, next_row %d\n", row, next_row);
		
		if (next_row == -1) {
			return -1;
		}
		
		next_path = e_tree_node_at_row (model->e_tree, next_row);
		if (!next_path) {
			return -1;
		}
		
		task = e_tree_memory_node_get_data (E_TREE_MEMORY (model->etm),
						    next_path);

		/* Sibling? */
		if (tdata->task->parentId == task->parentId) {
			return task->taskId;
		}

		row++;
	}
	
	return -1;
}

ETreePath
gantt_model_get_path (GanttModel *model, GM_Id id)
{
	TaskData *tdata;
	
	g_return_val_if_fail (model != NULL, NULL);
	g_return_val_if_fail (IS_GANTT_MODEL (model), NULL);

	tdata = id_map_lookup (model->priv->tdata_map, id);
	if (!tdata) {
		return NULL;
	}

	return tdata->path;
}

static gint
my_e_tree_model_get_child_offset (ETreeModel *etm,
				  ETreePath   node,
				  ETreePath   child)
{
	ETreePath *children;
	gint       i, n;
	
	n = e_tree_model_node_get_children (etm, node, &children);
	for (i = 0; i < n; i++) {
		if (children[i] == child) {
			return i;
		}
	}

	return -1;
}

gint
gantt_model_task_get_child_offset (GanttModel *model,
				   GM_Id       id)
{
	TaskData  *tdata;
	ETreePath  parent_path;
	gint       pos;

	g_return_val_if_fail (model != NULL, -1);
	g_return_val_if_fail (IS_GANTT_MODEL (model), -1);
	g_return_val_if_fail (id > 0, -1);

	tdata = id_map_lookup (model->priv->tdata_map, id);
	if (!tdata) {
		return -1;
	}

	parent_path = e_tree_model_node_get_parent (model->etm, tdata->path);
	pos = my_e_tree_model_get_child_offset (model->etm, parent_path, tdata->path);

	g_print ("Id %d has offset %d\n", id, pos);
	
	return pos;
}

/* Get all predecessors of this task.
 */
GSList *
gantt_model_task_get_predecessors (GanttModel *model,
				   GM_Id       id)
{
	TaskData *tdata;

	tdata = id_map_lookup (model->priv->tdata_map, id);
	if (tdata) {
		return tdata->predecessors;
	} else {
		return NULL;
	}
}

/* Get all tasks that has this task as predecessor.
 */
GSList *
gantt_model_task_get_successors (GanttModel *model,
				 GM_Id       id) 
{
	TaskData *tdata;

	tdata = id_map_lookup (model->priv->tdata_map, id);
	if (!tdata) {
		return NULL;
	}

	return tdata->successors;
}

GSList *
gantt_model_task_get_resources (GanttModel *model,
				GM_Id       id) 
{
	TaskData *tdata;

	tdata = id_map_lookup (model->priv->tdata_map, id);
	if (!tdata) {
		return NULL;
	}

	return tdata->resources;
}

ETreePath
gantt_model_get_root_path (GanttModel *model)
{
	return model->priv->root_path;
}

static void
get_first_func (gpointer key, gpointer value, gpointer user_data)
{
	time_t   *first;
	TaskData *tdata;

	first = user_data;
	tdata = value;

	if (!tdata->task) {
		/* Root. */
		return;
	}

	if (tdata->task->start < *first) {
		*first = tdata->task->start;
	}
}

time_t
gantt_model_get_first_time (GanttModel *model)
{
	time_t first;
	
	g_return_val_if_fail (model != NULL, -1);
	g_return_val_if_fail (IS_GANTT_MODEL (model), -1);

	first = G_MAXINT;
	id_map_foreach (model->priv->tdata_map,
			get_first_func,
			&first);

	if (first == G_MAXINT) {
		first = -1;
	}
	return first;
}

static void
get_last_func (gpointer key, gpointer value, gpointer user_data)
{
	time_t   *last;
	TaskData *tdata;

	last = user_data;
	tdata = value;

	if (!tdata->task) {
		/* Root. */
		return;
	}
	
	if (tdata->task->end > *last) {
		*last = tdata->task->end;
	}
}

time_t
gantt_model_get_last_time (GanttModel *model)
{
	time_t last;
	
	g_return_val_if_fail (model != NULL, -1);
	g_return_val_if_fail (IS_GANTT_MODEL (model), -1);

	last = 0;
	id_map_foreach (model->priv->tdata_map,
			get_last_func,
			&last);

	if (last == 0) {
		last = -1;
	}
	return last;
}

gint
gantt_model_get_num_tasks (GanttModel *model)
{
	g_return_val_if_fail (model != NULL, -1);
	g_return_val_if_fail (IS_GANTT_MODEL (model), -1);

	/* -1 because the invisible root does not count. */
	return id_map_size (model->priv->tdata_map) - 1;
}

