/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <glib.h>
#include <gtk/gtkobject.h>
#include <liboaf/liboaf.h>
#include <bonobo/bonobo-object.h>
#include <bonobo/bonobo-exception.h>
#include <gnome-xml/parser.h>
#include <gnome-xml/parserInternals.h>
#include <gnome-xml/xmlmemory.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnome/gnome-i18n.h>
#include "util/type-utils.h"
#include "file-task-manager.h"
#include "file-resource-manager.h"
#include "file-allocation-manager.h"
#include "file-project.h"


static void file_project_init       (FileProject      *project);
static void file_project_class_init (FileProjectClass *klass);

GNOME_CLASS_BOILERPLATE (FileProject, file_project,
			 Project, project);

#define d(x)

static void
fp_load (Project           *project,
	 const gchar       *uri,
	 CORBA_Environment *ev)
{
	const gchar           *mime_type;
	CORBA_Environment      e;
	CORBA_Object           filter_co;
	gchar                 *query;
	OAF_ServerInfoList    *oaf_result;
	guint                  i;
	OAF_ServerInfo         server;
	gboolean               loaded_ok, format_ok;
	GnomeVFSURI           *vfs_uri;

	vfs_uri = gnome_vfs_uri_new (uri);
	if (!gnome_vfs_uri_exists (vfs_uri)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_Project_URINotFound,
				     NULL);
		gnome_vfs_uri_unref (vfs_uri);
		return;
	}

	gnome_vfs_uri_unref (vfs_uri);
	
	mime_type = gnome_vfs_mime_type_from_name (uri);

	/* This is really just a workaround for a bug in gnome-vfs in that
	 * it does not use GNOME_PATH. Before this is fixed we do this.
	 */
	if (!strcmp (mime_type, "application/octet-stream")) {
		/* If we don't find a proper loader, always try the
		 * native format (if the user saved a file with no extention
		 * or if the mime type detection didn't work.
		 */
		
		if (strstr (uri, ".mrproject") == (uri + (size_t) strlen (uri) - 10)) {
			mime_type = "application/x-mrproject";
			g_message ("Workaround: no mime info, .mrproject file.");
		}
		else if (strstr (uri, ".mpx") == (uri + (size_t) strlen (uri) - 4)) {
			mime_type = "application/vnd.ms-project";
			g_message ("Workaround: no mime info, .mpx file.");
		}
		else {
			mime_type = "application/x-mrproject";
			g_message ("Workaround: no mime info, unknown file.");
		}
	}
	
	CORBA_exception_init (&e);
	
	/* Query for all components that can load a project file. */
	query = g_strdup_printf ("repo_ids.has ('IDL:GNOME/MrProject/FileFilter:1.0')"
				 "AND mrproject:mime_types.has ('%s')", mime_type);
	
	oaf_result = oaf_query (query, NULL, &e);

	if (BONOBO_EX (&e) || !oaf_result || oaf_result->_length == 0) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_Project_MethodNotSupported,
				     NULL);

		CORBA_free (oaf_result);
		CORBA_exception_free (&e);
		return;
	}
	
	loaded_ok = FALSE;
	if (!BONOBO_EX (&e) && oaf_result != NULL) {
		for (i = 0; i < oaf_result->_length; i++) {
			server = oaf_result->_buffer[i];

			filter_co = oaf_activate_from_id (server.iid,
							  0,	/* OAF_ActivationFlags */
							  NULL, /* OAF_ActivationID* */
							  &e);
			
			if (BONOBO_EX (&e) || !filter_co) {
				CORBA_exception_free (&e);
				continue;
			}

			format_ok = GNOME_MrProject_FileFilter_probe (
				filter_co,
				uri,
				&e);
			
			if (!format_ok || BONOBO_EX (&e)) {
				CORBA_exception_free (&e);
				bonobo_object_release_unref (filter_co, NULL);
				continue;
			}

			GNOME_MrProject_FileFilter_load (
				filter_co,
				uri,
				BONOBO_OBJREF (project),
				&e);

			if (BONOBO_EX (&e)) {
				/* Ok, try the next one. */
				bonobo_object_release_unref (filter_co, NULL);
				CORBA_exception_free (&e);
				continue;
			}
				
			loaded_ok = TRUE;
			bonobo_object_release_unref (filter_co, NULL);
			break;
		}
		
		if (oaf_result) {
			CORBA_free (oaf_result);
		}
				
		CORBA_exception_free (&e);
	}

	g_free (query);
	CORBA_exception_free (&e);
	
	if (!loaded_ok) {
		corba_util_set_exception (ev,
					  GNOME_MrProject_Project_IOFailure,
					  GNOME_MrProject_ERROR_WARNING,
					  _("Invalid file."));
	}

	d(g_print ("Loading done.\n"));
}

static void
fp_save (Project           *project,
	 const gchar       *uri,
	 CORBA_Environment *ev)
{
	const gchar           *mime_type;
	CORBA_Environment      e;
	CORBA_Object           filter_co;
	gchar                 *query;
	OAF_ServerInfoList    *oaf_result;
	guint                  i;
	OAF_ServerInfo         server;
	gboolean               saved_ok;
	
	CORBA_exception_init (&e);

	mime_type = gnome_vfs_mime_type_from_name (uri);
	if (!strcmp (mime_type, "application/octet-stream")) {
		/* If we don't find a proper loader, always try the
		 * native format (if the user saved a file with no extention
		 * or if the mime type detection didn't work.
		 */
		mime_type = "application/x-mrproject";
	}
	
	/* Query for all components that can save a project file. */
	query = g_strdup_printf ("repo_ids.has ('IDL:GNOME/MrProject/FileFilter:1.0')"
				 "AND mrproject:mime_types.has ('%s')", mime_type);
	
	oaf_result = oaf_query (query, NULL, &e);
		
	saved_ok = FALSE;
	if (!BONOBO_EX (&e) && oaf_result != NULL) {
		for (i = 0; i < oaf_result->_length; i++) {
			server = oaf_result->_buffer[i];

			filter_co = oaf_activate_from_id (server.iid,
							  0,	/* OAF_ActivationFlags */
							  NULL, /* OAF_ActivationID* */
							  &e);
			
			if (BONOBO_EX (&e) || !filter_co) {
				CORBA_exception_free (&e);
				continue;
			}

			GNOME_MrProject_FileFilter_save (
				filter_co,
				uri,
				BONOBO_OBJREF (project),
				&e);
			
			if (BONOBO_EX (&e)) {
				/* Ok, try the next one. */
				bonobo_object_release_unref (filter_co, NULL);
				CORBA_exception_free (&e);
				continue;
			}

			saved_ok = TRUE;
			bonobo_object_release_unref (filter_co, NULL);
			break;
		}
		
		if (oaf_result) {
			CORBA_free (oaf_result);
		}
		
		CORBA_exception_free (&e);
	}

	g_free (query);
	
	if (!saved_ok) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_Project_IOFailure,
				     NULL);
	}
}

static void
file_project_destroy (GtkObject *object)
{
	d(puts (__FUNCTION__));
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy) {
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
	}
}

static void
file_project_class_init (FileProjectClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ProjectClass   *project_class = (ProjectClass *) klass;

	object_class->destroy = file_project_destroy;

	project_class->load = fp_load;
	project_class->save = fp_save;
}

Project *
file_project_new (void)
{
	Project *project;
	
	project = gtk_type_new (FILE_PROJECT_TYPE);

	return project;
}

static void
file_project_init (FileProject *file_project)
{
	Project *project;

	project = PROJECT (file_project);
	
	/* Add a TaskManager object. */
	project->task_manager = file_task_manager_new (project->event_source);
 	bonobo_object_add_interface (BONOBO_OBJECT (project), 
				     BONOBO_OBJECT (project->task_manager));

	/* Add a ResourceManager object. */
	project->resource_manager = 
		file_resource_manager_new (project->event_source);
	bonobo_object_add_interface (BONOBO_OBJECT (project), 
				     BONOBO_OBJECT (project->resource_manager));

	/* Add an AllocationManager object. */
	project->allocation_manager = 
		file_allocation_manager_new (project->event_source);
	bonobo_object_add_interface (BONOBO_OBJECT (project),
				     BONOBO_OBJECT (project->allocation_manager));
	allocation_manager_start_listen (project->allocation_manager);
}
