/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <time.h>
#include <string.h>
#include <bonobo.h>
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "util/time-utils.h"
#include "util/id-map.h"
#include "util/marshallers.h"
#include "file-task-model.h"
#include "file-task-manager.h"

static void file_task_manager_init          (FileTaskManager       *manager);
static void file_task_manager_class_init    (FileTaskManagerClass  *klass);

static TaskModel *ftm_get_task_model        (TaskManager           *manager, 
					     GNOME_MrProject_Id     id);
static TaskModel *ftm_get_parent_task_model (TaskManager           *manager,
					     TaskModel             *task);
static TaskModel *ftm_get_first_sibling     (TaskManager           *manager, 
					     TaskModel             *task_model);
static TaskModel *ftm_get_next_sibling      (TaskManager           *manager, 
					     TaskModel             *task_model);
static GSList    *ftm_get_children          (TaskManager           *manager,
					     TaskModel             *task_model);
static gint       ftm_get_task_depth        (TaskManager           *manager,
					     TaskModel             *task_model);
static void       ftm_remove_task_model     (TaskManager           *manager,
					     TaskModel             *task_model);
static void       dump_tree                 (TaskManager           *manager);
static gboolean   check_for_loop            (GNode                 *parent,
					     GNode                 *node);


GNOME_CLASS_BOILERPLATE (FileTaskManager, file_task_manager,
			 TaskManager, task_manager);


/* Private members. */
struct _FileTaskManagerPriv {
	IdMap      *task_map;
	IdMap      *dependency_map;
	gboolean    block_bubble : 1; /* Used to prevent recursive bubbling. */
	gint        bubble_idle_id;

	IdMap      *notes_map;
};

static TaskModel *
ftm_insert_task (TaskManager                *manager, 
		 const GNOME_MrProject_Task *task,
		 TaskModel                  *parent_task_model,
		 TaskModel                     *sibling_task_model,
		 GM_TaskOrderType               type,
		 CORBA_Environment          *ev)
{
	FileTaskManagerPriv *priv;
	FileTaskModel       *ftm, *parent_ftm;
	GNode               *node, *parent_node;
	gint                 pos;

	priv = FILE_TASK_MANAGER (manager)->priv;

	/* This copies the task. */
	ftm = FILE_TASK_MODEL (file_task_model_new (task));
	
	if (ftm->task->taskId != -1) {
		if (!id_map_insert_id (priv->task_map, ftm->task->taskId, ftm)) {
			g_warning ("Task with id %d already in task tree.",
				   (int) ftm->task->taskId);
			gtk_object_destroy (GTK_OBJECT (ftm));
			return NULL;
		}
	} else {
		ftm->task->taskId = id_map_insert (priv->task_map, ftm);
		if (ftm->task->taskId == -1) {
			/* FIXME: raise exception. */
			gtk_object_destroy (GTK_OBJECT (ftm));
			return NULL; 
		}
	}

	parent_ftm = FILE_TASK_MODEL (parent_task_model);
	g_return_val_if_fail (parent_ftm != NULL, NULL);

	parent_node = parent_ftm->node;
	node = ftm->node;
	g_return_val_if_fail (node != NULL, NULL);

	/* Don't allow loops in the tree (wouldn't really be a tree otherwise... ;) */
	if (check_for_loop (parent_node, node)) {
		g_warning ("Insert would create loop! Id: %d, parent: %d",
			   ftm->task->taskId, parent_ftm->task->parentId);
		id_map_remove (priv->task_map, ftm->task->taskId);
		dump_tree (manager);
		return NULL;
	}

	pos = -1;
	if (sibling_task_model) {
		pos = g_node_child_position (
			parent_node,
			FILE_TASK_MODEL (sibling_task_model)->node);
		if (type == GNOME_MrProject_TASK_AFTER) {
			pos++;
		}
	}
	
	g_node_insert (parent_node, pos, node);
	return TASK_MODEL (ftm);
}

static void
ftm_remove_tasks (TaskManager       *manager, 
		  GSList            *task_list,
		  CORBA_Environment *ev)
{
	GNOME_MrProject_Task *task;
	TaskModel            *task_model;
	GSList               *list;

	for (list = task_list; list; list = list->next) {
		task = list->data;

		task_model = ftm_get_task_model (manager, task->taskId);
		gtk_object_unref (GTK_OBJECT (task_model));
	}
}

static void
ftm_reparent_task (TaskManager       *manager, 
		   TaskModel         *task_model,
		   TaskModel         *parent_task_model,
		   CORBA_Environment *ev)
{
	FileTaskModel *ftm, *parent_ftm;

	ftm = FILE_TASK_MODEL (task_model);
	parent_ftm = FILE_TASK_MODEL (parent_task_model);
	
	/* Detach the subtree. */
	g_node_unlink (ftm->node);
	
	/* Insert the subtree in its new place. */
	g_node_insert (parent_ftm->node, -1, ftm->node);
}

static void 
ftm_reposition_task (TaskManager                   *manager, 
		     TaskModel                     *task_model,
		     TaskModel                     *sibling_task_model,
		     GNOME_MrProject_TaskOrderType  type,
		     CORBA_Environment             *ev)
{
	FileTaskModel        *ftm, *parent_ftm, *sibling_ftm;
	TaskModel            *parent_tm;
	gint                  position;

	ftm = FILE_TASK_MODEL (task_model);

	parent_tm = task_manager_get_parent_task_model (manager, task_model);
	if (!parent_tm) {
		return;
	}
	
	parent_ftm = FILE_TASK_MODEL (parent_tm);
	sibling_ftm = FILE_TASK_MODEL (sibling_task_model);
	
	/* Detach the subtree. */
	g_node_unlink (ftm->node);

	/* Insert the subtree in its new place. */
	position = g_node_child_position (sibling_ftm->node->parent,
					  sibling_ftm->node);
	switch (type) {
	case GNOME_MrProject_TASK_BEFORE:
		break;
		
	case GNOME_MrProject_TASK_AFTER:
		position++;
		break;
	}

	g_node_insert (parent_ftm->node, position, ftm->node);
}

static GNOME_MrProject_Task *
ftm_get_task (TaskManager              *manager, 
	      const GNOME_MrProject_Id  id,
	      CORBA_Environment        *ev)
{
	TaskModel            *task_model;
	FileTaskModel        *ftm;

	if (id != 0) {
		task_model = task_manager_get_task_model (manager, id);
	} else {
		task_model = NULL;
	}

	if (task_model == NULL) {
		return CORBA_OBJECT_NIL;
	}

	ftm = FILE_TASK_MODEL (task_model);
	return ftm->task;
}

static gboolean
traverse_get_all_tasks (TaskManager *manager,
			TaskModel   *task_model,
			gpointer     data)
{
	GSList **list;
	GM_Task *task;

	task = task_model_get_task (task_model);
	
	list = data;
	*list = g_slist_prepend (*list, task);
	return FALSE;
}

static GSList *
ftm_get_all_tasks (TaskManager       *manager, 
		   CORBA_Environment *ev)
{
	TaskManagerClass    *klass;
	FileTaskManager     *ftm;
	FileTaskManagerPriv *priv;
	GSList              *list;

	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);
	ftm   = FILE_TASK_MANAGER (manager);
	priv  = ftm->priv;

	list = NULL;
	klass->traverse (manager,
			 task_manager_get_root_task_model (manager),
			 G_PRE_ORDER,
			 traverse_get_all_tasks,
			 &list);
	
	return g_slist_reverse (list);
}

static GNOME_MrProject_Id
ftm_link_tasks (TaskManager                *manager,
		GNOME_MrProject_Dependency *dependency,
		CORBA_Environment          *ev)
{
	FileTaskManager     *ftm;
	FileTaskManagerPriv *priv;

	ftm = FILE_TASK_MANAGER (manager);
	priv = ftm->priv;

	return id_map_insert (priv->dependency_map, dependency);
}

static void
ftm_unlink_tasks (TaskManager              *manager,
		  const GNOME_MrProject_Id  dependency_id,
		  CORBA_Environment        *ev)
{
	FileTaskManager            *ftm;
	FileTaskManagerPriv        *priv;
	GNOME_MrProject_Dependency *dependency;

	ftm = FILE_TASK_MANAGER (manager);
	priv = ftm->priv;

	dependency = id_map_lookup (priv->dependency_map, dependency_id);
	id_map_remove (priv->dependency_map, dependency_id);
}
					     
static void
ftm_remove_dependency (TaskManager        *manager,
		       GNOME_MrProject_Id  dependencyId,
		       CORBA_Environment  *ev)
{
	FileTaskManager     *ftm;
	FileTaskManagerPriv *priv;

	ftm = FILE_TASK_MANAGER (manager);
	priv = ftm->priv;

	id_map_remove (priv->dependency_map, dependencyId);
}

static GNOME_MrProject_Dependency *
ftm_get_dependency (TaskManager              *manager,
		    const GNOME_MrProject_Id  dependencyId,
		    CORBA_Environment        *ev)
{
	FileTaskManager     *ftm;
	FileTaskManagerPriv *priv;

	ftm = FILE_TASK_MANAGER (manager);
	priv = ftm->priv;

	return id_map_lookup (priv->dependency_map, dependencyId);
}

#if 0
static GNOME_MrProject_DependencySeq *
ftm_get_predecessors (TaskManager              *manager,
		      const GNOME_MrProject_Id  taskId,
		      CORBA_Environment        *ev)
{
	FileTaskManager     *ftm;
	FileTaskManagerPriv *priv;
	TaskModel           *task_model;
	GSList              *list;
	
	ftm = FILE_TASK_MANAGER (manager);
	priv = ftm->priv;

	if (taskId == 0) {
		return CORBA_OBJECT_NIL;
	}

	/* FIXME: this method should take a TaskModel directly instead of
	 *  the id.
	 */
	task_model = task_manager_get_task_model (manager, taskId);
	
	/* Get the dependencies. */
	list = task_model_get_predecessors (task_model);
	
	return list;
}

static GNOME_MrProject_DependencySeq *
ftm_get_successors (TaskManager              *manager,
		    const GNOME_MrProject_Id  taskId,
		    CORBA_Environment        *ev)
{
	/* FIXME: implement this. */
	
	g_warning ("Implement this! (ftm_get_successors)");
	return NULL;
}
#endif

static gchar *
ftm_get_note (TaskManager       *manager,
	      const GM_Id        taskId,
	      CORBA_Environment *ev)
{
	FileTaskManager     *ftm;
	FileTaskManagerPriv *priv;
	gchar               *note;
	
	g_return_val_if_fail (manager != NULL, NULL);
	g_return_val_if_fail (IS_FILE_TASK_MANAGER (manager), NULL);
	
	ftm  = FILE_TASK_MANAGER (manager);
	priv = ftm->priv;

	if (taskId == 0) {
		return 0;
	}

	note = (gchar *) id_map_lookup (priv->notes_map, taskId);

	return note;
}

static void
ftm_set_note (TaskManager       *manager,
	      const GM_Id        taskId,
	      const gchar       *note,
	      CORBA_Environment *ev)
{
	FileTaskManager     *ftm;
	FileTaskManagerPriv *priv;
	gchar               *copy;
	gchar               *tmp_note;
	
	g_return_if_fail (manager != NULL);
	g_return_if_fail (IS_FILE_TASK_MANAGER (manager));
	
	ftm  = FILE_TASK_MANAGER (manager);
	priv = ftm->priv;

	if (taskId == 0) {
		return;
	}
	
	tmp_note = (gchar *) id_map_lookup (priv->notes_map, taskId);
	
	if (tmp_note) {
		id_map_remove (priv->notes_map, taskId);
		CORBA_free (tmp_note);
	}

	copy = g_strdup (note);

	id_map_insert_id (priv->notes_map, taskId, copy);
}

static void
ftm_dump_tree (TaskManager        *manager, 
	       CORBA_Environment  *ev)
{
	dump_tree (manager);
}

typedef struct {
	FileTaskManager         *manager;
	TaskModel               *root_task_model;
	gpointer                 user_data;
	TaskManagerTraverseFunc  func;
} TraverseFuncData;

static gboolean
traverse_adapter_func (GNode *node, gpointer data)
{
	TraverseFuncData *traverse_data = (TraverseFuncData *) data;
	TaskModel        *task_model;

	/* Special case the root and never expose it. */
	if (node->data == traverse_data->root_task_model) {
		return FALSE;
	}

	task_model = TASK_MODEL (node->data);
	
	return (*traverse_data->func) (TASK_MANAGER (traverse_data->manager),
				       task_model,
				       traverse_data->user_data);
}

static void
ftm_traverse (TaskManager              *manager, 
	      TaskModel                *task_model,
	      GTraverseType		type,
	      TaskManagerTraverseFunc   func, 
	      gpointer                  user_data)
{
	GNode            *node;
	TraverseFuncData  traverse_data;

	g_return_if_fail (task_model != NULL);
	g_return_if_fail (IS_FILE_TASK_MODEL (task_model));
	
	node = FILE_TASK_MODEL (task_model)->node;
	g_return_if_fail (node != NULL);

	traverse_data.manager = FILE_TASK_MANAGER (manager);
	traverse_data.root_task_model = task_manager_get_root_task_model (manager);
	traverse_data.user_data = user_data;
	traverse_data.func = func;

	g_node_traverse (node,
			 type,
			 G_TRAVERSE_ALL,
			 -1,
			 traverse_adapter_func,
			 &traverse_data);
}

static void
file_task_manager_destroy (GtkObject *object)
{
	FileTaskManager     *manager;
	FileTaskManagerPriv *priv;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_TASK_MANAGER (object));

	manager = FILE_TASK_MANAGER (object);
	priv = manager->priv;

	gtk_object_destroy (GTK_OBJECT (priv->task_map));
	gtk_object_destroy (GTK_OBJECT (priv->dependency_map));

	g_free (priv);
	manager->priv = NULL;

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void
file_task_manager_init (FileTaskManager *manager)
{
	FileTaskManagerPriv  *priv;
	
	priv = g_new0 (FileTaskManagerPriv, 1);
	manager->priv = priv;

	priv->task_map = id_map_new (0);
	priv->dependency_map = id_map_new (0);
	priv->notes_map      = id_map_new (0);
}

TaskManager *
file_task_manager_new (BonoboEventSource *event_source)
{
	FileTaskManager      *manager;
	GNOME_MrProject_Task *root_task;
	TaskModel            *root_task_model;

	manager = gtk_type_new (TYPE_FILE_TASK_MANAGER);

	root_task = task_manager_create_root_task (TASK_MANAGER (manager));
	root_task_model = file_task_model_new (root_task);

	g_assert (id_map_insert_id (manager->priv->task_map, 0, root_task_model));

	task_manager_construct (TASK_MANAGER (manager),
				root_task_model,
				event_source);
	
	return TASK_MANAGER (manager);
}

static void
file_task_manager_class_init (FileTaskManagerClass *klass)
{
	GtkObjectClass *object_class;
	TaskManagerClass *task_manager_class;

	object_class = (GtkObjectClass*) klass;
	task_manager_class = (TaskManagerClass *) klass;

	/* CORBA methods. */
	task_manager_class->insert_task            = ftm_insert_task;
	task_manager_class->update_task            = NULL;
	task_manager_class->remove_tasks           = ftm_remove_tasks;
	task_manager_class->reparent_task          = ftm_reparent_task;
	task_manager_class->reposition_task        = ftm_reposition_task;
	task_manager_class->get_task               = ftm_get_task;
	task_manager_class->get_all_tasks          = ftm_get_all_tasks;
	task_manager_class->link_tasks             = ftm_link_tasks;
	task_manager_class->unlink_tasks           = ftm_unlink_tasks;
	task_manager_class->remove_dependency      = ftm_remove_dependency;
	task_manager_class->get_dependency         = ftm_get_dependency;
	task_manager_class->get_note               = ftm_get_note;
	task_manager_class->set_note               = ftm_set_note;
	task_manager_class->dump_tree              = ftm_dump_tree;

	/* Other methods. */
	task_manager_class->traverse               = ftm_traverse;
	task_manager_class->get_parent_task_model  = ftm_get_parent_task_model;
	task_manager_class->get_task_model         = ftm_get_task_model;
	task_manager_class->get_first_sibling      = ftm_get_first_sibling;
	task_manager_class->get_next_sibling       = ftm_get_next_sibling;
	task_manager_class->get_children           = ftm_get_children;
	task_manager_class->get_task_depth         = ftm_get_task_depth;
	task_manager_class->remove_task_model      = ftm_remove_task_model;
	
	object_class->destroy = file_task_manager_destroy;
}

static gboolean
dump_tree_traverse (TaskManager *manager,
		    TaskModel   *task_model,
		    gpointer     data)
{
	FileTaskModel *ftm;
	gint           i, depth, id;
	gint           count;

	count = (*(gint* )data)++;
	if (count > 100) {
		return TRUE;
	}
	
	ftm = FILE_TASK_MODEL (task_model);

	depth = g_node_depth (ftm->node);
	for (i = 1; i < depth; i++) {
		g_print (".");
	}

	id = ftm->task->taskId;
	g_print ("%d '%s'\n", id, ftm->task->name);

	return FALSE;
}

static void
dump_tree (TaskManager *manager)
{
	gint count;

	count = 0;
	
	g_print ("--------------------------\n");
	task_manager_traverse (manager, 
			       task_manager_get_root_task_model (manager),
			       G_PRE_ORDER,
			       dump_tree_traverse,
			       &count);

}

static TaskModel *
ftm_get_task_model (TaskManager        *manager, 
		    GNOME_MrProject_Id  id)
{
	g_return_val_if_fail (manager != NULL, NULL);
	g_return_val_if_fail (IS_FILE_TASK_MANAGER (manager), NULL);
	g_return_val_if_fail (id >= 0, NULL);
	
	return id_map_lookup (FILE_TASK_MANAGER (manager)->priv->task_map, id);
}

/* This should never happen but IF it does, it helps debugging a lot.
 */
static gboolean
check_for_loop (GNode *parent, GNode *node)
{
	FileTaskModel *parent_ftm, *ftm;
	gboolean       found;

	parent_ftm = parent->data;
	ftm = node->data;

	found = FALSE;
	found = (NULL != g_node_find (parent,
				     G_PRE_ORDER,
				     G_TRAVERSE_ALL,
				     node->data));
	found |= (NULL != g_node_find (node,
				       G_PRE_ORDER,
				       G_TRAVERSE_ALL,
				       parent->data));
	
	if (found) {
		g_print ("Loop, child %d has parent %d as child.\n",
			 ftm->task->taskId, parent_ftm->task->taskId);
	}

	return found;
}

static TaskModel *
ftm_get_parent_task_model (TaskManager *manager, TaskModel *task)
{
	FileTaskModel *ftm;
	GNode         *node;
	
	g_return_val_if_fail (manager != NULL, NULL);
	g_return_val_if_fail (IS_FILE_TASK_MANAGER (manager), NULL);
	g_return_val_if_fail (task != NULL, NULL);
	g_return_val_if_fail (IS_FILE_TASK_MODEL (task), NULL);

	ftm = FILE_TASK_MODEL (task);
	
	node = ftm->node;
	if (node == NULL) {
		return NULL;
	}
	
	if (node->parent == NULL) {
		return NULL;
	}
	
	g_return_val_if_fail (IS_FILE_TASK_MODEL (node->parent->data), NULL);
	return node->parent->data;
}


static TaskModel *
ftm_get_first_sibling (TaskManager *manager, 
		       TaskModel   *task_model)
{
	GNode *node;
	
	node = g_node_first_sibling (FILE_TASK_MODEL (task_model)->node);

	if (node) {
		return node->data;
	} else {
		return NULL;
	}
}

static TaskModel *
ftm_get_next_sibling (TaskManager *manager, 
		      TaskModel   *task_model)
{
	GNode *node;

	node = g_node_next_sibling (FILE_TASK_MODEL (task_model)->node);

	if (node) {
		return node->data;
	} else {
		return NULL;
	}
}

static GSList *
ftm_get_children (TaskManager *manager,
		  TaskModel   *task_model)
{
	GNode  *node;
	GSList *list;

	node = g_node_first_child (FILE_TASK_MODEL (task_model)->node);

	if (!node) {
		return NULL;
	}

	list = NULL;
	while (node) {
		if (node->data) {
			list = g_slist_prepend (list, node->data);
		} else {
			g_warning ("Task node has NULL task.");
		}
		
		node = g_node_next_sibling (node);
	}

	return g_slist_reverse (list);
}

static gint
ftm_get_task_depth (TaskManager *manager,
		    TaskModel   *task_model)
{
	return g_node_depth (FILE_TASK_MODEL (task_model)->node);
}


static void
ftm_remove_task_model (TaskManager *manager,
		       TaskModel   *task_model)
{
	FileTaskManager *ftman;
	FileTaskModel   *ftm;

	ftman = FILE_TASK_MANAGER (manager);
	ftm   = FILE_TASK_MODEL (task_model);
	
	if (!id_map_remove (ftman->priv->task_map, ftm->task->taskId)) {
		g_warning ("Could not remove task in file task manager.");
	}

	gtk_object_destroy (GTK_OBJECT (task_model));
}
