/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001      Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <time.h>
#include <bonobo.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "libmrproject/GNOME_MrProject.h"
#include "task-model.h"


/* GtkObject. */
static void     task_model_init       (TaskModel      *model);
static void     task_model_class_init (TaskModelClass *klass);
static void     task_model_destroy    (GtkObject      *object);

/* Signals. */
enum {
	CHANGED,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

GNOME_CLASS_BOILERPLATE (TaskModel, task_model, GtkObject, gtk_object);

static void
task_model_class_init (TaskModelClass *klass)
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass*) klass;

	object_class->destroy = task_model_destroy;

	/* Add signals. */
	signals[CHANGED] = 
		gtk_signal_new ("changed",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__ENUM,
				GTK_TYPE_NONE, 1, GTK_TYPE_ENUM);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

static void
task_model_destroy (GtkObject *object)
{
	(* parent_class->destroy) (object);
}

static void
task_model_init (TaskModel *task)
{
}

static gint
change_idle_cb (gpointer data)
{
	TaskModel *task;

	task = TASK_MODEL (data);

	if (task->change_mask == TASK_CHANGE_NONE) {
		return FALSE;
	}
	
	gtk_signal_emit (GTK_OBJECT (task), signals[CHANGED], task->change_mask);

	task->change_idle_id = 0;
	task->change_mask = TASK_CHANGE_NONE;

	return FALSE;
}

static void
enqueue_change (TaskModel *task)
{
	if (task->change_idle_id) {
		return;
	}

	task->change_idle_id = gtk_idle_add (change_idle_cb, task);
}

void
task_model_changed (TaskModel *task, TaskChangeMask mask)
{
	g_return_if_fail (task != NULL);
	g_return_if_fail (IS_TASK_MODEL (task));

	if (mask) {
		task->change_mask |= mask;
		enqueue_change (task);
	}
}

void
task_model_add_predecessor (TaskModel                  *task,
			    const GNOME_MrProject_Dependency *dependency)
{
	TaskModelClass *klass;

	g_return_if_fail (task != NULL);
	g_return_if_fail (IS_TASK_MODEL (task));

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);
	klass->add_predecessor (task, dependency);
}

void
task_model_add_successor (TaskModel                  *task,
			  const GNOME_MrProject_Dependency *dependency)
{
	TaskModelClass *klass;

	g_return_if_fail (task != NULL);
	g_return_if_fail (IS_TASK_MODEL (task));

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);
	klass->add_successor (task, dependency);
}

void
task_model_remove_predecessor (TaskModel          *task,
			       GNOME_MrProject_Id  id)
{
	TaskModelClass *klass;

	g_return_if_fail (task != NULL);
	g_return_if_fail (IS_TASK_MODEL (task));

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);
	klass->remove_predecessor (task, id);
}

void
task_model_remove_successor (TaskModel          *task,
			     GNOME_MrProject_Id  id)
{
	TaskModelClass *klass;

	g_return_if_fail (task != NULL);
	g_return_if_fail (IS_TASK_MODEL (task));

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);
	klass->remove_successor (task, id);
}

GNOME_MrProject_DependencySeq *
task_model_get_predecessors_seq (TaskModel *task)
{
	TaskModelClass *klass;
	GSList         *list;

	g_return_val_if_fail (task != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MODEL (task), NULL);

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);

	list = klass->get_predecessors (task);
	
	return corba_util_dependency_seq_from_list (list);
}

GNOME_MrProject_DependencySeq *
task_model_get_successors_seq (TaskModel *task)
{
	TaskModelClass *klass;
	GSList         *list;

	g_return_val_if_fail (task != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MODEL (task), NULL);

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);

	list = klass->get_successors (task);
	
	return corba_util_dependency_seq_from_list (list);
}

GSList *
task_model_get_predecessors (TaskModel *task)
{
	TaskModelClass *klass;

	g_return_val_if_fail (task != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MODEL (task), NULL);

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);
	return klass->get_predecessors (task);
}

GSList *
task_model_get_successors (TaskModel *task)
{
	TaskModelClass *klass;

	g_return_val_if_fail (task != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MODEL (task), NULL);

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);
	return klass->get_successors (task);
}

gboolean
task_model_has_predecessor (TaskModel          *task,
			    GNOME_MrProject_Id  id,
			    GNOME_MrProject_Id *dependency_id)
{
	TaskModelClass *klass;

	g_return_val_if_fail (task != NULL, FALSE);
	g_return_val_if_fail (IS_TASK_MODEL (task), FALSE);

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);
	return klass->has_predecessor (task, id, dependency_id);
}

gboolean
task_model_has_successor (TaskModel          *task,
			  GNOME_MrProject_Id  id,
			  GNOME_MrProject_Id *dependency_id)
{
	TaskModelClass *klass;

	g_return_val_if_fail (task != NULL, FALSE);
	g_return_val_if_fail (IS_TASK_MODEL (task), FALSE);

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);
	return klass->has_successor (task, id, dependency_id);
}

gboolean
task_model_has_ancestor (TaskModel *task, TaskModel *ancestor)
{
	TaskModelClass *klass;

	g_return_val_if_fail (task != NULL, FALSE);
	g_return_val_if_fail (IS_TASK_MODEL (task), FALSE);

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);
	return klass->has_ancestor (task, ancestor);
}

GNOME_MrProject_Task *
task_model_get_task (TaskModel *task)
{
	TaskModelClass *klass;

	g_return_val_if_fail (task != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MODEL (task), NULL);

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);
	return klass->get_task (task);
}

gint
task_model_get_num_children (TaskModel *task)
{
	TaskModelClass *klass;

	g_return_val_if_fail (task != NULL, -1);
	g_return_val_if_fail (IS_TASK_MODEL (task), -1);

	klass = TASK_MODEL_CLASS (GTK_OBJECT (task)->klass);
	return klass->get_num_children (task);
}

