/*************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 sha1thread.cpp  -  background hashing of the shared files
                             -------------------
    begin                : Tue Nov 5 2002
    copyright            : (C) 2002 by Max Zaitsev
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "structures.h"

#include "rcobject.h"
#include "event.h"
#include "messages.h"

#include "sha1thread.h"
#include "sha1.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "xsleep.h"
#include "conversions.h"
#include "common.h"

#if 1
struct Sha1CacheEntry {
	CString sName;
	CString sSha1; // sha1 is not compared as well :-)
	unsigned long nSize;
	unsigned long nModifyTime;
	// parameter which does not get compared
	mutable unsigned long nLastInUseTime;
};

typedef set< Sha1CacheEntry > Sha1Cache;

bool operator<(const Sha1CacheEntry& left,const Sha1CacheEntry& right)
{
	if (left.nSize < right.nSize)
		return true;
	if (left.nSize > right.nSize)
		return false;
	if (left.nModifyTime < right.nModifyTime)
		return true;
	if (left.nModifyTime > right.nModifyTime)
		return false;
	if (left.sName < right.sName)
		return true;
	if (left.sName > right.sName)
		return false;
	return false;
}

bool LoadSha1Cache(const CString& sPath, Sha1Cache& cache)
{
	FILE* pFile = fopen(sPath.c_str(), "r");
	if (!pFile)
		return false;
	// load line by line
	Sha1CacheEntry ce;
	char tmp[1024];
	char tmp1[1024];
	char* t;
	int nNamePos;
	while (!feof(pFile) && !ferror(pFile))
	{
		if (NULL!=fgets(tmp,1024,pFile))
		{
			tmp[1023] = '\0';
			t = StripWhite(tmp);
			if (strlen(t))
			{
				if (*t != '#')
				{
					// parse the line
					if (4 == sscanf(tmp, "%u \t%u \t%u \t%s%n", &ce.nLastInUseTime, &ce.nModifyTime, &ce.nSize, tmp1, &nNamePos))
					{
						ce.sSha1 = StripWhite(tmp1);
						ce.sName = StripWhite(tmp+nNamePos+1);
						cache.insert(ce);
					}
				}
			}
		}
	}
	fclose(pFile);
	return !cache.empty();
}

bool SaveSha1Cache(const CString& sPath, const Sha1Cache& cache) // save should drop too old entries
{
	FILE* pFile = fopen(sPath.c_str(), "w");
	if (!pFile)
		return false;
	fprintf(pFile,  "# this is mutella's hash cache for shared files\n"
					"# please do not edit this file -- incorrect file format\n"
					"# will require full rehashing of all the shared files\n"
					"# if you think you really need to edit it -- just remove\n"
					"# complete lines you dont like\n"
					"#\n");
	unsigned long nOldestUnused = xtime() - 604800 /*one week*/;
	for (Sha1Cache::const_iterator it = cache.begin(); it!=cache.end(); ++it)
	{
		if (it->nLastInUseTime < nOldestUnused)
			continue;
		fprintf(pFile, "%u \t%u \t%u \t%s \t%s\n",
				it->nLastInUseTime,
				it->nModifyTime,
				it->nSize,
				it->sSha1.c_str(),
				it->sName.c_str());
	}
	fclose(pFile);
	return true;
}

class MEventQueueSha1 : public MAsyncEventReceiver {
public:
	MEventQueueSha1() : MAsyncEventReceiver(INT_MAX) {}
protected:
	virtual bool IsOfInterest(MEvent* p){
		ASSERT(p);
		switch (p->GetID())
		{
			case MSG_SHA1_REQUEST:
			case MSG_SHA1_LAST_IN_THE_ROW:
				return true;
			case MSG_MUTELLA_EXIT: 
			case MSG_SHA1_RESET_QUEUE:
				// here we have to mart those as "high-priority" messages
				PriorityEvent(p);
				return true;
		}
		return false;
	}
};

#endif //0

MSha1Thread::MSha1Thread()
{
	m_pEventQueue = new MEventQueueSha1;
	ED().AddReceiver(m_pEventQueue);
}

MSha1Thread::~MSha1Thread()
{
	ASSERT(finished());
	ED().RemoveReceiver(m_pEventQueue); // this does not hurt
	delete m_pEventQueue;
}

#define HASH_BLOCK_SIZE 0x10000

void MSha1Thread::run()
{
	struct      timespec rqtp, rmtp;
	struct      stat sStat;
	//uint8_t     pBuffer[HASH_BLOCK_SIZE]; freeBSD cannot allocate 64K on stack!
	uint8_t*    pBuffer = (uint8_t*) malloc(HASH_BLOCK_SIZE * sizeof(uint8_t));
	uint8_t     pDigest[SHA1HashSize];
	SHA1Context context;

	clock_t TimeBefore;
	double dCalcTime;
	double dSleepTime;
	double dUpdateTime;
	int nRes;
	long nFileSize, nRead, nProcessed;
	char* szSha1;

	Sha1Cache sha1_cache;
	Sha1Cache::iterator itSha1;
	bool      bSha1CacheDirty = false;
	bool      bSha1CacheExists = true;
	bool      bInterrupt = false;
	
	CString   sSha1CachePath = ExpandPath("~/.mutella/share_hash.save").c_str();

	//
	for(;;)
	{
		if (!m_pEventQueue->Poll())
			continue;
		// now we do have an event!
		TSmartPtr<MEvent> spEvent = m_pEventQueue->SafeFront();
		m_pEventQueue->Pop();
		switch (spEvent->GetID())
		{
			case MSG_MUTELLA_EXIT:
				ED().RemoveReceiver(m_pEventQueue);
				TRACE("MSha1Thread is exiting...");
				if (bSha1CacheDirty && sha1_cache.size())
					SaveSha1Cache(sSha1CachePath, sha1_cache);
				// release the memory
				free(pBuffer);
				return;
			case MSG_SHA1_RESET_QUEUE:
				break;
			case MSG_SHA1_LAST_IN_THE_ROW:
				// time to save hash cache and free the memory
				if (bSha1CacheDirty && SaveSha1Cache(sSha1CachePath, sha1_cache))
					bSha1CacheExists = true;
				sha1_cache.clear();
				bSha1CacheDirty = false;
				break;
			case MSG_SHA1_REQUEST:
			if (!m_pEventQueue->PriorityEvents())
			{
				MHashRequest* pE = (MHashRequest*)(MEvent*)spEvent;
				int nFD = ::open(pE->m_sPath.c_str(), O_RDONLY);
				if (nFD >= 0)
				{
					// stat the file and get creation and modification dates
					if ( 0 == fstat(nFD, &sStat) )
					{
						// check if we have hashed it already (sStat.st_mtime)
						if (sha1_cache.empty() && bSha1CacheExists)
						{
							if (!LoadSha1Cache(sSha1CachePath, sha1_cache))
								bSha1CacheExists = false;
							bSha1CacheDirty = false;
						}
						Sha1CacheEntry ce;
						ce.sName = pE->m_sRelPath;
						ce.nSize = sStat.st_size;
						ce.nModifyTime = sStat.st_mtime;
						itSha1 = sha1_cache.find(ce);
						if ( itSha1 == sha1_cache.end() )
						{
							// cary on with hashing
							nFileSize = sStat.st_size;
							SHA1Reset(&context);
							nProcessed = 0;
							dUpdateTime = 0.0;
							bInterrupt = false;
							while( 0 < (nRead = ::read(nFD, pBuffer, HASH_BLOCK_SIZE)) )
							{
								TimeBefore = clock();
								nRes = SHA1Input(&context, pBuffer, nRead);
								if (nRes != shaSuccess) {
									TRACE2("SHA1 error while computing hash for ", pE->m_sPath);
									break;
								}
								nProcessed += nRead;
								// calculate execution times
								dCalcTime = ((double)(clock() - TimeBefore))/CLOCKS_PER_SEC;
								dSleepTime = 3*dCalcTime;
								// sleep for some time to get ~25% cpu consumption
								rqtp.tv_sec  = (time_t) dSleepTime;
								rqtp.tv_nsec = (long) ((dSleepTime - rqtp.tv_sec) * 1000000000L);
								rmtp.tv_sec = rmtp.tv_nsec = 0;
								safe_nanosleep(&rqtp, &rmtp);
								// stop it if we have priority events pending
								if (m_pEventQueue->PriorityEvents())
								{
									bInterrupt = true;
									break;
								}
								dUpdateTime += dCalcTime + dSleepTime;
								if (dUpdateTime > 10.0)
								{
									// each 10 seconds send a progress update message
									POST_EVENT( MHashProgress(
										int( nProcessed*100.0/nFileSize ),
										pE->m_sPath,
										pE->m_nShareIndex
									));
									dUpdateTime = 0.0;
								}
							}
							if (nProcessed == nFileSize)
							{
								SHA1Result(&context, pDigest);
								// convert SHA1 to base32 string
								szSha1 = base32_encode(pDigest, SHA1HashSize, NULL);
								if (szSha1)
								{
									//TRACE4("SHA1: ", pE->m_sPath, " \t", szSha1);
									// post message back to share thread
									POST_EVENT( MHashReady(
										szSha1,
										pE->m_sPath,
										pE->m_nShareIndex
									));
									// add to cache
									ce.sSha1 = szSha1;
									ce.nLastInUseTime = xtime();
									sha1_cache.insert(ce);
									bSha1CacheDirty = true;
									//
									free(szSha1);
								}
							}
							else
							{
								if (!bInterrupt)
								{
									POST_ERROR(ES_IMPORTANT, CString("Error while hashing file `") + pE->m_sPath + "' error=" + DWrdtoStr(errno) + " (" + DWrdtoStr(nProcessed) + "/" + DWrdtoStr(nFileSize) + ")" );
								}
							}
						}
						else
						{
							// we just have it in cache
							POST_EVENT( MHashReady(
								itSha1->sSha1.c_str(),
								pE->m_sPath,
								pE->m_nShareIndex
							));
							// Make note of the cache use
							itSha1->nLastInUseTime = xtime();
							bSha1CacheDirty = true;
						}
					}
					::close(nFD);
				}
			}
			break;
		}
	}
}


