/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_canvas.cpp 
 * @brief Generic canvas main class and entry point.
 * 
 */

#ifdef _WINDOWS
  #include <windows.h>
#else
#endif // ifdef _WINDOWS

#include <assert.h>
#include <gl/gl.h>
#include <gl/glu.h>

#include "myx_gc_figure.h"
#include "myx_gc_layer.h"
#include "myx_gc_model.h"
#include "myx_gc_canvas.h"
#include "myx_gc_gl_helper.h"
#include "myx_gc_font_manager.h"
#include "myx_gc_texture.h"

//----------------------------------------------------------------------------------------------------------------------

// Factory function for a canvas.
GENERIC_CANVAS_API CGenericCanvas* CreateGenericCanvas(GCContext Context)
{
	return new CGenericCanvas(Context);
}

//----------------------------------------------------------------------------------------------------------------------

void CHitResults::AddHit(CFigureInstance* Instance, double Min, double Max)
{
  THitEntry Entry;
  Entry.Instance = Instance;
  Entry.ZMax = Max;
  Entry.ZMin = Min;
  FEntries.push_back(Entry);
}

//----------------------------------------------------------------------------------------------------------------------

int CHitResults::Count(void)
{
  return FEntries.size();
}

//----------------------------------------------------------------------------------------------------------------------

THitEntry* CHitResults::Get(int I)
{
  return &FEntries[I];
}

//----------------------------------------------------------------------------------------------------------------------

void CHitResults::Release(void)
{
  delete this;
}

//----------------- CGenericCanvas -------------------------------------------------------------------------------------

CGenericCanvas::CGenericCanvas(GCContext Context)
{ 
  FIsPicking = false;
	FContext = Context;
  FViewport.Left = 0;
  FViewport.Top = 0;
  FViewport.Width = 100;
  FViewport.Height = 100;
  FUpdateCount = 0;
  FZoomX = 1;
  FZoomY = 1;
  FOffsetX = 0;
  FOffsetY = 0;

  FModel = new CGCModel(this);
  FontManager()->CreateDefaultEntry();
  FSelectionLayer = new CSelectionLayer(this);

  FBackgroundColor = 0;
  glClearColor(0, 0, 0, 1);

  // TODO: make this dynamically configurable (where applicable).
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
  glFrontFace(GL_CW);
  glEnable(GL_AUTO_NORMAL);
  glEnable(GL_CULL_FACE);
  glDisable(GL_DEPTH_TEST);

  glEnable(GL_LINE_SMOOTH);
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_POLYGON_SMOOTH);
  glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
}

//----------------------------------------------------------------------------------------------------------------------

CGenericCanvas::~CGenericCanvas(void)
{
  // Free any dynamically allocated memory. Explicitely increase update count. We don't want any recursive
  // notifcations anymore.
  FUpdateCount++;
  FSelectionLayer->BeginUpdate();
  for (CLayerIterator Iterator = FLayers.begin(); Iterator != FLayers.end(); Iterator++)
  {
    CLayer* Layer = *Iterator;
    delete Layer;
  };
  ClearTemplates();
  delete FModel;
  delete FSelectionLayer;
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::ApplyViewport(void)
{
  glViewport(FViewport.Left, FViewport.Top, FViewport.Width, FViewport.Height);
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::ClearBuffers(void)
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

//----------------------------------------------------------------------------------------------------------------------

/*
 * Looks through the templates and attempts to find one with the given name.
 *
 * @param Name The name of the template to fined.
 * @return Returns the -1 if the template could not be found. Otherwise the result is the associated OpenGL display list.
 */
GLuint CGenericCanvas::FindTemplate(const string& Name)
{
  CTemplateIterator Iterator = FTemplates.find(Name);
  if (Iterator == FTemplates.end())
    return -1;
  else
    return Iterator->second;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Adds the given layer at the end of the layer list. The new layer becomes the top layer in the view then.
 * @param Layer The layer to add.
 */
void CGenericCanvas::AddLayer(CLayer* Layer)
{
  FLayers.push_back(Layer);
  if (FUpdateCount == 0)
    Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Adds a listener to the internal list.
 *
 * @param Listener The listener to add.
 */
void CGenericCanvas::AddListener(CGCListener* Listener)
{
  FListeners.push_back(Listener);
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Reads the layout info stored in the given (XML) file and creates OpenGL display lists. These lists are stored in a 
 * hash map together with their names, so they can quickly be found by the figures.
 *
 * @param Filename The name of the file to load.
 * @return Returns GC_NO_ERROR if everything was ok, otherwise an error code.
 */
TGCError CGenericCanvas::AddTemplatesFromFile(const char* Filename)
{
  TGCError Result = GC_NO_ERROR;

  xmlDocPtr Document;
  xmlNodePtr Root, Current;

  Result = GC_NO_ERROR;

  string CurrentDir = GetCurrentDir();
  Document = xmlParseFile(Utf8ToANSI(string(Filename)).c_str());

  if (Document == NULL)
    return GC_XML_PARSE_ERROR;

  Root = xmlDocGetRootElement(Document);

  if (Root == NULL)
  {
    xmlFreeDoc(Document);
    return GC_XML_EMPTY_DOCUMENT;
  }
  
  if (xmlStrcmp(Root->name, (const xmlChar *) "gc-layout") != 0)
  {
    xmlFreeDoc(Document);
    return GC_XML_INVALID_DOCUMENT;
  }

  // Switch to the directory of the given file. This is necessary to make relative file names working.
  string Path = ExtractFilePath(Filename);
  SetCurrentDir(Path);

  // Parse description elements.
  Current = Root->children;
  while (Current != NULL)
  {
    // Be flexible, ignore any unknown layout entries.
    if ((Current->type == XML_ELEMENT_NODE) && (xmlStrcmp(Current->name, (const xmlChar *) "layout-definition") == 0))
    {
      xmlChar* Attribute = xmlGetProp(Current, (const xmlChar*) "id");
      if (Attribute != NULL)
      {
        // Handle sub entries of the layout definition.
        xmlNodePtr Element = Current->children;
        GLuint DisplayList = 0;
        while (Element != NULL)
        {
          if (Element->type == XML_ELEMENT_NODE)
          {
            if (xmlStrcmp(Element->name, (const xmlChar *) "svg") == 0)
              DisplayList = ReadTemplateDefinition(Element);
              CheckError();

            // Ignore everything else.
            break;
          };
          Element = Element->next;
        };
        string Key((char*) Attribute);
        FTemplates[Key] = DisplayList;
      };
    }
    else
      if ((Current->type == XML_ELEMENT_NODE) && (xmlStrcmp(Current->name, (const xmlChar *) "font") == 0))
      {
        // Adds a new font definition to the font manager.
        ParseFontEntry(Current);
        CheckError();
      }
      else
        if ((Current->type == XML_ELEMENT_NODE) && (xmlStrcmp(Current->name, (const xmlChar *) "texture") == 0))
        {
          // Adds a new texture to the texture list.
          ParseTextureEntry(Current);
          CheckError();
        };
    Current = Current->next;
  }

  SetCurrentDir(CurrentDir);
  xmlFreeDoc(Document);

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given figure instance to the current selection. This call is simply forwared to the selection layer.
 * 
 * @param Instance The instance to be added to the selection. It is taken care that an instance is only added once.
 */
void CGenericCanvas::AddToSelection(CFigureInstance* Instance)
{
  FSelectionLayer->AddToSelection(Instance);
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Increases the update count by 1 to stop any recursive update until (@see EndUpdate()) was called.
 */
void CGenericCanvas::BeginUpdate(void)
{
  FUpdateCount++;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Notifies all registered listeners that a change has occured.
 *
 * @param AObject The reference to an object for which the change event is triggered. Can be anything like a figure, figure
 *                instance, the canvas itself, a layer etc.
 * @param Reason Indicates what change actually happend.
 */
void CGenericCanvas::Change(void* AObject, TGCChangeReason Reason)
{
  for (CListenerIterator Iterator = FListeners.begin(); Iterator != FListeners.end(); Iterator++)
  {
    try
    {
      CGCListener* Listener = *Iterator;
      Listener->OnChange(AObject, Reason);
    }
    catch(...)
    {
      // If there was an exception while executing the method the listener is removed from our list
      // to avoid further harm.
      FListeners.erase(Iterator);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks if there is an OpenGL error registered and triggers the error method if so.
 */
void CGenericCanvas::CheckError(void)
{
  GLenum OGLError = glGetError();
  if (OGLError != GL_NO_ERROR)
  {
    char Buffer[100];
    sprintf(Buffer, "OpenGL error encountered (0x%x).", OGLError);
    Error(Buffer);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all currently selected figure instances from the selection set. This call is simply forwareded to the
 * selection layer.
 */
void CGenericCanvas::ClearSelection(void)
{
  FSelectionLayer->ClearSelection();
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::ClearTemplates(void)
{
  BeginUpdate();
  try
  {
    FTemplates.clear();
    EndUpdate();
  }
  catch(...)
  {
    EndUpdate();
    throw;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a new layer with the given name and returns it to the caller. The new layer is added to this canvas.
 *
 * @param Name The layer identification, encoded in UTF-8.
 */
CLayer* CGenericCanvas::CreateLayer(const char* Name, TGCLayerType Type)
{
  CLayer* Layer = NULL;
  switch (Type) 
  {
    case GC_LAYER_NORMAL:
      {
        Layer = new CLayer(this);
        break;
      };
    case GC_LAYER_GRID:
      {
        Layer = new CGridLayer(this);
        break;
      };
  };

  if (Layer != NULL)
  {
    Layer->FName = Utf8ToUtf16(Name);
    AddLayer(Layer);
  };

  return Layer;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * The counterpart to (@see BeginUpdate). It releases one update lock and invalidates the canvas if the count drops to 0.
 */
void CGenericCanvas::EndUpdate(void)
{
  if (FUpdateCount > 0)
    FUpdateCount--;
  if (FUpdateCount == 0)
    Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Notifies all registered listeners about an error that occured.
 */
void CGenericCanvas::Error(const char* Message)
{
  for (CListenerIterator Iterator = FListeners.begin(); Iterator != FListeners.end(); Iterator++)
  {
    try
    {
      CGCListener* Listener = *Iterator;
      Listener->OnError(Message);
    }
    catch(...)
    {
      // If there was an exception while executing the method the listener is removed from our list
      // to avoid further harm.
      FListeners.erase(Iterator);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the given coordinates and tries to find a figure that was rendered at this position.
 *
 * @param X The horizontal window (viewer) coordinate.
 * @param Y The vertical coordinate. 
 * @return A hit result class is returned regardless of the actual number of hits. It must be freed by the caller.
 * @note: Don't turn the Y value (e.g. on Windows where positive Y points down). All necessary adjustments are done implicitely.
 */
CHitResults* CGenericCanvas::GetHitTestInfoAt(const int X, const int Y)
{
  FIsPicking = true;

  GLint Viewport[4];
  glGetIntegerv(GL_VIEWPORT, Viewport);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPickMatrix(X, Y, 4, 4, Viewport);
  // The projection is set up so that the center of the scene (the origin) is located in the upper left corner of the viewport.
  glOrtho(0, Viewport[2], 0, Viewport[3], -100, 100);

  GLsizei BufferSize = 500;
  GLuint* Buffer = NULL;
  int Hits = 0;
  do
  {
    Buffer = (GLuint*) realloc(Buffer, BufferSize * sizeof(GLuint));
    glSelectBuffer(BufferSize, Buffer);

    glRenderMode(GL_SELECT);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    glInitNames();
    glPushName(0);

    // Render the scene (in select mode nothing is drawn).
    glTranslated(FOffsetX, FOffsetY, 0);
    glScaled(FZoomX, FZoomY, 1);
    for (CLayerIterator Iterator = FLayers.begin(); Iterator != FLayers.end(); Iterator++)
    {
      CLayer* Layer = *Iterator;
      if (Layer->FEnabled)
        Layer->Render();
    };

    Hits = glRenderMode(GL_RENDER);
    BufferSize <<= 1;
  }
  while (Hits < 0);

  CheckError();
  FIsPicking = false;

  CHitResults* Result = new CHitResults();

  GLuint* Run = Buffer;
  for (int I = 0; I < Hits; I++)
  {
    int Count = *Run++;
    // The depth values are normalized so that the largest value corresponds to 1, while the smallest one is 0.
    // To store this as integer a mapping is applied to make 1 <=> 0xFFFFFFFF.
    double ZMin = (double) *Run++ / 0xFFFFFFFF;
    double ZMax = (double) *Run++ / 0xFFFFFFFF;
    CFigureInstance* Instance = (CFigureInstance*) (*Run++);
    Result->AddHit(Instance, ZMin, ZMax);

    // Usually we have only one entry per hit record.
    // That's the way it works when we only use glLoadName when rendering figures.
    // In order to be error tolerant we skip unused entries accordingly.
    Run += Count - 1;
  };

  free(Buffer);
  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

CGCModel* CGenericCanvas::GetModel(void)
{
  return FModel;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the given property, if it is a property of this class.
 *
 * @param Property The property to retrieve.
 * @param Value [out] The value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case Value is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CGenericCanvas::GetProperty(TProperty Property, double& Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        break;
      };
    default:
      {
        // Property is not known here so try the special classes.
        Result = FSelectionLayer->GetProperty(Property, Value);
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the given property, if it is a property of this class.
 *
 * @param Property The property to retrieve.
 * @param Value [out] The value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case Value is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CGenericCanvas::GetProperty(TProperty Property, int& Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        break;
      };
    default:
      {
        // Property is not known here so try the special classes.
        Result = FSelectionLayer->GetProperty(Property, Value);
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines whether the given position corresponds to any of the parts (body, handles) of a selection decoration.
 * This test is quite fast and can be used for cursor feedback and such. This method is simply forwarded to
 * CSelectionLayer::GetSelectionInfo after modelview and projection matrix are constructed with the current settings.
 *
 * @param X The horizontal mouse coordinate.
 * @param Y The vertical mouse coordinate.
 * @return One of the selection info flags.
 */
TGCSelectionInfo CGenericCanvas::GetSelectionInfo(const int X, const int Y)
{
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  // The projection is set up so that the center of the scene (the origin) is located in the upper left corner of the viewport.
  glOrtho(0, FViewport.Width, 0, FViewport.Height, -100, 100);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslated(FOffsetX, FOffsetY, 0);
  glScaled(FZoomX, FZoomY, 1);

  return FSelectionLayer->GetSelectionInfo(X, Y);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Notifies all registered listeners that a change has occured and they need to invalidate their viewers.
 */
void CGenericCanvas::Invalidate(void)
{
  for (CListenerIterator Iterator = FListeners.begin(); Iterator != FListeners.end(); Iterator++)
  {
    try
    {
      CGCListener* Listener = *Iterator;
      Listener->OnInvalidate();
    }
    catch(...)
    {
      // If there was an exception while executing the method the listener is removed from our list
      // to avoid further harm.
      FListeners.erase(Iterator);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the selection layer to recompute the bounds of the selection decoration for the given figure instance.
 *
 * @param Instance The figure instance for which to recompute the selection decoration.
 */
void CGenericCanvas::InvalidateSelectionBounds(CFigureInstance* Instance)
{
  FSelectionLayer->InvalidateBounds(Instance);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the caller whether the canvas is currently being updated.
 */
bool CGenericCanvas::IsUpdating(void)
{
  return FUpdateCount != 0;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Destroys this canvas instance. The release method is used by non-C++ languages access this code in order to avoid
 * releasing memory in an environment where it wasn't allocated.
 */
void CGenericCanvas::Release(void)
{
  delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure instance from the current selection. This call is simply forwared to the selection layer.
 *
 * @param Instance The figure instance to be removed from the selection. Nothing happens if it isn't selected.
 */
void CGenericCanvas::RemoveFromSelection(CFigureInstance* Instance)
{
  FSelectionLayer->RemoveFromSelection(Instance);
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given layer from the internal layer list. The layer itself will not be destroed, just removed.
 *
 * @param Layer The layer to be removed.
 */
void CGenericCanvas::RemoveLayer(CLayer* Layer)
{
  for (CLayerIterator Iterator = FLayers.begin(); Iterator != FLayers.end(); Iterator++)
    if (*Iterator == Layer)
    {
      FLayers.erase(Iterator);
      Layer->FCanvas = NULL;
      Invalidate();
      break;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes a listener from the internal list. If the listener is not in the list then this call has no effect.
 *
 * @param Listener The listener to remove.
 */
void CGenericCanvas::RemoveListener(CGCListener* Listener)
{
  for (CListenerIterator Iterator = FListeners.begin(); Iterator != FListeners.end(); Iterator++)
    if (*Iterator == Listener)
    {
      FListeners.erase(Iterator);
      break;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * This is the main paint routine. It must be called by the viewer holding the reference to this canvas (e.g. when a 
 * window must be redrawn).
 */
void CGenericCanvas::Render(void)
{
  // No display if the canvas is currently being updated.
  if (FUpdateCount == 0)
  {
    ClearBuffers();
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    // The projection is set up so that the center of the scene (the origin) is located in the upper left corner of the viewport.
    glOrtho(0, FViewport.Width, 0, FViewport.Height, -100, 100);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    
    glTranslated(FOffsetX, FOffsetY, 0);
    glScaled(FZoomX, FZoomY, 1);
    for (CLayerIterator Iterator = FLayers.begin(); Iterator != FLayers.end(); Iterator++)
      (*Iterator)->Render();
    FSelectionLayer->Render();

    CheckError();
  };
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::SetBackgroundColor(COLORREF NewColor)
{
  if (NewColor != FBackgroundColor)
  {
    FBackgroundColor = NewColor;
    glClearColor((float) GetRValue(FBackgroundColor) / 255, (float) GetGValue(FBackgroundColor) / 255, 
      (float) GetBValue(FBackgroundColor) / 255, 1);
  }
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the current display offsets.
 *
 * @param X The horizontal display offset.
 * @param Y The vertical display offset.
 */
void CGenericCanvas::SetOffset(double X, double Y)
{
  FOffsetX = X;
  FOffsetY = Y;
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the the value of the given property, if it is a property of this class.
 *
 * @param Property The property to set.
 * @param Value The new value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case the property is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CGenericCanvas::SetProperty(TProperty Property, double Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        break;
      };
    default:
      {
        // Property is not known here so try the special classes.
        Result = FSelectionLayer->SetProperty(Property, Value);
        break;
      };
  };

  if (Result)
    Invalidate();

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the the value of the given property, if it is a property of this class.
 *
 * @param Property The property to set.
 * @param Value The new value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case the property is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CGenericCanvas::SetProperty(TProperty Property, int Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        break;
      };
    default:
      {
        // Property is not known here so try the special classes.
        Result = FSelectionLayer->SetProperty(Property, Value);
        break;
      };
  };

  if (Result)
    Invalidate();

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the viewport of the canvas. The viewport is the area in the viewer where the canvas may draw its content.
 * Hence the viewport is given in pixels in the viewer (window) space.
 *
 * @param NewViewport The new viewport to use.
 */
void CGenericCanvas::SetViewportV(TGCViewport* NewViewport)
{
  FViewport = *NewViewport;
  ApplyViewport();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the viewport of the canvas. The viewport is the area in the viewer where the canvas may draw its content.
 * Hence the viewport is given in pixels in the viewer (window) space.
 *
 * @param Left The left pixel position for the output area.
 * @param Top The upper pixel position for the output area.
 * @param Width The width for the output area.
 * @param Height The height for the output area.
 */
void CGenericCanvas::SetViewport(int Left, int Top, int Width, int Height)
{
  FViewport.Left = Left;
  FViewport.Top = Top;
  FViewport.Width = Width;
  FViewport.Height = Height;
  ApplyViewport();
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::SetZoom(double X, double Y)
{
  FZoomX = X;
  FZoomY = Y;
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::ShowSelection(bool Visible)
{
  FSelectionLayer->SetVisible(Visible);
}

//----------------------------------------------------------------------------------------------------------------------

