/* 
 * Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "../lf_mforms.h"
#include "../lf_textentry.h"

namespace mforms {
namespace gtk {

bool TextEntryImpl::create(::mforms::TextEntry *self, TextEntryType type)
{
  return new TextEntryImpl(self, type);
}

void TextEntryImpl::set_text(::mforms::TextEntry *self, const std::string &text)
{
  TextEntryImpl* cb = self->get_data<TextEntryImpl>();

  if ( cb )
  {
    cb->_entry->set_text(text);
  }
}

void TextEntryImpl::set_max_length(::mforms::TextEntry *self, int len)
{
  TextEntryImpl* cb = self->get_data<TextEntryImpl>();

  if ( cb )
  {
    cb->_entry->set_max_length(len);
  }
}

std::string TextEntryImpl::get_text(::mforms::TextEntry *self)
{
  TextEntryImpl* cb = self->get_data<TextEntryImpl>();
  std::string ret("");
  if ( cb )
  {
    ret = cb->_entry->get_text().raw();
  }
  return ret;
}

void TextEntryImpl::set_read_only(::mforms::TextEntry *self, bool flag)
{
  TextEntryImpl* cb = self->get_data<TextEntryImpl>();
  if (cb && cb->_entry)
    cb->_entry->set_editable(!flag);
}

TextEntryImpl::TextEntryImpl(::mforms::TextEntry *self, TextEntryType type)
  : ViewImpl(self)
{
  _entry= Gtk::manage(new Gtk::Entry());
  switch (type)
  {
  case mforms::NormalEntry:
    break;
  case mforms::PasswordEntry:
    _entry->set_visibility(false);
    break;
  case mforms::SearchEntry:
#if GTK_VERSION_GE(2, 16)
    _entry->set_icon_from_stock(Gtk::Stock::FIND);
    _entry->signal_changed().connect(sigc::mem_fun(this, &TextEntryImpl::text_changed));
    _entry->signal_icon_press().connect(sigc::mem_fun(this, &TextEntryImpl::icon_pressed));
#endif
    break;
  }
  _entry->signal_changed().connect(sigc::mem_fun(self, &::mforms::TextEntry::callback));
  _entry->signal_activate().connect(sigc::bind(sigc::mem_fun(this, &TextEntryImpl::activated), self));
  _entry->signal_key_press_event().connect(sigc::bind(sigc::mem_fun(this, &TextEntryImpl::key_press), self));
  _entry->show();
}

#if GTK_VERSION_GE(2, 16)
void TextEntryImpl::text_changed()
{
  if (_entry->get_text().empty())
    _entry->set_icon_from_pixbuf(Glib::RefPtr<Gdk::Pixbuf>(), Gtk::ENTRY_ICON_SECONDARY);
  else
    _entry->set_icon_from_stock(Gtk::Stock::CLEAR, Gtk::ENTRY_ICON_SECONDARY);
}

void TextEntryImpl::icon_pressed(Gtk::EntryIconPosition pos, const GdkEventButton *ev)
{
  if (pos == Gtk::ENTRY_ICON_SECONDARY)
    _entry->set_text("");
}
#endif

void TextEntryImpl::activated(mforms::TextEntry *self)
{
  self->action(mforms::EntryActivate);
}

bool TextEntryImpl::key_press(GdkEventKey *event, mforms::TextEntry *self)
{
  if (event->keyval == GDK_Up)
  {
    if (event->state & GDK_CONTROL_MASK)
      self->action(mforms::EntryCKeyUp);
    else
      self->action(mforms::EntryKeyUp);
  }
  else if (event->keyval == GDK_Down)
  {
    if (event->state & GDK_CONTROL_MASK)
      self->action(mforms::EntryCKeyDown);
    else
      self->action(mforms::EntryKeyDown);
  }
  return false;
}

void TextEntryImpl::init()
{
  ::mforms::ControlFactory *f = ::mforms::ControlFactory::get_instance();

  f->_textentry_impl.create   = &TextEntryImpl::create;
  f->_textentry_impl.set_text = &TextEntryImpl::set_text;
  f->_textentry_impl.set_max_length = &TextEntryImpl::set_max_length;
  f->_textentry_impl.get_text = &TextEntryImpl::get_text;
  f->_textentry_impl.set_read_only= &TextEntryImpl::set_read_only;
}

};
};
