/*

Copyright (C) 2000, 2001 Christian Kreibich <kreibich@aciri.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <glib.h>

#include <nd.h>
#include <nd_globals.h>
#include <nd_tcb.h>

GHashTable  *ht = NULL;

static gint
nd_tcb_cmp(gconstpointer p1, gconstpointer p2)
{
  ND_TCB *tcb1 = (ND_TCB*)p1;
  ND_TCB *tcb2 = (ND_TCB*)p2;

  if (tcb1->snd_ip < tcb2->snd_ip)
    return 1;
  if (tcb1->snd_ip > tcb2->snd_ip)
    return -1;

  if (tcb1->snd_seq < tcb2->snd_seq)
    return 1;
  if (tcb1->snd_seq > tcb2->snd_seq)
    return -1;

  if (tcb1->rec_ip < tcb2->rec_ip)
    return 1;
  if (tcb1->rec_ip > tcb2->rec_ip)
    return -1;

  if (tcb1->rec_seq < tcb2->rec_seq)
    return 1;
  if (tcb1->rec_seq > tcb2->rec_seq)
    return -1;

  return 0;
}


void
nd_tcb_init(void)
{
  ht = g_hash_table_new(nd_packet_hash, nd_packet_cmp);
}


void
nd_tcb_clear(void)
{
  g_hash_table_destroy(ht);
  ht = g_hash_table_new(nd_packet_hash, nd_packet_cmp);  
}


ND_TCB * 
nd_tcb_lookup(ND_Packet *p)
{
  gpointer result;
  
  result = g_hash_table_lookup(ht, p);

  if (result)
    return ((ND_TCB *)result);
  
  return NULL;
}


void        
nd_tcb_insert(ND_Packet *p)
{
  ND_TCB *tcb;
  
  tcb = (ND_TCB*)malloc(sizeof(ND_TCB));
  bzero(tcb, sizeof(ND_TCB));

  tcb->snd_ip  = nd_packet_ip(p)->ip_src.s_addr;
  tcb->rec_ip  = nd_packet_ip(p)->ip_dst.s_addr;
  tcb->snd_seq = ntohl(nd_packet_tcp(p)->th_seq);
  if (ntohl(nd_packet_tcp(p)->th_ack) != 0)
    tcb->rec_seq = ntohl(nd_packet_tcp(p)->th_ack) - 1; 

  g_hash_table_insert(ht, p, tcb);
}


void        
nd_tcb_remove(ND_Packet *p)
{
  g_hash_table_remove(ht, p);
}


gboolean
nd_tcb_rec_known(ND_TCB *tcb)
{
  if (!tcb)
    return FALSE;

  if (tcb->rec_seq == 0)
    return FALSE;

  return TRUE;
}


void
nd_tcb_set_rec(ND_TCB *tcb, ND_Packet *p)
{
  if (!tcb || !p)
    return;

  if (nd_packet_ip(p)->ip_src.s_addr == tcb->rec_ip)
    tcb->rec_seq = ntohl(nd_packet_tcp(p)->th_seq); 
}


gboolean
nd_tcb_get_rel_seq(ND_TCB *tcb, ND_Packet *p,
		   u_int32_t *seq_start, u_int32_t *seq_end)
{
  int       length;
  u_int32_t seq;

  if (!tcb || !p)
    return (FALSE);

  length = ntohs(nd_packet_ip(p)->ip_len) - (nd_packet_ip(p)->ip_hl * 4) -
    (nd_packet_tcp(p)->th_off * 4);
  
  if (nd_packet_ip(p)->ip_src.s_addr == tcb->snd_ip)
    {
      seq = ntohl(nd_packet_tcp(p)->th_seq);

      if (tcb->snd_seq == seq)
	{
	  *seq_start = tcb->snd_seq;
	  *seq_end   = tcb->snd_seq + length;
	  return (FALSE);
	}
      else
	{
	  *seq_start = seq - tcb->snd_seq;
	  *seq_end   = seq - tcb->snd_seq + length;
	  return (TRUE);
	}
    }
  else if (nd_packet_ip(p)->ip_src.s_addr == tcb->rec_ip)
    {
      seq = ntohl(nd_packet_tcp(p)->th_seq);

      if (tcb->rec_seq == seq)
	{
	  *seq_start = tcb->rec_seq;
	  *seq_end   = tcb->rec_seq + length;
	  return (FALSE);
	}
      else
	{
	  *seq_start = seq - tcb->rec_seq;
	  *seq_end   = seq - tcb->rec_seq + length;
	  return (TRUE);
	}
    }

  printf("Mismatch!\n");
  abort();
}


u_int32_t   
nd_tcb_get_rel_ack(ND_TCB *tcb, ND_Packet *p, gboolean force_rel)
{
  u_int32_t delta;

  if (!tcb || !p)
    return (0);

  if (nd_packet_ip(p)->ip_src.s_addr == tcb->snd_ip)
    {
      delta = (ntohl(nd_packet_tcp(p)->th_ack) - tcb->rec_seq);
      if (delta == 1 && !force_rel)
	return (tcb->rec_seq + 1);

      return delta;
    }
  else if (nd_packet_ip(p)->ip_src.s_addr == tcb->rec_ip)
    {
      delta = (ntohl(nd_packet_tcp(p)->th_ack) - tcb->snd_seq);
      if (delta == 1 && !force_rel)
	return (tcb->snd_seq + 1);

      return delta;
    }
  
  printf("Mismatch!\n");
  abort();

  return (0);
}
