/*

Copyright (C) 2000, 2001 Christian Kreibich <kreibich@aciri.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <signal.h>
#include <regex.h>
#include <pcap.h>

#include <nd.h>
#include <nd_packet.h>
#include <nd_tcb.h>
#include <nd_tcp.h>
#include <nd_tcpdump.h>
#include <nd_prefs.h>
#include <nd_misc.h>
#include <support.h>
#include <interface.h>

#ifdef HOST_BIGENDIAN
static int host_bigendian = 1;
#else
static int host_bigendian = 0;
#endif

#define	SWAP_LONG(y) \
        ((((y)&0xff)<<24) | (((y)&0xff00)<<8) | (((y)&0xff0000)>>8) | (((y)>>24)&0xff))
#define	SWAP_SHORT(y) \
	((((y)&0xff)<<8) | ((u_short)((y)&0xff00)>>8))

static struct pcap_file_header    pfh;
static int                        fd[2];
static pid_t                      tcpdump_pid = -1;
static regex_t                    regex_seq;
static regex_t                    regex_ack;
static regmatch_t                 regex_match[3];




int
nd_tcpdump_init(void)
{
  char tcpdump_path[MAXPATHLEN];

  if (nd_prefs_get_item("tcpdump_path", tcpdump_path))
    {
      if (nd_misc_can_exec(tcpdump_path))
	return (TRUE);
    }
  
  return (FALSE);
}


void
nd_tcpdump_show_error_dialog(const char* err_message)
{
  /* Couldn't launch tcpdump -- show error message. */
  
  GtkWidget *w = create_tcpdump_error_dialog();
  GtkWidget *w2;
  char       tcpdump_path[MAXPATHLEN];
  
  w2 = gtk_object_get_data(GTK_OBJECT(w), "tcpdump_error_label");
  D_ASSERT(w2);
  
  nd_prefs_get_item("tcpdump_path", tcpdump_path);    
  gtk_label_set_text(GTK_LABEL(w2), err_message);
  gtk_widget_show(GTK_WIDGET(w));
}


void    
nd_tcpdump_init_tracefile(char *tracefile)
{
  FILE *f;
  
  /* Check if we can read the tracefile */
  if ( (f = fopen(tracefile, "r")) == NULL)
    {
      fprintf(stderr, "tcpdump init error\n");
      return;
    }

  /* Read trace file header */
  if (fread(&pfh, sizeof(pfh), 1, f) != 1)
    fprintf(stderr, "tcpdump init error\n");

  /* Swap endianness if necessary */
  if (((pfh.magic == 0xd4c3b2a1) && !host_bigendian) ||
      ((pfh.magic == 0xa1b2c3d4) && host_bigendian)  ||
      ((pfh.magic == 0x34cdb2a1) && !host_bigendian) ||
      ((pfh.magic == 0xa1b2cd34) && host_bigendian))
    {
      /* We need to swap the header: */
      pfh.magic = SWAP_LONG(pfh.magic);
      pfh.version_major = SWAP_SHORT(pfh.version_major);
      pfh.version_minor = SWAP_SHORT(pfh.version_minor);
      pfh.thiszone = SWAP_LONG(pfh.thiszone);
      pfh.sigfigs  = SWAP_LONG(pfh.sigfigs);
      pfh.snaplen  = SWAP_LONG(pfh.snaplen);
      pfh.linktype = SWAP_LONG(pfh.linktype);
    }

  pfh.magic = 0xa1b2c3d4;

  fclose(f);

  if (regcomp(&regex_seq, ".* ([0-9]+):([0-9]+).*", REG_EXTENDED) < 0)
    fprintf(stderr, "tcpdump seq regex error\n");

  if (regcomp(&regex_ack, ".* ack ([0-9]+) .*", REG_EXTENDED) < 0)
    fprintf(stderr, "tcpdump ack regex error\n");
}


int
nd_tcpdump_open(void)
{
  char tcpdump_path[MAXPATHLEN];
  char message[MAXPATHLEN];

  if (!nd_prefs_get_item("tcpdump_path", tcpdump_path))
    return (FALSE);

  if (!nd_misc_can_exec(tcpdump_path))
    {
      snprintf(message, MAXPATHLEN,
	       "An error occured when trying to run\n"
	       "%s\n"
	       "Make sure the file exists and\n"
	       "you have correct permissions.",
	       tcpdump_path);
      
      nd_tcpdump_show_error_dialog(message);
      return (FALSE);
    }

  if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd) < 0)
    {
      fprintf(stderr, "Could not create socket pair -- exiting.\n");
      exit(1);
    }

  if ( (tcpdump_pid = fork()) < 0)
    {
      fprintf(stderr, "Fork error -- exiting.\n");
      exit(1);
    }
  else if (tcpdump_pid > 0)
    {
      close(fd[1]);

      /* Write pcap trace file header */
      if (write(fd[0], &pfh, sizeof(pfh)) != sizeof(pfh))
	fprintf(stderr, "Write error in pipe\n");
      
      if (fcntl(fd[0], F_SETFL, O_NONBLOCK) < 0)
	{
	  fprintf(stderr, "Can not fcntl socket -- exiting.\n");
	  exit(-1);
	}
    }
  else
    {
      close(fd[0]);
      if (fd[1] != STDIN_FILENO)
	{
	  if (dup2(fd[1], STDIN_FILENO) != STDIN_FILENO)
	    fprintf(stderr, "stdin pipe error\n");
	}
      if (fd[1] != STDOUT_FILENO)
	{
      	  if (dup2(fd[1], STDOUT_FILENO) != STDOUT_FILENO)
	    fprintf(stderr, "stdout pipe error\n");
	}
      if (execl(tcpdump_path, "tcpdump", "-l", "-t", "-r", "-", NULL) < 0)
	{
	  perror("Ooops");
	  fprintf(stderr, "tcpdump error\n");	
	}
    }

  return (TRUE);
}


void
nd_tcpdump_close(void)
{
  if (tcpdump_pid > 0)
    {
      kill(SIGKILL, tcpdump_pid);
      close(fd[0]);
      tcpdump_pid = -1;
    }
}


void
nd_tcpdump_get_packet_line(ND_Packet *p, char *dest, gboolean find_context)
{
  fd_set fdset;
  struct timeval tv;
  static char  s[LINESIZE], s2[LINESIZE];
  u_char      *data;
  char        *sp, *eol = NULL, *result = NULL;
  int          n = 0, remaining;
  gboolean     success, seq_nos_relative;
  u_int32_t    ack_no, seq_start, seq_end;

  if (!p)
    return;
  
  data = nd_packet_get_data(p, NULL);

  tv.tv_sec = 0;
  tv.tv_usec = 0;
  FD_ZERO(&fdset);
  FD_SET(fd[0], &fdset);

  while (select(fd[0] + 1, &fdset, NULL, NULL, &tv) > 0)
    {
      while ((n = read(fd[0], s, LINESIZE-1)) >= 0)
	;
    }

  FD_ZERO(&fdset);
  FD_SET(fd[0], &fdset);

  if (select(fd[0] + 1, NULL, &fdset, NULL, NULL) > 0)
    {
      /* XXX -- I should use data pointers here and check write()'s
	 return value to make sure everything is written! */

      if (write(fd[0], &(p->ph), sizeof(struct pcap_pkthdr)) != sizeof(struct pcap_pkthdr))
	fprintf(stderr, "Write error in pipe\n");
      if (write(fd[0], data, p->ph.caplen) != (int)p->ph.caplen)
	fprintf(stderr, "Write error in pipe\n");
    }
  else
    {
      fprintf(stderr, "Write error in pipe\n");
    }

  sp = s;
  remaining = LINESIZE-1;

  FD_ZERO(&fdset);
  FD_SET(fd[0], &fdset);
  select(fd[0] + 1, &fdset, NULL, NULL, NULL);

  success = FALSE;
  result = s;

  while (remaining > 0)
    {
      if ((n = read(fd[0], sp, LINESIZE-1)) > 0)
	{
	  eol = strchr(s, '\n');
	  if (eol)
	    {
	      *eol = '\0';
	      success = TRUE;
	      break;
	    }
	  
	  sp += n;
	  remaining -= n;
	}
    }

  if (success && find_context          &&
      (p->transp_prot == ND_PROT_TCP)  &&
      !tcp_syn_set(p))
    {
      ND_TCB *tcb;

      tcb = nd_tcb_lookup(p);
      if (tcb && nd_tcb_rec_known(tcb))
	{
	  if (regexec(&regex_seq, s, 3, regex_match, 0) == 0)
	    {
	      s[regex_match[1].rm_so] = '\0';
	      seq_nos_relative = nd_tcb_get_rel_seq(tcb, p, &seq_start, &seq_end);	      
	      snprintf(s2, LINESIZE, "%s%u:%u%s", s, seq_start, seq_end,
		       s + regex_match[2].rm_eo);
	      
	      result = s2;

	      if (regexec(&regex_ack, s2, 2, regex_match, 0) == 0)
		{
		  ack_no = strtoul(s2 + regex_match[1].rm_so, NULL, 0);
		  s2[regex_match[1].rm_so] = '\0';

		  if (!seq_nos_relative)
		    snprintf(dest, LINESIZE, "%s%u%s", s2, ntohl(nd_packet_tcp(p)->th_ack),
			     s2 + regex_match[1].rm_eo);
		  else
		    snprintf(dest, LINESIZE, "%s%u%s", s2, nd_tcb_get_rel_ack(tcb, p, TRUE),
			     s2 + regex_match[1].rm_eo);

		  return;
		}
	    }	  
	  else if (regexec(&regex_ack, s, 2, regex_match, 0) == 0)
	    {
	      ack_no = strtoul(s + regex_match[1].rm_so, NULL, 0);
	      s[regex_match[1].rm_so] = '\0';
	      snprintf(dest, LINESIZE, "%s%u%s", s, nd_tcb_get_rel_ack(tcb, p, FALSE),
		       s + regex_match[1].rm_eo);
	      return;
	    }
	}
    }

  /* printf("%s\n", result); */

  strncpy(dest, result, LINESIZE);
}
