
/* The following code to compute CRC-32 checksums is based upon
   documentation and code by Michael Barr, found on
   
   http://www.netrino.com/Connecting/2000-01/index.html.
   
   Its code contains the request to include the following note:

   **********************************************************************
   * Copyright (c) 2000 by Michael Barr.  This software is placed into
   * the public domain and may be used for any purpose.  However, this
   * notice must not be changed or removed and no warranty is either
   * expressed or implied by its publication or distribution.
   **********************************************************************
*/

#include <netdude/nd_crc_32.h>

#define CRC_32_POLY              0x04C11DB7
#define CRC_32_INITIAL_REMAINDER 0xFFFFFFFF
#define CRC_32_FINAL_XOR         0xFFFFFFFF

#define WIDTH    (8 * sizeof(guint32))
#define TOPBIT   (1 << (WIDTH - 1))

#define REFLECT_DATA(X)       ((guchar)  reflect((X), 8))
#define REFLECT_REMAINDER(X)  ((guint32) reflect((X), WIDTH))


static guint32  crc_table[256];

unsigned long
reflect(unsigned long data, guchar num_bits)
{
  unsigned long  reflection = 0;
  guchar         bit;

  /*
   * Reflect the data about the center bit.
   */
  for (bit = 0; bit < num_bits; ++bit)
    {
      /*
       * If the LSB bit is set, set the reflection of it.
       */
      if (data & 0x01)
	reflection |= (1 << ((num_bits - 1) - bit));
      
      data = (data >> 1);
    }
  
  return reflection;
}



void
nd_crc_32_init(void)
{
  guint32  remainder;
  guchar   bit;
  int      dividend;

  /*
   * Compute the remainder of each possible dividend.
   */
  for (dividend = 0; dividend < 256; ++dividend)
    {
      /*
       * Start with the dividend followed by zeros.
       */
      remainder = dividend << (WIDTH - 8);
      
      /*
       * Perform modulo-2 division, a bit at a time.
       */
      for (bit = 8; bit > 0; --bit)
        {
	  /*
	   * Try to divide the current data bit.
	   */			
	  if (remainder & TOPBIT)
	    remainder = (remainder << 1) ^ CRC_32_POLY;
	  else
	    remainder = (remainder << 1);
        }
      
      /*
       * Store the result into the table.
       */
      crc_table[dividend] = remainder;
    }  
}



guint32
nd_crc_32_compute(guchar const message[], int num_bytes)
{
  guint32 remainder = CRC_32_INITIAL_REMAINDER;
  guchar  data;
  int     byte;

  /*
   * Divide the message by the polynomial, a byte at a time.
   */
  for (byte = 0; byte < num_bytes; ++byte)
    {
      data = REFLECT_DATA(message[byte]) ^ (remainder >> (WIDTH - 8));
      remainder = crc_table[data] ^ (remainder << 8);
    }

  /*
   * The final remainder is the CRC.
   */
  return (REFLECT_REMAINDER(remainder) ^ CRC_32_FINAL_XOR);
}

