/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include "AsynchJobs_ASN1.h"
#include <PKI_ERR.h>
#include <openssl/asn1t.h>

ASN1_SEQUENCE(MAIL_QUEUE_ENTRY) = {
	ASN1_SIMPLE(MAIL_QUEUE_ENTRY, mail, MAIL_DATAS),
	ASN1_SIMPLE(MAIL_QUEUE_ENTRY, Author, ASN1_UTF8STRING),
	ASN1_SEQUENCE_OF(MAIL_QUEUE_ENTRY, Recipients, ASN1_UTF8STRING),
	ASN1_SIMPLE(MAIL_QUEUE_ENTRY, AdminMail, ASN1_INTEGER),
}ASN1_SEQUENCE_END(MAIL_QUEUE_ENTRY)
MailQueueEntry MailQueueEntry::EmptyInstance;
bool MailQueueEntry::set_adminmail(unsigned long c_adminmail)
{
	m_adminmail = c_adminmail;
	return true;
}

unsigned long MailQueueEntry::get_adminmail() const
{
	return m_adminmail;
}

bool MailQueueEntry::set_author(const mString & c_author)
{
	m_author = c_author;
	return true;
}

const mString & MailQueueEntry::get_author() const
{
	return m_author;
}

mString & MailQueueEntry::get_author()
{
	return m_author;
}

bool MailQueueEntry::set_recipients(const mVector< mString > & c_recipients)
{
	m_recipients = c_recipients;
	return true;
}

const mVector< mString > & MailQueueEntry::get_recipients() const
{
	return m_recipients;
}

mVector< mString > & MailQueueEntry::get_recipients()
{
	return m_recipients;
}

bool MailQueueEntry::set_mail(const MailDatas & c_mail)
{
	m_mail = c_mail;
	return true;
}

const MailDatas & MailQueueEntry::get_mail() const
{
	return m_mail;
}

MailDatas & MailQueueEntry::get_mail()
{
	return m_mail;
}

bool MailQueueEntry::to_PEM(mString & PemDatas) const
{
	MAIL_QUEUE_ENTRY * c_localvar = NULL;
	if(!give_Datas(&c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Private_toPEM("NEWPKI MAIL", get_ASN1_ITEM(), (ASN1_VALUE*)c_localvar, PemDatas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

bool MailQueueEntry::from_PEM(const mString & PemDatas)
{
	MAIL_QUEUE_ENTRY * c_localvar = NULL;
	if(!Private_fromPEM("NEWPKI MAIL", get_ASN1_ITEM(), (ASN1_VALUE**)&c_localvar, PemDatas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!load_Datas(c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

MailQueueEntry::MailQueueEntry():NewPKIObject()
{
	resetAll();
	m_id=0;
}

MailQueueEntry::MailQueueEntry(const MailQueueEntry & other):NewPKIObject()
{
	resetAll();
	m_id=0;
	*this = other;
}

MailQueueEntry::~MailQueueEntry()
{
	Clear();
}

void MailQueueEntry::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void MailQueueEntry::freeAll()
{
}

void MailQueueEntry::resetAll()
{
	m_adminmail = 0;
	m_author = "";
	m_recipients.clear();
	m_mail.Clear();
}

bool MailQueueEntry::load_Datas(const MAIL_QUEUE_ENTRY * Datas)
{
	Clear();
	int i;
	ASN1_UTF8STRING * currrecipients;
	if(Datas->AdminMail)
	{
		m_adminmail = ASN1_INTEGER_GET(Datas->AdminMail);
	}
	if(Datas->Author)
	{
		m_author = Datas->Author;
	}
	if(Datas->Recipients)
	{
		for(i=0; i<SKM_sk_num(ASN1_UTF8STRING, Datas->Recipients); i++)
		{
			currrecipients = SKM_sk_value(ASN1_UTF8STRING, Datas->Recipients, i);
			if(!currrecipients)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_recipients.insert(m_recipients.begin() + i);
			m_recipients[i] = currrecipients;
		}
	}
	if(Datas->mail)
	{
		if(!m_mail.load_Datas(Datas->mail))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool MailQueueEntry::give_Datas(MAIL_QUEUE_ENTRY ** Datas) const
{
	if(!(*Datas) && !(*Datas = (MAIL_QUEUE_ENTRY*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	ASN1_UTF8STRING * currrecipients;
	size_t i;
	if(!(*Datas)->AdminMail && !((*Datas)->AdminMail = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->AdminMail, m_adminmail) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->AdminMail);
		(*Datas)->AdminMail = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->Author && !((*Datas)->Author = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_author.c_ASN1_UTF8STRING(&(*Datas)->Author))
	{
		ASN1_UTF8STRING_free((*Datas)->Author);
		(*Datas)->Author = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->Recipients && !((*Datas)->Recipients = SKM_sk_new_null(ASN1_UTF8STRING)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_recipients.size() ; i++)
	{
		currrecipients = NULL;
		if(!m_recipients[i].c_ASN1_UTF8STRING(&currrecipients))
		{
			ASN1_UTF8STRING_free(currrecipients);
			currrecipients = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(ASN1_UTF8STRING, (*Datas)->Recipients, currrecipients) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currrecipients, ASN1_ITEM_rptr(ASN1_UTF8STRING));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(!(*Datas)->mail && !((*Datas)->mail = (MAIL_DATAS*)ASN1_item_new(ASN1_ITEM_rptr(MAIL_DATAS))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_mail.give_Datas(&(*Datas)->mail))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->mail, ASN1_ITEM_rptr(MAIL_DATAS));
		(*Datas)->mail = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool MailQueueEntry::operator=(const MailQueueEntry & other)
{
	Clear();
	m_adminmail = other.m_adminmail;
	m_author = other.m_author;
	m_recipients = other.m_recipients;
	m_mail = other.m_mail;
	m_id = other.m_id;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * MailQueueEntry::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(MAIL_QUEUE_ENTRY);
}

void MailQueueEntry::set_id(unsigned long c_id)
{
	m_id = c_id;
}

unsigned long MailQueueEntry::get_id() const
{
	return m_id;
}

