/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASYNCHMSGS_ASN1_H
#define ASYNCHMSGS_ASN1_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <ASN1/Asn1Cert.h>

/*!
	This structure represent a request to be sent
*/
typedef struct st_STACK_REQUEST
{
	ASN1_INTEGER * id;				//!< The ID of the request
	CRYPTED_NEWPKI_REQUEST * req;	//!< The request
}STACK_REQUEST;
DECLARE_ASN1_ITEM(STACK_REQUEST)


/*!
	This structure represent a stored request for a responder store
*/
typedef struct st_WORKING_REQUEST
{
	ASN1_UTF8STRING * SenderName;			//!< The name of the sender
	X509_PUBKEY * sender;					//!< The public key of the sender
	ASN1_OCTET_STRING * transactionID;		//!< The transaction ID of the request
	NEWPKI_REQUEST * request;				//!< The request
}WORKING_REQUEST;
DECLARE_ASN1_ITEM(WORKING_REQUEST)


/*!
	This enumeration defines the type of store
*/
typedef enum en_ASYNCHMSGS_TYPE
{
	ASYNCHMSGS_TYPE_NONE=0,				//!< The store does nothing
	ASYNCHMSGS_TYPE_REQUESTER=1,			//!< The store is a requester
	ASYNCHMSGS_TYPE_RESPONDER=2,			//!< The store is a responder
	ASYNCHMSGS_RESPONDER_ASYNCH=4			//!< The responder is asynchronous
}ASYNCHMSGS_TYPE;


/*!
	This enumeration defines the status of a transaction
*/
typedef enum en_TRANSACTION_STATUS
{
	TRANSACTION_STATUS_UNKNOWN=0,		//!< Transaction is unknown
	TRANSACTION_STATUS_WAITING=1,		//!< Transaction is waiting to be processed or sent
	TRANSACTION_STATUS_PROCESSED=2,		//!< Transaction has been processed
	TRANSACTION_STATUS_SENT=3,			//!< Transaction has been sent
	TRANSACTION_STATUS_HOLD=4			//!< Transaction is on hold
}TRANSACTION_STATUS;


/*!
  This class is the representation of WORKING_REQUEST
*/
class WorkingRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	WorkingRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	WorkingRequest(const WorkingRequest & other);

	/*! \brief This is the destructor.
	 */
	~WorkingRequest();


	/*! \brief This function converts a WORKING_REQUEST to a WorkingRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const WORKING_REQUEST * Datas);

	/*! \brief This function converts a WorkingRequest to a WORKING_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(WORKING_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of WORKING_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::SenderName.
	 *  \param c_sendername [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sendername(const mString & c_sendername);

	/*! \brief This function returns ASN1_UTF8STRING::SenderName.
	 *  \return The value of ASN1_UTF8STRING::SenderName.
	 */
	const mString & get_sendername() const;

	/*! \brief This function returns ASN1_UTF8STRING::SenderName.
	 *  \return The value of ASN1_UTF8STRING::SenderName.
	 */
	mString & get_sendername();


	/*! \brief This function sets NEWPKI_REQUEST::request.
	 *  \param c_request [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_request(const NewpkiRequest & c_request);

	/*! \brief This function returns NEWPKI_REQUEST::request.
	 *  \return The value of NEWPKI_REQUEST::request.
	 */
	const NewpkiRequest & get_request() const;

	/*! \brief This function returns NEWPKI_REQUEST::request.
	 *  \return The value of NEWPKI_REQUEST::request.
	 */
	NewpkiRequest & get_request();


	/*! \brief This function sets X509_PUBKEY::sender.
	 *  \param c_sender [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sender(const X509_PUBKEY * c_sender);

	/*! \brief This function returns X509_PUBKEY::sender.
	 *  \return The value of X509_PUBKEY::sender.
	 */
	const X509_PUBKEY * get_sender() const;

	/*! \brief This function returns X509_PUBKEY::sender.
	 *  \return The value of X509_PUBKEY::sender.
	 */
	X509_PUBKEY * get_sender();


	/*! \brief This function sets ASN1_OCTET_STRING::transactionID.
	 *  \param c_transactionid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_transactionid(const Asn1OctetString & c_transactionid);

	/*! \brief This function returns ASN1_OCTET_STRING::transactionID.
	 *  \return The value of ASN1_OCTET_STRING::transactionID.
	 */
	const Asn1OctetString & get_transactionid() const;

	/*! \brief This function returns ASN1_OCTET_STRING::transactionID.
	 *  \return The value of ASN1_OCTET_STRING::transactionID.
	 */
	Asn1OctetString & get_transactionid();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a WorkingRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const WorkingRequest & other);
	/*! \brief This member is an empty instance of WorkingRequest.
	 */
	static WorkingRequest EmptyInstance;

private:
	mString m_sendername;
	NewpkiRequest m_request;
	X509_PUBKEY * m_sender;
	Asn1OctetString m_transactionid;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of STACK_REQUEST
*/
class StackRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	StackRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	StackRequest(const StackRequest & other);

	/*! \brief This is the destructor.
	 */
	~StackRequest();


	/*! \brief This function converts a STACK_REQUEST to a StackRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const STACK_REQUEST * Datas);

	/*! \brief This function converts a StackRequest to a STACK_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(STACK_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of STACK_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::id.
	 *  \return The value of ASN1_INTEGER::id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets CRYPTED_NEWPKI_REQUEST::req.
	 *  \param c_req [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_req(const CryptedNewpkiRequest & c_req);

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::req.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::req.
	 */
	const CryptedNewpkiRequest & get_req() const;

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::req.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::req.
	 */
	CryptedNewpkiRequest & get_req();


	/*! \brief This function copies a StackRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const StackRequest & other);
	/*! \brief This member is an empty instance of StackRequest.
	 */
	static StackRequest EmptyInstance;

private:
	unsigned long m_id;
	CryptedNewpkiRequest m_req;
	void resetAll();
	void freeAll();
protected:
};


#endif // #ifndef ASYNCHMSGS_ASN1_H
