/**
	Cyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// AutoSynchLDAP.h: interface for the AutoSynchLDAP class.
//
//////////////////////////////////////////////////////////////////////

#ifndef AUTOSYNCHLDAP_H
#define AUTOSYNCHLDAP_H

#include "ClientLDAP.h"
#include <newpki_threads.h>
#include <ASN1/Asn1Cert.h>
#include <ASN1/Asn1Conf.h>
#include "EntityLog.h"


/*!
	This class is the declaration of an event handler for the LDAP auto synch
*/
class AutoSynchLDAPEvent
{
public:
	/*! \brief This is the destructor.
	 */
	virtual ~AutoSynchLDAPEvent()
	{
	}

	/*! \brief Event triggered when a new profils has been seeked from the LDAP.
	 *  \param newProfile [IN] The new profile.
	 *  \return true on success, false on failure.
	 */
	virtual bool OnNewProfile(const NewpkiProfile & newProfile)=0;

	/*! \brief Event triggered to get the list of known profiles.
	 *  \param KnownUIDs [OUT] The list of known profiles.
	 *  \return true on success, false on failure.
	 */
	virtual bool GetKnownUIDs(mVector<mString> & KnownUIDs)=0;
};

#ifndef LDAP_SYNCH_WAIT_TIME
#define LDAP_SYNCH_WAIT_TIME 60
#endif


/*!
	This class holds the auto LDAP synchronization optionnaly used by a RA
*/
class AutoSynchLDAP
{
public:
	/*! \brief This is the constructor.
	 */
	AutoSynchLDAP();

	/*! \brief This is the destructor.
	 */
	virtual ~AutoSynchLDAP();

	/*! \brief This function starts the LDAP synchro.
	 *  \param ldap_server [IN] The ldap server's address.
	 *  \param ldap_port [IN] The ldap server's port.
	 *  \param ldap_username [IN] The username.
	 *  \param ldap_password [IN] The password.
	 *  \param ldap_base [IN] The base to search in.
	 *  \param ldap_attr_name [IN] The name of the attribute that hold the uid.
	 *  \param utf8 [IN] The LDAP server uses UTF-8.
	 *  \return true on success, false on failure.
	 */
	bool Start(const mString & ldap_server, unsigned long ldap_port, const mString & ldap_username, const mString & ldap_password, const mString & ldap_base, const mString & ldap_attr_name, unsigned long utf8);

	/*! \brief This function stops the synchro.
	 */
	void Stop();

	/*! \brief This function sets the instance that should receive the events.
	 *  \param event [IN] The event instance.
	 */
	void SetEventHandler(AutoSynchLDAPEvent * event);

	/*! \brief This function sets the logging class.
	 *  \param Logging [IN] The logging class.
	 */
	void SetLogging(EntityLog * Logging);

	/*! \brief This function sets the synchronization options.
	 *  \param RemoveUnknownFields [IN] Should the LDAP attributes not found in DnSpecs be removed?
	 *  \param DnSpecs [IN] The list of DN attributes.
	 *  \param Policies [IN] The policies.
	 *  \param group_id [IN] The group id to delegate the ownership of the new profiles to.
	 *  \param ldap_filters [IN] The LDAP filters.
	 *  \return true on success, false on failure.
	 */
	bool SetOptions(bool RemoveUnknownFields, const mVector<DnSpecs> & DnSpecs, const HashTable_String & Policies, unsigned long group_id, const mString & ldap_filters);

private:
	bool DoSynch();
	mVector<mString> MustNotProcessUIDs;
	bool ProcessResult(const LdapResult & Result);
	static void ThreadSynchLdap(const NewpkiThread * Thread, void * param);
	NewpkiThread hThreadSynchLdap;
	ClientLDAP m_LdapClient;
	AutoSynchLDAPEvent * m_event;
	unsigned long m_group_id;
	bool m_RemoveUnknownFields;
	mString m_ldap_filters;
	mString m_ldap_attr_name;
	mString m_ldap_username;
	mVector<DnSpecs> m_DnSpecs;
	HashTable_String m_Policies;
	CriticalSection ConfAccessLock;
	EntityLog * m_Logging;
};

#endif
