/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/



// Entity_PKI.cpp: implementation of the Entity_PKI class.
//
//////////////////////////////////////////////////////////////////////

#include "Entity_CA.h"
#include "Entity_RA.h"
#include "Entity_KEYSTORE.h"
#include "Entity_EE.h"
#include "Entity_BACKUP.h"
#include "Entity_PUBLICATION.h"
#include "Entity_REPOSITORY.h"

#include <PKI_P7B.h>
#include <PKI_PKCS12.h>
#include <PKI_CRL.h>


#include "Entity_PKI.h"
#include "svintl.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


Entity_PKI::Entity_PKI(ENTITY_CONSTRUCTOR_PARAMETERS):Entity(ENTITY_CONSTRUCTOR_PARAM_PASSTHRU, &pki_conf, NULL, ASYNCHMSGS_TYPE_NONE)
{
	RootCa = NULL;
	UsersCa = NULL;
	EntitiesCa = NULL;
	OcspCa = NULL;
	
	try
	{
		RootCa = new CA_Handler(EntityName, INTERNAL_CA_TYPE_ROOT);
	}
	catch(ExceptionNewPKI e)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		throw ExceptionNewPKI();		
	}
	
	
	try
	{
		UsersCa = new CA_Handler(EntityName, INTERNAL_CA_TYPE_USER);
	}
	catch(ExceptionNewPKI e)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		throw ExceptionNewPKI();		
	}
	
	try
	{
		EntitiesCa = new CA_Handler(EntityName, INTERNAL_CA_TYPE_ENTITY);
	}
	catch(ExceptionNewPKI e)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		throw ExceptionNewPKI();		
	}
	
	try
	{
		OcspCa = new CA_Handler(EntityName, INTERNAL_CA_TYPE_OCSP);
	}
	catch(ExceptionNewPKI e)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		throw ExceptionNewPKI();		
	}
	
	if(!RootCa || !UsersCa || !EntitiesCa || !OcspCa)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		throw ExceptionNewPKI();
	}
	
	RootCa->set_ENGINE(m_Engine);
	UsersCa->set_ENGINE(m_Engine);
	EntitiesCa->set_ENGINE(m_Engine);
	OcspCa->set_ENGINE(m_Engine);
}

Entity_PKI::~Entity_PKI()
{
	m_Jobs.StopAll();
	Free_PkiConf();

	if(RootCa) delete RootCa;
	if(UsersCa) delete UsersCa;
	if(EntitiesCa) delete EntitiesCa;
	if(OcspCa) delete OcspCa;
}

bool Entity_PKI::Create(const EntityCreationDatas & Params, AdminResponseBody & response)
{
	if(!Params)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	if(Params.get_type() != ENTITY_TYPE_PKI)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	mString Cn;
	EntityLinks link;
	unsigned long serial;

	if(!Params.get_pkiCreate())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(!response.get_creEntity().set_type(ENTITY_TYPE_PKI))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}

	//We create the database
	if(!Common_Create(Params.get_pkiCreate().get_entityKey(), NULL))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!CA_Handler::CreateTables(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}

	if(!RootCa->SetDbConn(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	if(!UsersCa->SetDbConn(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	if(!EntitiesCa->SetDbConn(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	if(!OcspCa->SetDbConn(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}


	//Initialize it link struct
	link.get_src().set_name(m_EntityName);
	link.get_src().set_type(ENTITY_TYPE_PKI);
	pki_conf.get_links().push_back(link);

	//Initialize the pki conf
	if(!pki_conf.get_conf().get_body().set_type(ENTITY_TYPE_PKI))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	pki_conf.get_conf().set_name(m_EntityName);
	if(!pki_conf.get_conf().get_body().get_pkiConf().set_type(PKI_CONF_VERSION))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	pki_conf.get_conf().get_body().get_pkiConf().PKI_CONF_PTR.set_offline(0);
	pki_conf.get_conf().set_version(0);

	
	// We now create the internal CAs and
	// the PKI admin certificate

	//Create ROOT CA
	Cn = m_EntityName;
	Cn += " - Internal ROOT CA";
	if(!InternalCA_Create(INTERNAL_CA_TYPE_ROOT, Params.get_pkiCreate().get_rootCa(), Params.get_pkiCreate().get_dn(), Cn.c_str(), Params.get_pkiCreate().get_validity()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	//We create initial CRL
	if(!InternalCA_Revoke(INTERNAL_CA_TYPE_ROOT, 0))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	if(!pki_conf.get_cas().set_rootca(*RootCa->get_CaCert()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		Destroy();
		return false;
	}
	
	//Create USERS CA
	Cn = m_EntityName;
	Cn += " - Internal USERS CA";
	if(!InternalCA_Create(INTERNAL_CA_TYPE_USER, Params.get_pkiCreate().get_usersCa(), Params.get_pkiCreate().get_dn(), Cn.c_str(), Params.get_pkiCreate().get_validity()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	//We create initial CRL
	if(!InternalCA_Revoke(INTERNAL_CA_TYPE_USER, 0))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	if(!pki_conf.get_cas().set_usersca(*UsersCa->get_CaCert()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		Destroy();
		return false;
	}
	
	//Create ENTITIES CA
	Cn = m_EntityName;
	Cn += " - Internal ENTITIES CA";
	if(!InternalCA_Create(INTERNAL_CA_TYPE_ENTITY, Params.get_pkiCreate().get_entitiesCa(), Params.get_pkiCreate().get_dn(), Cn.c_str(), Params.get_pkiCreate().get_validity()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	//We create initial CRL
	if(!InternalCA_Revoke(INTERNAL_CA_TYPE_ENTITY, 0))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	if(!pki_conf.get_cas().set_entitiesca(*EntitiesCa->get_CaCert()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		Destroy();
		return false;
	}

	//Create OCSP CA
	Cn = m_EntityName;
	Cn += " - Internal OCSP CA";
	if(!InternalCA_Create(INTERNAL_CA_TYPE_OCSP, Params.get_pkiCreate().get_ocspCa(), Params.get_pkiCreate().get_dn(), Cn.c_str(), Params.get_pkiCreate().get_validity()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	//We create initial CRL
	if(!InternalCA_Revoke(INTERNAL_CA_TYPE_OCSP, 0))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	if(!pki_conf.get_cas().set_ocspca(*OcspCa->get_CaCert()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		Destroy();
		return false;
	}



	//We now create the PKI Entity certificate
	//Load entity's private key
	
	// m_EntityKey has been loaded/created by Common_Create

	if(!InternalCA_SignForceDn(INTERNAL_CA_TYPE_ENTITY, m_EntityKey.GetPublicKey(), m_EntityName.c_str(), Params.get_pkiCreate().get_email().c_str(), m_EntityCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	if(!pki_conf.get_cas().set_pkicert(m_EntityCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		Destroy();
		return false;
	}

	// We init the entity
	if(!Common_Init(m_EntityCert, EntitySignatureResp::EmptyInstance))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}

	
	//We now create the PKI administrator certificate
	Cn = m_EntityName;
	Cn += " - PKI Administrator";
	((EntityCreationDatas &)Params).get_pkiCreate().get_admincreate().set_cn(Cn);
	((EntityCreationDatas &)Params).get_pkiCreate().get_admincreate().set_pkiadmin(1);

	if(!CreatePkiUser(Params.get_pkiCreate().get_admincreate(), response.get_creEntity().get_pkiCreate(), serial))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}

	Free_PkiConf();
	return true;
}

bool Entity_PKI::Init(const EntitySignatureResp & init_datas)
{
	NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
	return false;
}

bool Entity_PKI::Load()
{
	if(!Common_Load())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!IsFullyInit())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
		
	//We load the CAs
	if(!RootCa->SetDbConn(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!UsersCa->SetDbConn(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!EntitiesCa->SetDbConn(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!OcspCa->SetDbConn(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!RootCa->Load())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!UsersCa->Load())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!EntitiesCa->Load())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!OcspCa->Load())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//Start the conf synch
	if(!m_Jobs.StartConfPush(false))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	// Synch the conf
	ConfAccessLock.LockRead();
	if(pki_conf.get_cryptConfs().get_sig()->digest &&
		pki_conf.get_cryptConfs().get_sig()->algor &&
		pki_conf.get_cryptConfs().get_sig()->algor->algorithm)
	{
		if(!m_Jobs.SetPkiConf(pki_conf.get_cryptConfs()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ConfAccessLock.UnlockRead();
			return false;
		}
	}
	ConfAccessLock.UnlockRead();

	return true;
}

bool Entity_PKI::Upgrade(const char * Version)
{
	PkiConfBeta4 lconf;
	Asn1EncryptSign encrypt;
	size_t i;
	EVP_PKEY * repPubKey;

	if(!Entity::Common_Upgrade(Version))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!CA_Handler::Upgrade(Version, m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	while(strcmp(Version, NEWPKI_VERSION) != 0)
	{
		if(strcmp(Version, "2.0.0-beta4") == 0)
		{
			if(!Load_Conf(&encrypt))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!lconf.from_SignEncrypt(encrypt, m_EntityKey.GetRsaKey(), m_EntityKey.GetRsaKey()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!Upgrade_EntityConf_From_Beta4_To_2_0_0(
					lconf.get_conf(), pki_conf.get_conf()) )
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!pki_conf.set_cas(lconf.get_cas()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!pki_conf.set_links(lconf.get_links()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!pki_conf.set_publications(lconf.get_publications()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!pki_conf.set_repositories(lconf.get_repositories()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			
			pki_conf.get_cryptConfs().get_confs().get_allConfs().clear();
			for(i=0; i<lconf.get_confs().size(); i++)
			{
				pki_conf.get_confs().insert(pki_conf.get_confs().begin() + i);
				
				if(!Upgrade_EntityConf_From_Beta4_To_2_0_0(lconf.get_confs()[i].get_conf(),
						pki_conf.get_confs()[i].get_conf()) )
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				pki_conf.get_confs()[i].get_conf().set_version(lconf.get_confs()[i].get_conf().get_version() + 1);
				if(!pki_conf.get_confs()[i].set_certificate(lconf.get_confs()[i].get_certificate()))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				//Sign and crypt conf
				repPubKey = X509_get_pubkey(pki_conf.get_confs()[i].get_certificate().GetX509());
				if(!repPubKey)
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				
				pki_conf.get_cryptConfs().get_confs().get_allConfs().insert(
						pki_conf.get_cryptConfs().get_confs().get_allConfs().begin() + i);
				
				//Encrypt the body with the entity public key and generate signature with our private key
				if(!pki_conf.get_confs()[i].get_conf().to_SignEncrypt(
					pki_conf.get_cryptConfs().get_confs().get_allConfs()[i].get_crypted(), 
					m_EntityKey.GetRsaKey(), repPubKey, EVP_sha1(), EVP_des_ede3_cbc()))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					EVP_PKEY_free(repPubKey);
					return false;
				}
				if(!pki_conf.get_cryptConfs().get_confs().get_allConfs()[i].set_recipient(
					pki_conf.get_confs()[i].get_certificate().GetX509_PUBKEY()))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					EVP_PKEY_free(repPubKey);
					return false;
				}
				EVP_PKEY_free(repPubKey);
			}
			Version = "2.0.0-rc1";
		}
		NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Upgraded to version %s"), Version);
	}

	pki_conf.get_cryptConfs().get_confs().set_version(pki_conf.get_cryptConfs().get_confs().get_version() + 1);
	if(!WritePersonnalConf())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}


bool Entity_PKI::InternalCA_Create(INTERNAL_CA_TYPE Type, const GenPrivateKey & KeyGen, const X509_NAME * Dn, const char * Cn, int days)
{
	PKI_CERT ResultCert;
	PKI_CSR csr;
	PKI_RSA privKey;
	HashTable_Dn dn;
	HashTable_String Exts;
	Exts.AllowDuplicateNames();
	CA_Handler * currCa;
	InternalCaKey caKey;


	if(!KeyGen || !Dn || !Cn || !days)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(!dn.From_X509_NAME(Dn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(!dn.Add("commonName", Cn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}


	if(!CA_Handler::GEN_PRIVATE_KEY_load(KeyGen, privKey, caKey, m_Engine))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!csr.GenerateCSR(dn, privKey))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}


	if(Type != INTERNAL_CA_TYPE_ROOT)
	{
		//This is a sub CA not ROOT CA
		if(!InternalCA_Sign(INTERNAL_CA_TYPE_ROOT, csr, ResultCert, days, true))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	else
	{
		Exts.Add("basicConstraints", "critical, CA:TRUE");
		Exts.Add("nsComment", "NewPKI Generated CA Certificate");
		Exts.Add("subjectKeyIdentifier", "hash");
		Exts.Add("authorityKeyIdentifier", "keyid:always");
		Exts.Add("nsCertType", "sslCA, emailCA, objCA");
		Exts.Add("keyUsage", "keyCertSign, cRLSign, digitalSignature, nonRepudiation, keyEncipherment");
		
		if(!ResultCert.CreateSelfSigned(csr, &Exts, days, 1))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(!ResultCert.SetPrivateKey(privKey))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	

	//We have the key and the cert we can create the corresponding
	//database entry
	currCa = GetInternalCA(Type);
	if(!currCa)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//We physicaly "create" it
	if(!currCa->Create(ResultCert, caKey))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity_PKI::InternalCA_SignForceDn(INTERNAL_CA_TYPE Type, const X509_PUBKEY * pubKey, const mString & cn, const mString & email, PKI_CERT & ResultCert)
{
	time_t currTime;
	CA_Handler * currCa;
	PKI_CSR csrK;
	PKI_CERT * caCert;
	int days;

	currCa = GetInternalCA(Type);
	if(!currCa)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	caCert = currCa->get_CaCert();
	if(!caCert)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	const PKI_RSA & caKey = caCert->GetPrivateKey();
	if(!caKey)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//We format the CSR DN with the parent DN

	if(!Format_CSR(cn, email, pubKey, csrK, caCert->GetX509()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Calculate maximum life time for the cert
	time_gmt(&currTime);
	days = (caCert->GetEndDate() - currTime) / (3600*24) - 1;

	//We sign the cert
	if(!InternalCA_Sign(Type, csrK, ResultCert, days, false))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity_PKI::Format_CSR(const mString & strCn, const mString & strEmail, const X509_PUBKEY * pubKey, PKI_CSR & csrK, const X509 * parent_cert)
{
	X509_REQ * csr;
	HashTable_Dn dn;
	int pos;
	X509_NAME * tmpDN;
	EVP_PKEY * tmpKey;

	if(!strCn.size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	csr = X509_REQ_new();
	if(!csr)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		return false;
	}

	tmpKey = X509_PUBKEY_get((X509_PUBKEY *)pubKey);
	if(!tmpKey)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		X509_REQ_free(csr);
		return false;
	}
	if(X509_PUBKEY_set(&csr->req_info->pubkey, tmpKey) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		EVP_PKEY_free(tmpKey);
		X509_REQ_free(csr);
		return false;
	}
	EVP_PKEY_free(tmpKey);


	dn.From_X509_NAME(parent_cert->cert_info->subject);

	pos = -1;
	while( (pos = dn.SeekEntryName("commonName", pos)) != HASHTABLE_NOT_FOUND)
	{
		dn.Delete(pos);
	}
	dn.Add("commonName", strCn.c_str());

	pos = -1;
	while( (pos = dn.SeekEntryName("emailAddress", pos)) != HASHTABLE_NOT_FOUND)
	{
		dn.Delete(pos);
	}
	if(strEmail.size())
	{
		dn.Add("emailAddress", strEmail.c_str());
	}


	tmpDN = X509_NAME_new();
	if(!tmpDN)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		X509_REQ_free(csr);
		return false;
	}
	if(!dn.To_X509_NAME(tmpDN))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		X509_NAME_free(tmpDN);
		X509_REQ_free(csr);
		return false;
	}

	if(X509_NAME_set(&csr->req_info->subject, tmpDN) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		X509_NAME_free(tmpDN);
		X509_REQ_free(csr);
		return false;
	}
	X509_NAME_free(tmpDN);

	if(!csrK.SetCSR(csr))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		X509_REQ_free(csr);
		return false;
	}
	X509_REQ_free(csr);

	return true;
}

bool Entity_PKI::InternalCA_Sign(INTERNAL_CA_TYPE Type, const PKI_CSR & csr, PKI_CERT & ResultCert, int Days, bool check_sig)
{

	HashTable_String Exts;
	Exts.AllowDuplicateNames();
	CA_Handler * currCa;

	switch(Type)
	{
		case INTERNAL_CA_TYPE_ROOT:
			Exts.Add("basicConstraints", "critical, CA:TRUE");
			Exts.Add("nsComment", "NewPKI Internal CA Certificate");
			Exts.Add("subjectKeyIdentifier", "hash");
			Exts.Add("authorityKeyIdentifier", "keyid:always");
			Exts.Add("nsCertType", "sslCA, emailCA, objCA");
			Exts.Add("keyUsage", "keyCertSign, cRLSign, digitalSignature, nonRepudiation, keyEncipherment");
			break;
		case INTERNAL_CA_TYPE_USER:
			Exts.Add("basicConstraints", "CA:FALSE");
			Exts.Add("nsComment", "NewPKI Internal CA EndUser Certificate");
			Exts.Add("subjectKeyIdentifier", "hash");
			Exts.Add("authorityKeyIdentifier", "keyid:always");
			Exts.Add("nsCertType", "client");
			Exts.Add("keyUsage", "digitalSignature, nonRepudiation, keyEncipherment");
			Exts.Add("extendedKeyUsage", "clientAuth");
			break;
		case INTERNAL_CA_TYPE_ENTITY:
			Exts.Add("basicConstraints", "CA:FALSE");
			Exts.Add("nsComment", "NewPKI Internal CA Entity Certificate");
			Exts.Add("subjectKeyIdentifier", "hash");
			Exts.Add("authorityKeyIdentifier", "keyid:always");
			Exts.Add("nsCertType", "client, email");
			Exts.Add("keyUsage", "digitalSignature, nonRepudiation, keyEncipherment");
			Exts.Add("extendedKeyUsage", "clientAuth, emailProtection");
			break;
		case INTERNAL_CA_TYPE_OCSP:
			Exts.Add("basicConstraints", "CA:FALSE");
			Exts.Add("nsComment", "NewPKI Internal CA OCSP-Server Certificate");
			Exts.Add("subjectKeyIdentifier", "hash");
			Exts.Add("authorityKeyIdentifier", "keyid:always");
			Exts.Add("extendedKeyUsage", "OCSPSigning");
			break;
		default:
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
			return false;
			break;
			
	}

	//Search corresponding CA_Handler
	currCa = GetInternalCA(Type);
	if(!currCa)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(!currCa->Sign(csr, ResultCert, "", Days, &Exts, false, false, check_sig))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity_PKI::InternalCA_Revoke(INTERNAL_CA_TYPE Type, unsigned long serial)
{
	CA_Handler * currCa;
	PKI_CRL resCrl;
	PKI_CERT revCert;
	CERT_STATE CertState;
	time_t revdate;
	mString uid;

	currCa = GetInternalCA(Type);
	if(!currCa)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(!currCa->Revoke(serial, revCert, CertState, revdate, uid))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!currCa->Generate_CRL(resCrl, 30, 0, NULL, "sha1"))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockWrite();
	//Copy result CRL into destination
	switch(Type)
	{
		case INTERNAL_CA_TYPE_ROOT:
			if(!pki_conf.get_conf().get_crls().set_rootcacrl(resCrl))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}
			break;
		case INTERNAL_CA_TYPE_USER:
			if(!pki_conf.get_conf().get_crls().set_userscacrl(resCrl))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}
			break;
		case INTERNAL_CA_TYPE_ENTITY:
			if(!pki_conf.get_conf().get_crls().set_entitiescacrl(resCrl))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}
			break;
		case INTERNAL_CA_TYPE_OCSP:
			if(!pki_conf.get_conf().get_crls().set_ocspcacrl(resCrl))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}
			break;
		default:
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
			ConfAccessLock.UnlockWrite();
			return false;
			break;
	}
	ConfAccessLock.UnlockWrite();
	
	return true;
}



void Entity_PKI::Free_PkiConf()
{
	ConfAccessLock.LockWrite();
	pki_conf.Clear();
	ConfAccessLock.UnlockWrite();
}

bool Entity_PKI::ParseAdminCommand(AdminResponseBody & response, const PKI_CERT & ClientCert, const AdminRequest & AdminRequest)
{
	//We only accept SSLv3 connection	
	if(!ClientCert)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ERR_to_ADMIN_RESPONSE(response);
		return false;
	}
	
	PARSER_COMMAND_BEGIN(Entity_PKI, response, 0, AdminRequest, ClientCert, GetUserHandle(), true, this, LogsType)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_SEND_ADMIN_MAIL)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_SEND_MAIL)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_CONFIG_PUSH)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_LOGIN,					Entity_PKI::UserLogin,				LOG_MESSAGE_TYPE_USER_LOGIN, (ClientCert)?ClientCert.GetStringName():"", LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_ENUM_LOGS,				Entity_PKI::EnumLogs)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_LOGS_COUNT,			Entity_PKI::GetLogsCount)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_ENUM_ENTITIES,			Entity_PKI::EnumEntities)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SIGN_ENTITY,				Entity_PKI::SignEntity,				LOG_MESSAGE_TYPE_SIGN_ENTITY, body.get_signEntity().get_name(), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_ENUM_USERS,				Entity_PKI::EnumUsers)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_CREATE_PKI_USER,			Entity_PKI::CreateUser,				LOG_MESSAGE_TYPE_USER_ADD, body.get_createPkiUser().get_cn(), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_REVOKE_USER,				Entity_PKI::RevokeUserCert,			LOG_MESSAGE_TYPE_USER_REVOCATION, "", body.get_serial())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_REVOKE_ENTITY,			Entity_PKI::RevokeEntityCert,		LOG_MESSAGE_TYPE_ENTITY_REVOCATION, "", body.get_serial())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SET_OFFLINE,				Entity_PKI::SetOfflineState,		LOG_MESSAGE_TYPE_SET_OFFLINE, "", body.get_offline())
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_OFFLINE,				Entity_PKI::GetOfflineState)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_MY_ACL,				Entity_PKI::GetMyACL)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_GET_CONFIG,				Entity_PKI::GetConfiguration,		LOG_MESSAGE_TYPE_GET_CONFIGURATION, _sv("none"), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_ENTITY_CONF,			Entity_PKI::GetEntityConf)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SET_ENTITY_CONF,			Entity_PKI::SetEntityConf,			LOG_MESSAGE_TYPE_SET_ENTITY_CONF, body.get_entityConf().get_entityCert().GetStringName(), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_REPOSITORIES,		Entity_PKI::GetRepositoriesList)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SET_REPOSITORIES,		Entity_PKI::SetRepositoriesList,	LOG_MESSAGE_TYPE_SET_REPOSITORIES, _sv("none"), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_LOGS_TYPE,			Entity_PKI::GetLogsType)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_LOCK_RESOURCE,			Entity_PKI::UserLockResourse,		LOG_MESSAGE_TYPE_LOCK_RESOURCE, body.get_resourceName(), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_UNLOCK_RESOURCE,			Entity_PKI::UserUnlockResourse,		LOG_MESSAGE_TYPE_UNLOCK_RESOURCE, body.get_resourceName(), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_ENTITY_ACL,			Entity_PKI::GetEntityAcl)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SET_ENTITY_ACL,			Entity_PKI::SetEntityAcl,			LOG_MESSAGE_TYPE_SET_ENTITY_ACL, body.get_entityAcl().get_entityCert()?body.get_entityAcl().get_entityCert().GetStringName():"PKI Entity", LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_ENTITY_MAIL_CONF,	Entity_PKI::GetEntityMailConf)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SET_ENTITY_MAIL_CONF,	Entity_PKI::SetEntityMailConf,		LOG_MESSAGE_TYPE_SET_ENTITY_MAIL_CONF, body.get_entityMailConf().get_entityCert()?body.get_entityMailConf().get_entityCert().GetStringName():"PKI Entity", LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SEND_ADMIN_MAIL,			Entity_PKI::AdminSendMail,			LOG_MESSAGE_TYPE_ADD_ADMIN_MAIL_QUEUE,	_sv("none"), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_ENTITY_AUDITS,		Entity_PKI::GetEntityAudits)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SET_ENTITY_AUDITS,		Entity_PKI::SetEntityAudits,		LOG_MESSAGE_TYPE_SET_ENTITY_AUDITS, body.get_entityAudits().get_entityCert()?body.get_entityAudits().get_entityCert().GetStringName():"PKI Entity", LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_ENTITY_LOGS_TYPE,	Entity_PKI::GetEntityLogsType)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_ENTITIES_LINKS,		Entity_PKI::GetEntitiesLinks)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SET_ENTITIES_LINKS,		Entity_PKI::SetEntitiesLinks,		LOG_MESSAGE_TYPE_SET_ENTITIES_LINKS, _sv("none"), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_GROUPS,				Entity_PKI::GetGroups)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SET_GROUPS,				Entity_PKI::SetGroups,				LOG_MESSAGE_TYPE_SET_GROUPS, _sv("none"), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_ADD_GROUP,				Entity_PKI::AddGroup,				LOG_MESSAGE_TYPE_ADD_GROUP, body.get_groupName(), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_DEL_GROUP,				Entity_PKI::DelGroup,				LOG_MESSAGE_TYPE_DEL_GROUP, "", body.get_groupId())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_REN_GROUP,				Entity_PKI::RenameGroup,			LOG_MESSAGE_TYPE_REN_GROUP, "", body.get_renameGroup().get_id())
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_CHECK_LOGS,				Entity_PKI::CheckLogsIntegrity)
	PARSER_COMMAND_END(Entity_PKI)
}


bool Entity_PKI::LoginUser(UserHandle & hUser, int & UserType)
{
	if(!AclValidator.CanUserPerform(hUser.GetUserCert(), ACL_TYPE_AUTHENTICATE_ON_ENTITY))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	UserType = USER_TYPE_PKI;
	return true;
}

void Entity_PKI::LogoutUser(const UserHandle & hUser)
{
}


bool Entity_PKI::SignEntity(COMMAND_PARAMETERS)
{
	EntitySignatureResp sign_entity;
	PKI_CERT resCert;
	const X509_PUBKEY * pubKey;
	RepEntryInfo repository;
	PubEntryInfo publication;
	EntityConfCrypted ExportedConf;
	unsigned long entity_cert_serial=0;
	unsigned long ocsp_cert_serial=0;
	X509 * EntityCert = NULL;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_ENTITIES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!body.get_signEntity().get_name().size() || !body.get_signEntity().get_email().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	ConfAccessLock.LockWrite();
	//Doing basic verification
	switch(body.get_signEntity().get_body().get_type())
	{
		case ENTITY_TYPE_KEY_STORE:
		case ENTITY_TYPE_EE:
		case ENTITY_TYPE_RA:
		case ENTITY_TYPE_CA:
		case ENTITY_TYPE_BACKUP:
			//We need to make sur there is a repository available
			if(!pki_conf.get_repositories().size())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_NO_REPOSITORY);
				ConfAccessLock.UnlockWrite();
				return false;
			}
			pubKey = body.get_signEntity().get_body().get_entitypubkey();
			break;
		case ENTITY_TYPE_PUBLICATION:
			//We need to make sur there is a repository available
			if(!pki_conf.get_repositories().size())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_NO_REPOSITORY);
				ConfAccessLock.UnlockWrite();
				return false;
			}
			pubKey = body.get_signEntity().get_body().get_signPub().get_entitypubkey();
			break;
		case ENTITY_TYPE_REPOSITORY:
			pubKey = body.get_signEntity().get_body().get_signRep().get_entitypubkey();
			break;
		default:
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
			ConfAccessLock.UnlockWrite();
			return false;
			break;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_SIGN_ENTITY))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	//Generate entity cert
	if(!InternalCA_SignForceDn(INTERNAL_CA_TYPE_ENTITY, pubKey, body.get_signEntity().get_name(), body.get_signEntity().get_email(), resCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}
	entity_cert_serial = resCert.GetSerial();

	if(!response.get_signEntity().get_body().set_type(body.get_signEntity().get_body().get_type()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		NewPKI_LockErr();
		Private_RevokeEntityCert(entity_cert_serial); 
		NewPKI_UnlockErr();
		ConfAccessLock.UnlockWrite();
		return false;
	}

	//Generate the real signature response
	switch(body.get_signEntity().get_body().get_type())
	{
		case ENTITY_TYPE_RA:
		case ENTITY_TYPE_CA:
		case ENTITY_TYPE_REPOSITORY:
		case ENTITY_TYPE_KEY_STORE:
		case ENTITY_TYPE_EE:
		case ENTITY_TYPE_BACKUP:
			if(!response.get_signEntity().get_body().set_entitycert(resCert))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				NewPKI_LockErr();
				Private_RevokeEntityCert(entity_cert_serial); 
				NewPKI_UnlockErr();
				ConfAccessLock.UnlockWrite();
				return false;
			}
			EntityCert = response.get_signEntity().get_body().get_entitycert().GetX509();
			break;

		case ENTITY_TYPE_PUBLICATION:
			if(!response.get_signEntity().get_body().get_signPub().set_entitycert(resCert))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				NewPKI_LockErr();
				Private_RevokeEntityCert(entity_cert_serial); 
				NewPKI_UnlockErr();
				ConfAccessLock.UnlockWrite();
				return false;
			}
			EntityCert = response.get_signEntity().get_body().get_signPub().get_entitycert().GetX509();

			//Now we generate the OCSP cert
			if(!InternalCA_SignForceDn(INTERNAL_CA_TYPE_OCSP, body.get_signEntity().get_body().get_signPub().get_ocsppubkey(), body.get_signEntity().get_name(), body.get_signEntity().get_email(), resCert))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				NewPKI_LockErr();
				Private_RevokeEntityCert(entity_cert_serial); 
				NewPKI_UnlockErr();
				ConfAccessLock.UnlockWrite();
				return false;
			}
			ocsp_cert_serial = resCert.GetSerial();

			if(!response.get_signEntity().get_body().get_signPub().set_ocspcert(resCert))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				NewPKI_LockErr();
				Private_RevokeEntityCert(entity_cert_serial); 
				Private_RevokeEntityCert(ocsp_cert_serial); 
				NewPKI_UnlockErr();
				ConfAccessLock.UnlockWrite();
				return false;
			}
			break;
	}


	//If we just created a publication, we add it to the list
	if(body.get_signEntity().get_body().get_type() == ENTITY_TYPE_PUBLICATION)
	{
		publication.set_name(body.get_signEntity().get_name());

		if(!publication.set_pubocsp(response.get_signEntity().get_body().get_signPub().get_ocspcert()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			NewPKI_LockErr();
			Private_RevokeEntityCert(entity_cert_serial); 
			Private_RevokeEntityCert(ocsp_cert_serial); 
			NewPKI_UnlockErr();
			ConfAccessLock.UnlockWrite();
			return false;
		}

		if(!publication.set_pubssl(response.get_signEntity().get_body().get_signPub().get_entitycert()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			NewPKI_LockErr();
			Private_RevokeEntityCert(entity_cert_serial); 
			Private_RevokeEntityCert(ocsp_cert_serial); 
			NewPKI_UnlockErr();
			ConfAccessLock.UnlockWrite();
			return false;
		}

		pki_conf.get_publications().get_list().push_back(publication);
		publication.Clear();
	}
	else if(body.get_signEntity().get_body().get_type() == ENTITY_TYPE_REPOSITORY)
	{
		//If we just created a repository, we add it to the list
		repository.set_name(body.get_signEntity().get_name());
		repository.set_address(body.get_signEntity().get_body().get_signRep().get_address());
		repository.set_port(body.get_signEntity().get_body().get_signRep().get_port());
		
		if(!repository.set_repositoryssl(response.get_signEntity().get_body().get_entitycert()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			NewPKI_LockErr();
			Private_RevokeEntityCert(entity_cert_serial); 
			NewPKI_UnlockErr();
			ConfAccessLock.UnlockWrite();
			return false;
		}

		pki_conf.get_repositories().push_back(repository);
		repository.Clear();
	}
	
	//Creating a default conf for the new entity
	if(!Add_ENTITY_CONF(body.get_signEntity().get_name(), EntityCert, body.get_signEntity().get_body().get_type()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		NewPKI_LockErr();
		Private_RevokeEntityCert(entity_cert_serial); 
		if(ocsp_cert_serial)
			Private_RevokeEntityCert(ocsp_cert_serial); 
		NewPKI_UnlockErr();
		ConfAccessLock.UnlockWrite();
		return false;
	}

	//writing conf to Database
	if(!WritePersonnalConf())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		NewPKI_LockErr();
		Private_RevokeEntityCert(entity_cert_serial); 
		if(ocsp_cert_serial)
			Private_RevokeEntityCert(ocsp_cert_serial); 
		NewPKI_UnlockErr();
		ConfAccessLock.UnlockWrite();
		return false;
	}

	//We copy the exported conf to the response
	if(!EXPORTED_PKI_CONF_get_conf(EntityCert, response.get_signEntity().get_conf()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		NewPKI_LockErr();
		Private_RevokeEntityCert(entity_cert_serial); 
		if(ocsp_cert_serial)
			Private_RevokeEntityCert(ocsp_cert_serial); 
		NewPKI_UnlockErr();
		ConfAccessLock.UnlockWrite();
		return false;
	}

	//We copy the internal ca certs to the signature response
	if( !(response.get_signEntity().get_cas() = pki_conf.get_cas()) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		NewPKI_LockErr();
		Private_RevokeEntityCert(entity_cert_serial); 
		if(ocsp_cert_serial)
			Private_RevokeEntityCert(ocsp_cert_serial); 
		NewPKI_UnlockErr();
		ConfAccessLock.UnlockWrite();
		return false;
	}

	ConfAccessLock.UnlockWrite();
	return true;
}


bool Entity_PKI::EnumEntities(COMMAND_PARAMETERS)
{
	size_t i, v_index;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_ENUM_ENTITIES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	ConfAccessLock.LockRead();

	v_index = 0;
	for(i=0; i<pki_conf.get_confs().size(); i++)
	{
		response.get_entities().insert(response.get_entities().begin() + v_index);
		
		response.get_entities()[v_index].set_name(pki_conf.get_confs()[i].get_conf().get_name());
		response.get_entities()[v_index].set_type(pki_conf.get_confs()[i].get_conf().get_body().get_type());
		if(!response.get_entities()[v_index].set_certificate(pki_conf.get_confs()[i].get_certificate()))
			continue;
		v_index++;
	}
	ConfAccessLock.UnlockRead();
	return true;
}


bool Entity_PKI::EnumUsers(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	size_t i, v_index;
	CA_Handler * currCa;
	mVector<InternalCaCert> certs;
	bool IsAdmin;
	bool UserIsAdmin;


	IsAdmin = AclValidator.IsPkiAdministrator(UserCert);

	currCa = GetInternalCA(INTERNAL_CA_TYPE_USER);
	if(!currCa)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_ENUM_USERS_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!currCa->get_Certs(certs, CERT_STATE_VALID))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	v_index = 0;
	for(i=0; i<certs.size() ; i++)
	{
		if(certs[i].get_state() != CERT_STATE_VALID)
			continue;

		UserIsAdmin = AclValidator.IsPkiAdministrator(certs[i].get_serial());
		// Only a PKI Admin user can see other PKI Admin users
		if(!IsAdmin && UserIsAdmin)
			continue;

		response.get_usersCert().insert(response.get_usersCert().begin() + v_index);
		if(!response.get_usersCert()[v_index].set_userCert(certs[i].get_cert()))
			continue;
		response.get_usersCert()[v_index].set_admin(UserIsAdmin?1:0);
		v_index++;
	}
	return true;
}

bool Entity_PKI::CreateUser(COMMAND_PARAMETERS)
{
	unsigned long serial;


	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_USERS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// Only a PKI Admin can create another PKI Admin
	if(!AclValidator.IsPkiAdministrator(UserCert) && body.get_createPkiUser().get_pkiadmin())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}


	if(!response.set_type(ADMIN_RESP_TYPE_CREATE_PKI_USER))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!CreatePkiUser(body.get_createPkiUser(), response.get_createPkiUser(), serial))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity_PKI::RevokeUserCert(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_USERS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// Only a PKI admin can revoke another PKI Admin
	if(!AclValidator.IsPkiAdministrator(UserCert.GetSerial()) && 
		AclValidator.IsPkiAdministrator(body.get_serial()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	// We need to have at least one PKI Admin
	if(AclValidator.IsPkiAdministrator(body.get_serial())  && 
		pki_conf.get_conf().get_acls().get_adminserials().size() <= 1)
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	ConfAccessLock.UnlockRead();


	if(!RevokePkiUser(body.get_serial()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	return true;
}

bool Entity_PKI::Private_RevokeEntityCert(unsigned long serial)
{
	size_t i;
	size_t j;
	size_t k;
	bool FoundRep;
	int What2Update = ENTITY_CONF_UPDATE_LINKS;

	// Search in publications list for this entity cert
	// if we find it then we revoke the OCSP cert as well
	// and we remove it from the publications list
	for(i=0; i<pki_conf.get_publications().get_list().size(); i++)
	{
		//Does the publication cert have our serial ?
		if(pki_conf.get_publications().get_list()[i].get_pubssl().GetSerial() == serial)
		{
			//We have to revoke the OCSP cert and remove it from the list
			if(!InternalCA_Revoke(INTERNAL_CA_TYPE_OCSP, pki_conf.get_publications().get_list()[i].get_pubocsp().GetSerial()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			pki_conf.get_publications().get_list().erase(pki_conf.get_publications().get_list().begin() + i);
			break;
		}
	}
	// Did we find it as in the publication list ?
	if(i == pki_conf.get_publications().get_list().size())
	{	
		// Search in repositories list for this entity cert
		// if we find it then we remove it from the repositories list
		for(i=0; i<pki_conf.get_repositories().size(); i++)
		{
			//Does the repository cert have our serial ?
			if(pki_conf.get_repositories()[i].get_repositoryssl().GetSerial() == serial)
			{
				// We check that we can revoke it, meaning, is it 
				// the only repository for some of the others entities ?
				for(j=0; j<pki_conf.get_links().size(); j++)
				{
					// The PKI Entity can ignore all reps
					if(pki_conf.get_links()[j].get_src().get_type() == ENTITY_TYPE_PKI ||
						pki_conf.get_links()[j].get_src().get_type() == ENTITY_TYPE_REPOSITORY)
						continue;

					// Check to see if the respository to revoke 
					// is the only one of the current conf !
					FoundRep = false;
					for(k=0; k < pki_conf.get_links()[j].get_dsts().size(); k++)
					{
						// We only want repositories
						if(pki_conf.get_links()[j].get_dsts()[k].get_type() != ENTITY_TYPE_REPOSITORY)
							continue;

						//Is it the seeked one ?
						if(pki_conf.get_links()[j].get_dsts()[k].get_name() == pki_conf.get_repositories()[i].get_name())
						{
							FoundRep = true;
							break;
						}
					}

					//Did we find another repository ?
					if(FoundRep && k == pki_conf.get_links()[j].get_dsts().size())
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_LAST_ENTITY_REPOSITORY);
						ERR_add_error_data(1, pki_conf.get_links()[j].get_src().get_name().c_str());
						return false;
					}
				}

				pki_conf.get_repositories().erase(pki_conf.get_repositories().begin() + i);
				break;
			}
		}
	}

	//We need to delete the conf for this entity
	for(i=0; i<pki_conf.get_confs().size(); i++)
	{
		//Does the conf cert have our serial ?
		if(pki_conf.get_confs()[i].get_certificate().GetSerial() == serial)
		{
			// We cannot revoke a RA when an EE is attached to it !
			if(pki_conf.get_confs()[i].get_conf().get_body().get_type() == ENTITY_TYPE_RA &&
				pki_conf.get_confs()[i].get_conf().get_ees().get_list().size())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
				ERR_add_error_data(1, _sv("You cannot revoke a RA while an EE is attached to it"));
				return false;
			}

			//We need to delete the crypted conf for this entity
			for(j=0; j<pki_conf.get_cryptConfs().get_confs().get_allConfs().size(); j++)
			{
				//Is it the seeked one ?
				if(pki_conf.get_confs()[i].get_certificate() ==
					pki_conf.get_cryptConfs().get_confs().get_allConfs()[j].get_recipient())
				{
					pki_conf.get_cryptConfs().get_confs().get_allConfs().erase(pki_conf.get_cryptConfs().get_confs().get_allConfs().begin() + j);
					break;
				}
			}

			//We remove its entry in the links table
			for(j=0; j < pki_conf.get_links().size(); j++)
			{
				if(pki_conf.get_links()[j].get_src().get_name() ==
						pki_conf.get_confs()[i].get_conf().get_name() )
				{
					pki_conf.get_links().erase(pki_conf.get_links().begin() + j);
					break;
				}
			}

			pki_conf.get_confs().erase(pki_conf.get_confs().begin() + i);
			break;
		}
	}

	//We revoke the entity cert
	if(!InternalCA_Revoke(INTERNAL_CA_TYPE_ENTITY, serial))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	
	//Yell to other entities the new CRL
	What2Update |= ENTITY_CONF_UPDATE_CRL;
	if(!UpdateEntityConfs(What2Update))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	// We flush the new conf
	if(!WritePersonnalConf())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity_PKI::RevokeEntityCert(COMMAND_PARAMETERS)
{

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_ENTITIES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockWrite();

	if(!Private_RevokeEntityCert(body.get_serial()))
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.UnlockWrite();

	return true;
}

CA_Handler * Entity_PKI::GetInternalCA(INTERNAL_CA_TYPE Type)
{
	switch(Type)
	{
		case INTERNAL_CA_TYPE_ROOT:
			return RootCa;
			break;
		case INTERNAL_CA_TYPE_USER:
			return UsersCa;
			break;
		case INTERNAL_CA_TYPE_ENTITY:
			return EntitiesCa;
			break;
		case INTERNAL_CA_TYPE_OCSP:
			return OcspCa;
			break;
		default:
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
			return NULL;
			break;
	}
}

bool Entity_PKI::GetOfflineState(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MODIFY_PKI_STATUS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_STATUS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	response.set_status(pki_conf.get_conf().get_body().get_pkiConf().PKI_CONF_PTR.get_offline());

	return true;
}

bool Entity_PKI::SetOfflineState(COMMAND_PARAMETERS)
{

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MODIFY_PKI_STATUS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockWrite();
	pki_conf.get_conf().get_body().get_pkiConf().PKI_CONF_PTR.set_offline(body.get_offline());
	if(!WritePersonnalConf())
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ConfAccessLock.UnlockWrite();

	return true;
}

void Entity_PKI::GetACL_List(mVector<unsigned long> & acl_list)
{
	int i;
	static ACL_TYPE list_acls[] =
	{
		ACL_TYPE_MANAGE_USERS,
		ACL_TYPE_MANAGE_ENTITIES,
		ACL_TYPE_VIEW_LOGS,
		ACL_TYPE_MANAGE_ACLS,
		ACL_TYPE_MODIFY_PKI_STATUS, 
		ACL_TYPE_SEND_ADMIN_MAIL,
		(ACL_TYPE)0
	};
	for(i=0; list_acls[i]; i++)
	{
		acl_list.push_back(list_acls[i]);
	}
}

bool Entity_PKI::GetConfiguration(COMMAND_PARAMETERS)
{
	if(!response.set_type(ADMIN_RESP_TYPE_CONF))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	if(!response.set_conf(pki_conf.get_cryptConfs()))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ConfAccessLock.UnlockRead();
	
	return true;
}

bool Entity_PKI::GetEntityConf(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_ENTITY_CONF))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	
	ConfAccessLock.LockRead();

	EntityConf & conf = GetEntityConfiguration(body.get_entityCert());
	if(!conf)
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN_ENTITY);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_ENTITIES))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!ResourceOwnerIsMe(body.get_entityCert().GetStringName()))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_entityConf(conf.get_body()))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	ConfAccessLock.UnlockRead();

	return true;
}

bool Entity_PKI::SetEntityConf(COMMAND_PARAMETERS)
{
	bool ret;
	EntityConfBody tmpConf;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_ENTITIES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!ResourceOwnerIsMe(body.get_entityConf().get_entityCert().GetStringName()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	size_t i;
	size_t j;
	ConfAccessLock.LockWrite();
	for(i=0; i < pki_conf.get_confs().size(); i++)
	{
		if(pki_conf.get_confs()[i].get_certificate() == body.get_entityConf().get_entityCert())
		{
			if(! (tmpConf = body.get_entityConf().get_conf()) )
			{
				ConfAccessLock.UnlockWrite();
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}

			//Verify entity conf
			switch(tmpConf.get_type())
			{
				case ENTITY_TYPE_CA:
					ret = CheckCaConf(tmpConf.get_caConf());
					break;
				case ENTITY_TYPE_RA:
					ret = CheckRaConf(tmpConf.get_raConf());
					break;
				case ENTITY_TYPE_REPOSITORY:
					ret = CheckRepositoryConf(tmpConf.get_repConf());
					break;
				default:
					ret = true;
					break;
			}
			
			if(!ret)
			{
				ConfAccessLock.UnlockWrite();
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}

			if(!pki_conf.get_confs()[i].get_conf().set_body(tmpConf))
			{
				ConfAccessLock.UnlockWrite();
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			//We update the crypted conf
			if(!UpdateEntityConfs(0, pki_conf.get_confs()[i].get_conf(), true))
			{
				ConfAccessLock.UnlockWrite();
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}

			// If the RA is linked to an EE, we must copy the RA's DN Specs and
			// Policies to the EE
			if(tmpConf.get_type() == ENTITY_TYPE_RA &&
				pki_conf.get_confs()[i].get_conf().get_ees().get_list().size())
			{
				// We get the EE's conf
				for(j=0; j < pki_conf.get_confs().size(); j++)
				{
					if(pki_conf.get_confs()[j].get_certificate() == 
						pki_conf.get_confs()[i].get_conf().get_ees().get_list()[0].get_eessl())
					{

						if(!SynchronizeEeOnRa(pki_conf.get_confs()[j].get_conf(),
											pki_conf.get_confs()[i].get_conf()))
						{
							ConfAccessLock.UnlockWrite();
							NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
							return false;
						}
						//We update the crypted conf
						if(!UpdateEntityConfs(0, pki_conf.get_confs()[j].get_conf(), true))
						{
							ConfAccessLock.UnlockWrite();
							NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
							return false;
						}
						break;
					}
				}
			}

			if(!WritePersonnalConf())
			{
				ConfAccessLock.UnlockWrite();
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}

			ConfAccessLock.UnlockWrite();
			return true;
		}
	}
	ConfAccessLock.UnlockWrite();
	return true;
}

bool Entity_PKI::CheckRepositoryConf(EntityConfBodyRep &Conf)
{
	return true;
}

bool Entity_PKI::CheckRaConf(EntityConfBodyRa & Conf)
{
	size_t i, j;
	bool ret;
	PolicyValue newPolicy;

	//We check the validity of the DN fields
	ret = true;
	for(i=0; i<Conf.RA_CONF_PTR.get_dnspecs().size(); i++)
	{
		if(!IsValidObject(Conf.RA_CONF_PTR.get_dnspecs()[i].get_name()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_GET_DN_SPECIF);
			ERR_add_error_data(1, Conf.RA_CONF_PTR.get_dnspecs()[i].get_name().c_str());
			ret = false;
		}
	}

	//We remove any policy field that cannot be found in the DN specs
	for(i=0; i<Conf.RA_CONF_PTR.get_policies().size(); i++)
	{
		if(HashTable_String::IsValidPolicyField(Conf.RA_CONF_PTR.get_policies()[i].get_name(), Conf.RA_CONF_PTR.get_policies()[i].get_value()) == DN_TYPE_UNKNOWN)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		for(j=0; j<Conf.RA_CONF_PTR.get_dnspecs().size(); j++)
		{
			if(Conf.RA_CONF_PTR.get_dnspecs()[j].get_name() == Conf.RA_CONF_PTR.get_policies()[i].get_name())
				break;
		}
		//The DN spec is not known, remove the policy entry
		if(j == Conf.RA_CONF_PTR.get_dnspecs().size())
		{
			Conf.RA_CONF_PTR.get_policies().erase(Conf.RA_CONF_PTR.get_policies().begin() + i);
			i--;
		}
	}

	// We set all the DN fields not found
	// in the policy to "optional"
	for(i=0; i<Conf.RA_CONF_PTR.get_dnspecs().size(); i++)
	{
		for(j=0; j<Conf.RA_CONF_PTR.get_policies().size(); j++)
		{
			if(Conf.RA_CONF_PTR.get_policies()[j].get_name() == Conf.RA_CONF_PTR.get_dnspecs()[i].get_name())
				break;
		}
		//The DN field is not known, add a policy entry
		if(j == Conf.RA_CONF_PTR.get_policies().size())
		{
			newPolicy.set_name(Conf.RA_CONF_PTR.get_dnspecs()[i].get_name());
			newPolicy.set_value(STR_OPTIONAL);
			Conf.RA_CONF_PTR.get_policies().push_back(newPolicy);
		}
	}
	return ret;
}


bool Entity_PKI::CheckCaConf(EntityConfBodyCa & Conf)
{
	size_t i;
	bool ret;

	//We check the validity of the extensions
	ret = true;
	for(i=0; i<Conf.CA_CONF_PTR.get_exts().size(); i++)
	{
		if(!PKI_EXT::VerifyExtensionValue(Conf.CA_CONF_PTR.get_exts()[i].get_name().c_str(), Conf.CA_CONF_PTR.get_exts()[i].get_value().c_str()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_EXT_SYNTAX);
			ERR_add_error_data(2, "Cert Exts:", Conf.CA_CONF_PTR.get_exts()[i].get_name().c_str());
			ret = false;
		}
	}

	for(i=0; i<Conf.CA_CONF_PTR.get_crlexts().size(); i++)
	{
		if(!PKI_EXT::VerifyExtensionValue(Conf.CA_CONF_PTR.get_crlexts()[i].get_name().c_str(), Conf.CA_CONF_PTR.get_crlexts()[i].get_value().c_str()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_EXT_SYNTAX);
			ERR_add_error_data(2, "Cert Exts:", Conf.CA_CONF_PTR.get_crlexts()[i].get_name().c_str());
			ret = false;
		}
	}

	return ret;
}


bool Entity_PKI::GetRepositoriesList(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_ENTITIES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!ResourceOwnerIsMe(PKI_REPOSITORIES_RESOURCE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_GET_REPOSITORIES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}


	ConfAccessLock.LockRead();
	if(!response.set_repositories(pki_conf.get_repositories()))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ConfAccessLock.UnlockRead();
	
	return true;
}

bool Entity_PKI::SetRepositoriesList(COMMAND_PARAMETERS)
{
	size_t i, j;
	mVector<RepEntryInfo> tmpRep1;
	mVector<RepEntryInfo> tmpRep2;
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_ENTITIES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!ResourceOwnerIsMe(PKI_REPOSITORIES_RESOURCE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockWrite();
	tmpRep1 = pki_conf.get_repositories();
	tmpRep2 = body.get_repositories();

	// We validate the repositories, to make sure
	// the user didn't send something phoney
	for(i=0; i<tmpRep1.size(); i++)
	{
		for(j=0; j<tmpRep2.size(); j++)
		{
			if(tmpRep1[i].get_repositoryssl() == tmpRep2[j].get_repositoryssl() &&
				tmpRep1[i].get_name() == tmpRep2[j].get_name())
			{
				tmpRep1.erase(tmpRep1.begin()+i);
				tmpRep2.erase(tmpRep2.begin()+j);
				i--;
				break;
			}
		}		
	}

	if(tmpRep1.size() > 0 ||
		tmpRep2.size() > 0)
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	if(!pki_conf.set_repositories(body.get_repositories()))
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//Yell to other entities the new repositories list
	if(!UpdateEntityConfs(ENTITY_CONF_UPDATE_LINKS))
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!WritePersonnalConf())
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ConfAccessLock.UnlockWrite();
	
	return true;
}

X509_ACL_INIT Entity_PKI::X509_ACL_init(X509Acl & acls, int EntityType)
{
	bool Modified = false;
	size_t i, j;
	CA_Handler * usersCa;
	bool Found;
	mVector<InternalCaCert> certs;
	AclEntry newAcl;

	usersCa = GetInternalCA(INTERNAL_CA_TYPE_USER);
	if(!usersCa)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return X509_ACL_INIT_ERROR;
	}

	if(!usersCa->get_Certs(certs))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return X509_ACL_INIT_ERROR;
	}

	for(i=0; i<certs.size(); i++)
	{
		//We never add the an admin cert in the ACLs, it has all the rights
		if(X509_ACL_Validator::Static_IsPkiAdministrator(certs[i].get_cert().GetSerial(), acls.get_adminserials()))
			continue;
		

		// We add the new certs and delete the revoked ones
		Found = false;
		for(j=0; j<acls.get_aclEntries().size(); j++)
		{
			if(acls.get_aclEntries()[j].get_type() != ACL_ENTRY_TYPE_USER)
				continue;

			//Is it the certificate
			if(certs[i].get_cert().GetSerial() == acls.get_aclEntries()[j].get_serial())
			{
				//Is it revoked
				if(certs[i].get_state() == CERT_STATE_REVOKED)
				{
					acls.get_aclEntries().erase(acls.get_aclEntries().begin() + j);
					j--;
					Modified=true;
				}
				Found = true;
			}
		}

		//If we don't know the user, we add it to the list
		if(!Found && certs[i].get_state() == CERT_STATE_VALID)
		{
			//We add the user in the ACL, with no rights
			newAcl.set_type(ACL_ENTRY_TYPE_USER);
			newAcl.set_serial(certs[i].get_cert().GetSerial());
			newAcl.set_name(certs[i].get_cert().GetStringName());
			acls.get_aclEntries().push_back(newAcl);
			Modified = true;
		}
	}
	certs.clear();


	//Update the ACL with the groups
	for(i=0; i<pki_conf.get_conf().get_groups().size(); i++)
	{
		// We add the new groups
		Found = false;
		for(j=0; j<acls.get_aclEntries().size(); j++)
		{
			if(acls.get_aclEntries()[j].get_type() != ACL_ENTRY_TYPE_GROUP)
				continue;

			//Is it the group
			if(pki_conf.get_conf().get_groups()[i].get_serial() == acls.get_aclEntries()[j].get_serial())
			{
				Found = true;
				break;
			}
		}

		//If we don't know the user, we add it to the list
		if(!Found)
		{
			//We add the group in the ACL, with no rights
			newAcl.set_type(ACL_ENTRY_TYPE_GROUP);
			newAcl.set_serial(pki_conf.get_conf().get_groups()[i].get_serial());
			newAcl.set_name(pki_conf.get_conf().get_groups()[i].get_name());
			acls.get_aclEntries().push_back(newAcl);
			Modified = true;
		}
	}

	//We now delete the removed groups
	for(i=0; i<acls.get_aclEntries().size(); i++)
	{
		if(acls.get_aclEntries()[i].get_type() == ACL_ENTRY_TYPE_GROUP)
		{
			//Search corresponding group
			Found = false;
			for(j=0; j<pki_conf.get_conf().get_groups().size(); j++)
			{
				if(pki_conf.get_conf().get_groups()[j].get_name() == acls.get_aclEntries()[i].get_name())
				{
					Found = true;
					break;
				}
			}

			//Group was no found we remove it
			if(!Found)
			{
				acls.get_aclEntries().erase(acls.get_aclEntries().begin() + i);
			}
		}
	}

	
	
	
	//We initialize the available ACL types
	acls.get_aclType().clear();
	switch(EntityType)
	{
		case ENTITY_TYPE_PKI:
			Entity_PKI::GetACL_List(acls.get_aclType());
			break;
			
		case ENTITY_TYPE_REPOSITORY:
			Entity_REPOSITORY::GetACL_List(acls.get_aclType());
			break;
			
		case ENTITY_TYPE_RA:
			Entity_RA::GetACL_List(acls.get_aclType());
			break;
			
		case ENTITY_TYPE_KEY_STORE:
			Entity_KEYSTORE::GetACL_List(acls.get_aclType());
			break;
			
		case ENTITY_TYPE_EE:
			Entity_EE::GetACL_List(acls.get_aclType());
			break;
			
		case ENTITY_TYPE_BACKUP:
			Entity_BACKUP::GetACL_List(acls.get_aclType());
			break;
			
		case ENTITY_TYPE_PUBLICATION:
			Entity_PUBLICATION::GetACL_List(acls.get_aclType());
			break;
			
		case ENTITY_TYPE_CA:
			Entity_CA::GetACL_List(acls.get_aclType());
			break;
	}

	if(Modified)
		return X509_ACL_INIT_MODIFIED;
	else
		return X509_ACL_INIT_SAME;
}

bool Entity_PKI::Add_ENTITY_CONF(const mString & EntityName, const X509 * EntityCert, int Type)
{
	size_t i;
	DnSpecs newDnSpecs;
	EntityLinks newLink;
	EntityLinkInfo newLinkEntry;
	mVector< StoredEntityConf >::iterator newConf;
	mVector< EntityLinks >::iterator LinkIndex;

	//Adding the new entity to the list of available one
	newConf = pki_conf.get_confs().insert(pki_conf.get_confs().end());
	
	//Copy entity cert
	if(!newConf->get_certificate().SetCert(EntityCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		pki_conf.get_confs().erase(newConf);
		return false;
	}
	newConf->get_conf().set_name(EntityName);	

	//Copy default PKI email conf
	if(!newConf->get_conf().set_mailConf(pki_conf.get_conf().get_mailConf()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		pki_conf.get_confs().erase(newConf);
		return false;
	}

	//Allocate a default conf
	if(!newConf->get_conf().get_body().set_type(Type))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		pki_conf.get_confs().erase(newConf);
		return false;
	}

	switch(Type)
	{
		case ENTITY_TYPE_CA:
			if(!newConf->get_conf().get_body().get_caConf().set_type(CA_CONF_VERSION))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				pki_conf.get_confs().erase(newConf);
				return false;
			}
			ASN1_BIT_STRING_set_bit((ASN1_BIT_STRING*)newConf->get_conf().get_body().get_caConf().CA_CONF_PTR.get_flags(), CA_ALLOW_CSR_EXTS, 0);
			ASN1_BIT_STRING_set_bit((ASN1_BIT_STRING*)newConf->get_conf().get_body().get_caConf().CA_CONF_PTR.get_flags(), CA_CSR_EXTS_OVERWRITE, 0);
			break;
		case ENTITY_TYPE_RA:
			if(!newConf->get_conf().get_body().get_raConf().set_type(RA_CONF_VERSION))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				pki_conf.get_confs().erase(newConf);
				return false;
			}

			newDnSpecs.set_name("commonName");
			newDnSpecs.set_default("");
			newDnSpecs.set_comment("Common Name (eg. YOUR name)");
			newDnSpecs.set_min(0);
			newDnSpecs.set_max(64);
			newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_dnspecs().push_back(newDnSpecs);

			newDnSpecs.set_name("emailAddress");
			newDnSpecs.set_default("");
			newDnSpecs.set_comment("Email Address");
			newDnSpecs.set_min(2);
			newDnSpecs.set_max(40);
			newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_dnspecs().push_back(newDnSpecs);

			newDnSpecs.set_name("countryName");
			newDnSpecs.set_default("");
			newDnSpecs.set_comment("Country Name");
			newDnSpecs.set_min(2);
			newDnSpecs.set_max(2);
			newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_dnspecs().push_back(newDnSpecs);

			newDnSpecs.set_name("stateOrProvinceName");
			newDnSpecs.set_default("");
			newDnSpecs.set_comment("State or Province Name (full name)");
			newDnSpecs.set_min(0);
			newDnSpecs.set_max(0);
			newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_dnspecs().push_back(newDnSpecs);

			newDnSpecs.set_name("localityName");
			newDnSpecs.set_default("");
			newDnSpecs.set_comment("Locality Name (eg. city)");
			newDnSpecs.set_min(0);
			newDnSpecs.set_max(0);
			newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_dnspecs().push_back(newDnSpecs);

			newDnSpecs.set_name("organizationName");
			newDnSpecs.set_default("");
			newDnSpecs.set_comment("Organization Name (eg. company)");
			newDnSpecs.set_min(0);
			newDnSpecs.set_max(0);
			newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_dnspecs().push_back(newDnSpecs);

			newDnSpecs.set_name("organizationalUnitName");
			newDnSpecs.set_default("");
			newDnSpecs.set_comment("Organizational Unit Name (eg. section)");
			newDnSpecs.set_min(0);
			newDnSpecs.set_max(0);
			newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_dnspecs().push_back(newDnSpecs);

			CheckRaConf(newConf->get_conf().get_body().get_raConf());

			newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.set_defaultValidity(365);
			newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.set_minkeylen(1024);
			newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.set_minpwdlen(6);

			ASN1_BIT_STRING_set_bit((ASN1_BIT_STRING*)newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_ALLOW_SERVER_SIDE_KEY_GEN, 1);
			ASN1_BIT_STRING_set_bit((ASN1_BIT_STRING*)newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_ALLOW_PKCS10_IMPORT, 1);
			ASN1_BIT_STRING_set_bit((ASN1_BIT_STRING*)newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_ALLOW_OP_SC_KEY_GEN, 1);
			ASN1_BIT_STRING_set_bit((ASN1_BIT_STRING*)newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_PUSH_KEYS_TO_KEY_ESCROW, 1);
			ASN1_BIT_STRING_set_bit((ASN1_BIT_STRING*)newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_REMOVE_UNKNOWN_DN_FIELDS, 1);
			ASN1_BIT_STRING_set_bit((ASN1_BIT_STRING*)newConf->get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_WARN_PROFILE_OWNER_EXPIRING, 1);

			break;
		case ENTITY_TYPE_PUBLICATION:
			if(!newConf->get_conf().get_body().get_pubConf().set_type(PUB_CONF_VERSION))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				pki_conf.get_confs().erase(newConf);
				return false;
			}

			newConf->get_conf().get_body().get_pubConf().PUB_CONF_PTR.set_ocspenabled(0);
			break;
		case ENTITY_TYPE_EE:
			if(!newConf->get_conf().get_body().get_eeConf().set_type(EE_CONF_VERSION))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				pki_conf.get_confs().erase(newConf);
				return false;
			}
			newConf->get_conf().get_body().get_eeConf().EE_CONF_PTR.set_siteName("http://localhost");
			
			newConf->get_conf().get_body().get_eeConf().EE_CONF_PTR.set_dnValDisclaimer(
				_sv("In consideration of your use of the Service, you agree to provide " \
				"true, accurate, current and complete information about yourself as " \
				"prompted by the DN  validation form. If you provide any information " \
				"that is untrue, inaccurate, not current or incomplete, or we have " \
				"reasonable grounds to suspect that such information is untrue, " \
				"inaccurate, not current or incomplete, we have the right to suspend " \
				"or terminate your account and refuse any and all current or future  " \
				"use of the Service (or any portion thereof)."));
			break;
		case ENTITY_TYPE_KEY_STORE:
		case ENTITY_TYPE_BACKUP:
			break;
		case ENTITY_TYPE_REPOSITORY:
			if(!newConf->get_conf().get_body().get_repConf().set_type(REP_CONF_VERSION))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				pki_conf.get_confs().erase(newConf);
				return false;
			}

			newConf->get_conf().get_body().get_repConf().REP_CONF_PTR.set_enabled(0);
			break;
	}
	
	newConf->get_conf().set_version(0);

	
	//Initialize its link struct
	newLink.get_src().set_name(EntityName);
	newLink.get_src().set_type(Type);

	// We add by default all the repositories to the list, it
	// will allow the newly created entity to go seek
	// its conf
	for(i=0; i < pki_conf.get_repositories().size(); i++)
	{
		//Is it me ?
		if(EntityName == pki_conf.get_repositories()[i].get_name())
			continue;

		newLinkEntry.set_name(pki_conf.get_repositories()[i].get_name());
		newLinkEntry.set_type(ENTITY_TYPE_REPOSITORY);

		newLink.get_dsts().push_back(newLinkEntry);
	}

	//Are we creating a repository? if yes then 
	//we let all other reps know we exists
	newLinkEntry.set_name(EntityName);
	newLinkEntry.set_type(ENTITY_TYPE_REPOSITORY);

	if(Type == ENTITY_TYPE_REPOSITORY)
	{
		for(i=0; i < pki_conf.get_links().size(); i++)
		{
			//Is it a rep
			if(pki_conf.get_links()[i].get_src().get_type() != ENTITY_TYPE_REPOSITORY)
				continue;

			//Is it me ?
			if(EntityName == pki_conf.get_links()[i].get_src().get_name())
				continue;

			pki_conf.get_links()[i].get_dsts().push_back(newLinkEntry);
		}
	}
	//add new link entry to list
	LinkIndex = pki_conf.get_links().insert(pki_conf.get_links().end(), newLink);

	newConf->get_conf().set_isOK();

	if(!UpdateEntityConfs(ENTITY_CONF_UPDATE_ALL, newConf->get_conf()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		pki_conf.get_confs().erase(newConf);
		pki_conf.get_links().erase(LinkIndex);
		// This will delete the added link entries
		if(Type == ENTITY_TYPE_REPOSITORY)
			UpdateEntityConfs(ENTITY_CONF_UPDATE_LINKS);

		return false;
	}

	// We synchronize the links of the modified entities
	if(Type == ENTITY_TYPE_REPOSITORY && !UpdateEntityConfs(ENTITY_CONF_UPDATE_LINKS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		pki_conf.get_confs().erase(newConf);
		pki_conf.get_links().erase(LinkIndex);
		return false;
	}

	return true;
}


bool Entity_PKI::UserLockResourse(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_ENTITIES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!ResourceLock(body.get_resourceName()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity_PKI::UserUnlockResourse(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_ENTITIES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ResourceUnlock(body.get_resourceName());
	return true;
}

bool Entity_PKI::GetEntityAcl(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_ACLS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	
	ConfAccessLock.LockRead();

	EntityConf & conf = GetEntityConfiguration(body.get_entityCert());
	if(!conf)
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN_ENTITY);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, (conf.get_body().get_type() != ENTITY_TYPE_PKI?ACL_TYPE_MANAGE_ENTITIES:ACL_TYPE_MODIFY_PKI_STATUS)))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!ResourceOwnerIsMe(body.get_entityCert().GetStringName()))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_aclsConf(conf.get_acls()))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	ConfAccessLock.UnlockRead();

	return true;
}

bool Entity_PKI::SetEntityAcl(COMMAND_PARAMETERS)
{
	size_t i, j;	

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockWrite();

	EntityConf & conf = GetEntityConfiguration(body.get_entityAcl().get_entityCert());
	if(!conf)
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN_ENTITY);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, (conf.get_body().get_type() != ENTITY_TYPE_PKI?ACL_TYPE_MANAGE_ENTITIES:ACL_TYPE_MODIFY_PKI_STATUS)))
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!ResourceOwnerIsMe(body.get_entityAcl().get_entityCert().GetStringName()))
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	for(i=0; i<conf.get_acls().get_aclEntries().size(); i++)
	{
		// Get the associated ACL from the request
		for(j=0; j<body.get_entityAcl().get_aclEntries().size(); j++)
		{
			if(conf.get_acls().get_aclEntries()[i].get_serial() == body.get_entityAcl().get_aclEntries()[j].get_serial() &&
				conf.get_acls().get_aclEntries()[j].get_type() == body.get_entityAcl().get_aclEntries()[j].get_type())
			{
				conf.get_acls().get_aclEntries()[i].set_acls(body.get_entityAcl().get_aclEntries()[j].get_acls());
				break;
			}
		}
	}

	//Is it me ?
	if(conf.get_body().get_type() == ENTITY_TYPE_PKI)
	{
		if(!AclValidator.SetACL(conf.get_acls()))
		{
			ConfAccessLock.UnlockWrite();
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	else
	{
		//We update the crypted conf
		if(!UpdateEntityConfs(0, conf, true))
		{
			ConfAccessLock.UnlockWrite();
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	if(!WritePersonnalConf())
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ConfAccessLock.UnlockWrite();
	return true;
}

bool Entity_PKI::GetEntityMailConf(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_MAIL_CONF))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	ConfAccessLock.LockRead();

	EntityConf & conf = GetEntityConfiguration(body.get_entityCert());
	if(!conf)
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN_ENTITY);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, (conf.get_body().get_type() != ENTITY_TYPE_PKI?ACL_TYPE_MANAGE_ENTITIES:ACL_TYPE_MODIFY_PKI_STATUS)))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!ResourceOwnerIsMe(body.get_entityCert().GetStringName()))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_mailConf(conf.get_mailConf()))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	ConfAccessLock.UnlockRead();

	return true;
}


bool Entity_PKI::SetEntityMailConf(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockWrite();

	EntityConf & conf = GetEntityConfiguration(body.get_entityMailConf().get_entityCert());
	if(!conf)
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN_ENTITY);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, (conf.get_body().get_type() != ENTITY_TYPE_PKI?ACL_TYPE_MANAGE_ENTITIES:ACL_TYPE_MODIFY_PKI_STATUS)))
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!ResourceOwnerIsMe(body.get_entityMailConf().get_entityCert().GetStringName()))
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!conf.set_mailConf(body.get_entityMailConf().get_mailConf()))
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//Is it me ?
	if(conf.get_body().get_type() == ENTITY_TYPE_PKI)
	{
		if(!m_Jobs.SetMailerInfo(m_EntityMail, conf.get_mailConf()))
		{
			ConfAccessLock.UnlockWrite();
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	else
	{
		//We update the crypted conf
		if(!UpdateEntityConfs(0, conf, true))
		{
			ConfAccessLock.UnlockWrite();
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	if(!WritePersonnalConf())
	{
		ConfAccessLock.UnlockWrite();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ConfAccessLock.UnlockWrite();
	return true;
}

bool Entity_PKI::STORED_ENTITY_CONF_to_name_x509(const StoredEntityConf & conf, mString & name, PKI_CERT & cert)
{
	name = conf.get_conf().get_name();

	if(!(cert = conf.get_certificate()) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	return true;
}


bool Entity_PKI::SynchronizeLinks(EntityConf & currConf)
{
	size_t linkIndex, i, j, v_index;
	bool Found;
	EntityConf savedConf;



	//We save it, not to affect the conf
	//if an error occurs
	if(!(savedConf = currConf))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}


	//Seek for corresponding entity in the links conf
	for(linkIndex=0; linkIndex < pki_conf.get_links().size(); linkIndex++)
	{
		if(pki_conf.get_links()[linkIndex].get_src().get_name() == savedConf.get_name())
		{
			break;
		}
	}
	if(linkIndex == pki_conf.get_links().size())
	{
		return true;
	}


	EntityLinks & lEntity = pki_conf.get_links()[linkIndex];
	
	//We first empty the REP, CA, RA, Publication and Key Store list
	savedConf.get_repositories().clear();
	savedConf.get_cas().Clear();
	savedConf.get_ras().Clear();
	savedConf.get_ees().Clear();
	savedConf.get_backups().Clear();
	savedConf.get_keystores().Clear();
	savedConf.get_publications().Clear();


	// Let's do the synchro !
	for(i=0; i < lEntity.get_dsts().size(); i++)
	{

		Found = false;
		//Update repositories for all kind of entities

		switch(lEntity.get_dsts()[i].get_type())
		{
			case ENTITY_TYPE_REPOSITORY:
				//Seek corresponding respositories info
				for(j=0; j < pki_conf.get_repositories().size(); j++)
				{
					//Is the repository we're looking for ?
					if(!(pki_conf.get_repositories()[j].get_name() == lEntity.get_dsts()[i].get_name()))
						continue;


					//Yes, add it to the list for this entity
					v_index = savedConf.get_repositories().size();
					savedConf.get_repositories().insert(savedConf.get_repositories().end());
					savedConf.get_repositories()[v_index] = pki_conf.get_repositories()[j];

					if(!savedConf.get_repositories()[v_index].set_flags(lEntity.get_dsts()[i].get_flags()))
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					Found = true;
					break;
				}
				break;
				
			case  ENTITY_TYPE_RA:
				//Seek corresponding RA info
				for(j=0; j <pki_conf.get_confs().size(); j++)
				{
					//Is it the seeked one ?
					if(!(pki_conf.get_confs()[j].get_conf().get_name() == lEntity.get_dsts()[i].get_name()) ||
						pki_conf.get_confs()[j].get_conf().get_body().get_type() != (int)lEntity.get_dsts()[i].get_type())
						continue;

					v_index = savedConf.get_ras().get_list().size();
					savedConf.get_ras().get_list().insert(savedConf.get_ras().get_list().end());

					// We're setting up an EE, we must copy its RA's DnSpecs and Policies
					if(savedConf.get_body().get_type() == ENTITY_TYPE_EE &&
						!SynchronizeEeOnRa(savedConf, pki_conf.get_confs()[j].get_conf()))
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					
					if(!STORED_ENTITY_CONF_to_name_x509(pki_conf.get_confs()[j], savedConf.get_ras().get_list()[v_index].get_name(), savedConf.get_ras().get_list()[v_index].get_rassl()))
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					Found=true;
					break;
				}
				break;

			case  ENTITY_TYPE_CA:
				//Seek corresponding CA info
				for(j=0; j <pki_conf.get_confs().size(); j++)
				{
					//Is it the seeked one ?
					if(!(pki_conf.get_confs()[j].get_conf().get_name() == lEntity.get_dsts()[i].get_name()) ||
						pki_conf.get_confs()[j].get_conf().get_body().get_type() != (int)lEntity.get_dsts()[i].get_type())
						continue;

					v_index = savedConf.get_cas().get_list().size();
					savedConf.get_cas().get_list().insert(savedConf.get_cas().get_list().end());
					
					if(!STORED_ENTITY_CONF_to_name_x509(pki_conf.get_confs()[j], savedConf.get_cas().get_list()[v_index].get_name(), savedConf.get_cas().get_list()[v_index].get_cassl()))
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					Found=true;
					break;
				}
				break;

			case  ENTITY_TYPE_KEY_STORE:
				//Seek corresponding Key Store info
				for(j=0; j <pki_conf.get_confs().size(); j++)
				{
					//Is it the seeked one ?
					if(!(pki_conf.get_confs()[j].get_conf().get_name() == lEntity.get_dsts()[i].get_name()) ||
						pki_conf.get_confs()[j].get_conf().get_body().get_type() != (int)lEntity.get_dsts()[i].get_type())
						continue;

					v_index = savedConf.get_keystores().get_list().size();
					savedConf.get_keystores().get_list().insert(savedConf.get_keystores().get_list().end());
					
					if(!STORED_ENTITY_CONF_to_name_x509(pki_conf.get_confs()[j], savedConf.get_keystores().get_list()[v_index].get_name(), savedConf.get_keystores().get_list()[v_index].get_keyssl()))
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					Found=true;
					break;
				}
				break;

			case  ENTITY_TYPE_EE:
				//Seek corresponding EE info
				for(j=0; j <pki_conf.get_confs().size(); j++)
				{
					//Is it the seeked one ?
					if(!(pki_conf.get_confs()[j].get_conf().get_name() == lEntity.get_dsts()[i].get_name()) ||
						pki_conf.get_confs()[j].get_conf().get_body().get_type() != (int)lEntity.get_dsts()[i].get_type())
						continue;

					v_index = savedConf.get_ees().get_list().size();
					savedConf.get_ees().get_list().insert(savedConf.get_ees().get_list().end());
					
					if(!STORED_ENTITY_CONF_to_name_x509(pki_conf.get_confs()[j], savedConf.get_ees().get_list()[v_index].get_name(), savedConf.get_ees().get_list()[v_index].get_eessl()))
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					Found=true;
					break;
				}
				break;

			case  ENTITY_TYPE_BACKUP:
				//Seek corresponding Key Store info
				for(j=0; j <pki_conf.get_confs().size(); j++)
				{
					//Is it the seeked one ?
					if(!(pki_conf.get_confs()[j].get_conf().get_name() == lEntity.get_dsts()[i].get_name()) ||
						pki_conf.get_confs()[j].get_conf().get_body().get_type() != (int)lEntity.get_dsts()[i].get_type())
						continue;

					v_index = savedConf.get_backups().get_list().size();
					savedConf.get_backups().get_list().insert(savedConf.get_backups().get_list().end());
					
					if(!STORED_ENTITY_CONF_to_name_x509(pki_conf.get_confs()[j], savedConf.get_backups().get_list()[v_index].get_name(), savedConf.get_backups().get_list()[v_index].get_backupssl()))
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					Found=true;
					break;
				}
				break;

			case  ENTITY_TYPE_PUBLICATION:
				//Seek corresponding Pub Store info
				for(j=0; j < pki_conf.get_publications().get_list().size(); j++)
				{
					//Is the repository we're looking for ?
					if(!(pki_conf.get_publications().get_list()[j].get_name() == lEntity.get_dsts()[i].get_name()))
						continue;


					//Yes, add it to the list for this entity
					v_index = savedConf.get_publications().get_list().size();
					savedConf.get_publications().get_list().insert(savedConf.get_publications().get_list().end());
					savedConf.get_publications().get_list()[v_index] = pki_conf.get_publications().get_list()[j];
					Found = true;
					break;
				}
				break;

			default:
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
		}

		// If we didn't find the corresponding entry
		// we erase the link
		if(!Found)
		{
			lEntity.get_dsts().erase(lEntity.get_dsts().begin() + i);
			i--;
		}
	}

	// We need to synchronize the Publication method 
	// when working on a Publication entity

	if(savedConf.get_body().get_type() == ENTITY_TYPE_PUBLICATION)
	{
		// First delete the CA that have been 
		// removed from our link
		for(i=0; i<savedConf.get_body().get_pubConf().PUB_CONF_PTR.get_publications().size(); i++)
		{
			for(j=0; j<savedConf.get_cas().get_list().size(); j++)
			{
				if(savedConf.get_body().get_pubConf().PUB_CONF_PTR.get_publications()[i].get_caname() == savedConf.get_cas().get_list()[j].get_name())
					break;
			}
			// Not found, has been removed
			if(j == savedConf.get_cas().get_list().size())
			{
				savedConf.get_body().get_pubConf().PUB_CONF_PTR.get_publications().erase(savedConf.get_body().get_pubConf().PUB_CONF_PTR.get_publications().begin() + i);
				i--;
			}
		}

		// Now add the new CAs that have been linked to me
		for(i=0; i<savedConf.get_cas().get_list().size(); i++)
		{
			for(j=0; j<savedConf.get_body().get_pubConf().PUB_CONF_PTR.get_publications().size(); j++)
			{
				if(savedConf.get_body().get_pubConf().PUB_CONF_PTR.get_publications()[j].get_caname() == savedConf.get_cas().get_list()[i].get_name())
					break;
			}
			// Not found, need to be added
			if(j == savedConf.get_body().get_pubConf().PUB_CONF_PTR.get_publications().size())
			{
				savedConf.get_body().get_pubConf().PUB_CONF_PTR.get_publications().insert(savedConf.get_body().get_pubConf().PUB_CONF_PTR.get_publications().end());
				savedConf.get_body().get_pubConf().PUB_CONF_PTR.get_publications()[j].set_caname(savedConf.get_cas().get_list()[i].get_name());
			}
		}
	}

	//We copy the working conf to the real conf
	if(! (currConf = savedConf) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}


bool Entity_PKI::UpdateEntityConfs(int What, EntityConf & OneConf, bool ForceCryptGen)
{
	size_t i, j, k;
	CA_Handler * usersCa;
	EVP_PKEY * repPubKey;
	bool ConfChanged;
	mVector<InternalCaCert> certs;

	ConfAccessLock.LockWrite();

	if(!OneConf || OneConf.get_body().get_type() == ENTITY_TYPE_PKI)
	{
		// We update for the local pki conf 
		// with what was requested

		//Shall we synchronize the groups ?
		if(What & ENTITY_CONF_UPDATE_ALL || What & ENTITY_CONF_UPDATE_GROUPS)
		{
			//We remove any revoke cert
			usersCa = GetInternalCA(INTERNAL_CA_TYPE_USER);
			if(!usersCa)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}

			if(!usersCa->get_Certs(certs))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}

			for(i=0; i<certs.size(); i++)
			{
				//We ignore the Admin cert
				if(X509_ACL_Validator::Static_IsPkiAdministrator(certs[i].get_serial(), pki_conf.get_conf().get_acls().get_adminserials()))
					continue;

				//We're only interessed by the revoked ones
				if(certs[i].get_state() != CERT_STATE_REVOKED)
					continue;
				

				// We remove the cert form all the groups we can find it in
				for(j=0; j<pki_conf.get_conf().get_groups().size(); j++)
				{
					for(k=0; k<pki_conf.get_conf().get_groups()[j].get_usersSerial().size(); k++)
					{
						if(pki_conf.get_conf().get_groups()[j].get_usersSerial()[k] == certs[i].get_serial())
						{
							pki_conf.get_conf().get_groups()[j].get_usersSerial().erase(pki_conf.get_conf().get_groups()[j].get_usersSerial().begin() + k);
							k--;
						}
					}
				}
			}
			certs.clear();
		}


		//Shall we update the ACL ?
		if(What & ENTITY_CONF_UPDATE_ALL || What & ENTITY_CONF_UPDATE_ACL || What & ENTITY_CONF_UPDATE_GROUPS)
		{
			switch(X509_ACL_init(pki_conf.get_conf().get_acls(), ENTITY_TYPE_PKI))
			{
				case X509_ACL_INIT_ERROR:
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					ConfAccessLock.UnlockWrite();
					return false;
					break;
				case X509_ACL_INIT_MODIFIED:
					if(!AclValidator.SetACL(pki_conf.get_conf().get_acls()))
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
						ConfAccessLock.UnlockWrite();
						return false;
					}
					break;
				default:
					break;
			}
		}
		//Shall we update the links
		if(What & ENTITY_CONF_UPDATE_ALL || What & ENTITY_CONF_UPDATE_LINKS)
		{
			if(!SynchronizeLinks(pki_conf.get_conf()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;			
			}
			
			if(!m_Jobs.SetRepositories(pki_conf.get_conf().get_repositories()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}
		}
		
		// Update admins mails
		if(What & ENTITY_CONF_UPDATE_ALL || What & ENTITY_CONF_UPDATE_ADMINS)
		{
			if(!m_Jobs.SetAdminMails(pki_conf.get_conf().get_acls().get_adminserials()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}
		}
		
		// Update CRLs
		if(What & ENTITY_CONF_UPDATE_ALL || What & ENTITY_CONF_UPDATE_CRL)
		{
			if(!AclValidator.SetCRL(pki_conf.get_conf().get_crls()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}
		}
	}



	//For each conf we update what was requested
	for(i=0; i<pki_conf.get_confs().size(); i++)
	{
		if(OneConf && !(OneConf.get_name() == pki_conf.get_confs()[i].get_conf().get_name()) ) continue;
		
		ConfChanged = ForceCryptGen;

		//Shall we update the CRL
		if(What & ENTITY_CONF_UPDATE_ALL || What & ENTITY_CONF_UPDATE_CRL)
		{
			if(!(pki_conf.get_confs()[i].get_conf().get_crls() = pki_conf.get_conf().get_crls()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;			
			}
			ConfChanged = true;
		}

		//Shall we update the PKI admins list ?
		if(What & ENTITY_CONF_UPDATE_ALL || What & ENTITY_CONF_UPDATE_ADMINS)
		{
			if(!pki_conf.get_confs()[i].get_conf().get_acls().set_adminserials(pki_conf.get_conf().get_acls().get_adminserials()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;			
			}
			ConfChanged = true;
		}

		//Shall we update the groups ?
		if(What & ENTITY_CONF_UPDATE_ALL || What & ENTITY_CONF_UPDATE_GROUPS)
		{
			if(!(pki_conf.get_confs()[i].get_conf().set_groups(pki_conf.get_conf().get_groups())))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;			
			}
			ConfChanged = true;
		}


		//Shall we update the ACL
		if(What & ENTITY_CONF_UPDATE_ALL || What & ENTITY_CONF_UPDATE_ACL || What & ENTITY_CONF_UPDATE_GROUPS)
		{
			switch(X509_ACL_init(pki_conf.get_confs()[i].get_conf().get_acls(), pki_conf.get_confs()[i].get_conf().get_body().get_type()))
			{
				case X509_ACL_INIT_MODIFIED:
					ConfChanged = true;
					break;

				case X509_ACL_INIT_ERROR:
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					ConfAccessLock.UnlockWrite();
					return false;
					break;
				
				default:
					break;
			}
		}

		//Shall we update the Repositories list or the links
		if(What & ENTITY_CONF_UPDATE_ALL || What & ENTITY_CONF_UPDATE_LINKS)
		{
			if(!SynchronizeLinks(pki_conf.get_confs()[i].get_conf()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;			
			}
			ConfChanged = true;
		}
		if(ConfChanged)
		{
			pki_conf.get_confs()[i].get_conf().set_version(pki_conf.get_confs()[i].get_conf().get_version() + 1);

			//Seek crypted conf to update it id we don't find it we add it
			for(j=0; j<pki_conf.get_cryptConfs().get_confs().get_allConfs().size(); j++)
			{
				//Is it the seeked one ?
				if(pki_conf.get_confs()[i].get_certificate() == pki_conf.get_cryptConfs().get_confs().get_allConfs()[j].get_recipient())
					break;
			}
			//Did we find it ?
			if(j == pki_conf.get_cryptConfs().get_confs().get_allConfs().size())
			{
				//No
				pki_conf.get_cryptConfs().get_confs().get_allConfs().insert(pki_conf.get_cryptConfs().get_confs().get_allConfs().begin() + j);
				//Set public key
				if(!pki_conf.get_cryptConfs().get_confs().get_allConfs()[j].set_recipient(pki_conf.get_confs()[i].get_certificate().GetX509_PUBKEY()))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					ConfAccessLock.UnlockWrite();
					return false;
				}
			}

			//Sign and crypt conf
			repPubKey = X509_get_pubkey(pki_conf.get_confs()[i].get_certificate().GetX509());
			if(!repPubKey)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}
			//Encrypt the body with the entity public key and generate signature with our private key
			if(!pki_conf.get_confs()[i].get_conf().to_SignEncrypt(pki_conf.get_cryptConfs().get_confs().get_allConfs()[j].get_crypted(), m_EntityKey.GetRsaKey(), repPubKey, EVP_sha1(), EVP_des_ede3_cbc()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				EVP_PKEY_free(repPubKey);
				return false;
			}
			EVP_PKEY_free(repPubKey);
		}
	}

	ConfAccessLock.UnlockWrite();
	return true;
}



bool Entity_PKI::GetEntityAudits(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_AUDITS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	
	ConfAccessLock.LockRead();

	EntityConf & conf = GetEntityConfiguration(body.get_entityCert());
	if(!conf)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN_ENTITY);
		ConfAccessLock.UnlockRead();
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, (conf.get_body().get_type() != ENTITY_TYPE_PKI?ACL_TYPE_MANAGE_ENTITIES:ACL_TYPE_MODIFY_PKI_STATUS)))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		return false;
	}
	
	if(!ResourceOwnerIsMe(body.get_entityCert().GetStringName()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		return false;
	}

	if(!response.set_audits(conf.get_audits()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();

	return true;
}

bool Entity_PKI::SetEntityAudits(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockWrite();

	EntityConf & conf = GetEntityConfiguration(body.get_entityAudits().get_entityCert());
	if(!conf)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN_ENTITY);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, (conf.get_body().get_type() != ENTITY_TYPE_PKI?ACL_TYPE_MANAGE_ENTITIES:ACL_TYPE_MODIFY_PKI_STATUS)))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockWrite();
		return false;
	}
	
	if(!ResourceOwnerIsMe(body.get_entityAudits().get_entityCert().GetStringName()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	if(!conf.set_audits(body.get_entityAudits().get_audits()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	//Is it me ?
	if(conf.get_body().get_type() == ENTITY_TYPE_PKI)
	{
		if(!m_Logging->SetAudit(conf.get_audits()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ConfAccessLock.UnlockWrite();
			return false;
		}
	}
	else
	{
		//We update the crypted conf
		if(!UpdateEntityConfs(0, conf, true))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ConfAccessLock.UnlockWrite();
			return false;
		}
	}

	if(!WritePersonnalConf())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}
	ConfAccessLock.UnlockWrite();
	return true;
}

bool Entity_PKI::GetEntityLogsType(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_ENTITIES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_LOGS_TYPE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	

	switch(body.get_entityType())
	{
		case ENTITY_TYPE_PKI:
			response.set_logsType(LogsType);
			break;
			
		case ENTITY_TYPE_REPOSITORY:
			Entity_REPOSITORY::LogsTypeGet(response.get_logsType());
			break;
			
		case ENTITY_TYPE_RA:
			Entity_RA::LogsTypeGet(response.get_logsType());
			break;
			
		case ENTITY_TYPE_PUBLICATION:
			Entity_PUBLICATION::LogsTypeGet(response.get_logsType());
			break;
			
		case ENTITY_TYPE_KEY_STORE:
			Entity_KEYSTORE::LogsTypeGet(response.get_logsType());
			break;
			
		case ENTITY_TYPE_EE:
			Entity_KEYSTORE::LogsTypeGet(response.get_logsType());
			break;
			
		case ENTITY_TYPE_BACKUP:
			Entity_BACKUP::LogsTypeGet(response.get_logsType());
			break;
			
		case ENTITY_TYPE_CA:
			Entity_CA::LogsTypeGet(response.get_logsType());
			break;

		default:
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
			return false;
	}
	return true;
}



EntityConf & Entity_PKI::GetEntityConfiguration(const PKI_CERT & EntityCert)
{
	if(EntityCert == pki_conf.get_cas().get_pkicert())
	{
		return pki_conf.get_conf();
	}

	return STORED_ENTITY_CONF_get(pki_conf.get_confs(), EntityCert).get_conf();
}


bool Entity_PKI::GetEntitiesLinks(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MODIFY_PKI_STATUS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!ResourceOwnerIsMe(PKI_ENTITIES_LINKS_RESOURCE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_ENTITIES_LINKS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();

	if(!response.set_entitiesLinks(pki_conf.get_links()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();
	return true;
}


bool Entity_PKI::IsEntityKnown(const mString & Name, int Type)
{
	size_t i;

	//Check corresponding entity
	for(i=0; i<pki_conf.get_confs().size(); i++)
	{

		if(pki_conf.get_confs()[i].get_conf().get_name() == Name && 
			Type == pki_conf.get_confs()[i].get_conf().get_body().get_type())
		{
			return true;
		}
	}
	return false;
}


bool Entity_PKI::SetEntitiesLinks(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MODIFY_PKI_STATUS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!ResourceOwnerIsMe(PKI_ENTITIES_LINKS_RESOURCE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//Let's do some integrity check !
	size_t i;
	size_t j;
	size_t uniqueTypeCtr;
	int FoundCtr;
	int EntityType;
	int LinkType;
	EntityLinks * currEntry;
	EntityLinks * currEntry2;
	EntityLinkInfo * currLink2;
	mVector<EntityLinks> Links;


	Links = body.get_entitiesLinks();

	ConfAccessLock.LockWrite();

	//Check that all conf are there
	FoundCtr = 0;
	for(i=0; i<pki_conf.get_confs().size(); i++)
	{
		//Is the conf present in the links ?
		currEntry = get_EntityLink(Links,  pki_conf.get_confs()[i].get_conf().get_name());
		if( currEntry &&
			(int)currEntry->get_src().get_type() == pki_conf.get_confs()[i].get_conf().get_body().get_type())
		{
			FoundCtr++;
		}
	}
	if(FoundCtr != (int)pki_conf.get_confs().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	//Is the PKI present in the links ?
	currEntry = get_EntityLink(Links, pki_conf.get_conf().get_name());
	if( !currEntry ||
		currEntry->get_src().get_type() != ENTITY_TYPE_PKI)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		ConfAccessLock.UnlockWrite();
		return false;
	}



	// Check that the user didn't add some unknown entities
	// And that it links to some known entities
	for(i=0; i<Links.size(); i++)
	{
		EntityType = Links[i].get_src().get_type();

		// We ignore the PKI entity
		if(!(pki_conf.get_conf().get_name() == Links[i].get_src().get_name()) || 
			EntityType != pki_conf.get_conf().get_body().get_type())
		{
			//Check corresponding entity
			if(!IsEntityKnown(Links[i].get_src().get_name(), EntityType))
			{
				// The conf of the entity doesn't exist, the user added
				// an unknown entity
				// We remove it from the list
				Links.erase(Links.begin() + i);
				i--;
				continue;
			}
		}

		//Verify its links
		FoundCtr = 0;
		uniqueTypeCtr = 0;
		for(j=0; j<Links[i].get_dsts().size(); j++)
		{
			LinkType = Links[i].get_dsts()[j].get_type();


			if( (EntityType == ENTITY_TYPE_RA && 
				LinkType == ENTITY_TYPE_EE ) ||
				(EntityType == ENTITY_TYPE_EE && 
				LinkType == ENTITY_TYPE_RA) )
			{
				uniqueTypeCtr++;
			}

			// Does the entity exist, and is it allowed ?
			if(!IsEntityKnown(Links[i].get_dsts()[j].get_name(), LinkType) ||
				!IsLinkAllowed(EntityType, LinkType) ||
				Links[i].get_dsts()[j].get_name() == Links[i].get_src().get_name())
			{
				//Delete it
				Links[i].get_dsts().erase(Links[i].get_dsts().begin() + j);
				j--;
				continue;
			}

			// add_LinkToEntity, will add the missing reciprocal links,
			// so we always call it.
			if(!add_LinkToEntity(Links, Links[i], Links[i].get_dsts()[j].get_name()))
			{
				Links[i].get_dsts().erase(Links[i].get_dsts().begin() + j);
				j--;
				continue;
			}

			// Is a repository ?
			if(LinkType == ENTITY_TYPE_REPOSITORY)
			{
				FoundCtr++;
			}
		}

		if(uniqueTypeCtr > 1)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
			ERR_add_error_data(1, _sv("You can only have one link between an EE and an RA"));
			ConfAccessLock.UnlockWrite();
			return false;
		}


		// If we're dealing with a CA or a RA it must be connected to
		// at least one repositoy
		if( (EntityType == ENTITY_TYPE_CA || 
			EntityType == ENTITY_TYPE_RA ||
			EntityType == ENTITY_TYPE_KEY_STORE ||
			EntityType == ENTITY_TYPE_EE ||
			EntityType == ENTITY_TYPE_BACKUP ||
			EntityType == ENTITY_TYPE_PUBLICATION) && 
			!FoundCtr)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_RA_CA_LINK_REP);
			ERR_add_error_data(1, Links[i].get_src().get_name().c_str());
			ConfAccessLock.UnlockWrite();
			return false;
		}
	}

	// Check the flags of the repositories.
	for(i=0; i<Links.size(); i++)
	{
		if(Links[i].get_src().get_type() != ENTITY_TYPE_REPOSITORY)
			continue;

		for(j=0; j<Links[i].get_dsts().size(); j++)
		{

			if(Links[i].get_dsts()[j].get_type() != ENTITY_TYPE_REPOSITORY)
			{
				ASN1_BIT_STRING_set_bit(Links[i].get_dsts()[j].get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED, 0);
				ASN1_BIT_STRING_set_bit(Links[i].get_dsts()[j].get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO, 0);
				continue;
			}
			else
			{
				// Get the full entry for the entity
				currEntry2 = get_EntityLink(Links, Links[i].get_dsts()[j].get_name());
				if(!currEntry2)
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
					ConfAccessLock.UnlockWrite();
					return false;
				}

				// Get its link to me
				currLink2 = get_LinkToEntity(*currEntry2, Links[i].get_src().get_name());
				if(!currLink2)
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
					ConfAccessLock.UnlockWrite();
					return false;
				}

				// Now verify/reset the flags
				if(ASN1_BIT_STRING_get_bit(Links[i].get_dsts()[j].get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO))
				{
					// If I should full shynchro, then it means I have to be firewalled for it
					ASN1_BIT_STRING_set_bit(currLink2->get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED, 1);
					ASN1_BIT_STRING_set_bit(currLink2->get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO, 0);

					ASN1_BIT_STRING_set_bit(Links[i].get_dsts()[j].get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED, 0);
				}
				else if(ASN1_BIT_STRING_get_bit(Links[i].get_dsts()[j].get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED))
				{
					// If its firewalled to me then it should full synchro for me
					ASN1_BIT_STRING_set_bit(currLink2->get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED, 0);
					ASN1_BIT_STRING_set_bit(currLink2->get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO, 1);

					ASN1_BIT_STRING_set_bit(Links[i].get_dsts()[j].get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO, 0);
				}
				else
				{
					// both repositories can access each other
					ASN1_BIT_STRING_set_bit(currLink2->get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED, 0);
					ASN1_BIT_STRING_set_bit(currLink2->get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO, 0);
				}
			}
		}
	}
	
	if(!pki_conf.set_links(Links))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	//Yell to other entities the new repositories list
	if(!UpdateEntityConfs(ENTITY_CONF_UPDATE_LINKS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	if(!WritePersonnalConf())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}
	ConfAccessLock.UnlockWrite();
	return true;
}

bool Entity_PKI::ParseNewConf()
{
	return true;
}

bool Entity_PKI::DeclarePkiAdminCert(const PKI_CERT & cert, const mString & Email)
{
	PkiAdminEntry entry;
	entry.set_serial(cert.GetSerial());
	entry.set_email(Email);

	pki_conf.get_conf().get_acls().get_adminserials().push_back(entry);
	return true;
}

bool Entity_PKI::GetGroups(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_USERS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!ResourceOwnerIsMe(PKI_ENTITIES_GROUPS_RESOURCE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_GROUPS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	response.set_groups(pki_conf.get_conf().get_groups());
	ConfAccessLock.UnlockRead();
	return true;
}

bool Entity_PKI::SetGroups(COMMAND_PARAMETERS)
{
	size_t i, j;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_USERS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!ResourceOwnerIsMe(PKI_ENTITIES_GROUPS_RESOURCE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}


	ConfAccessLock.LockWrite();
	for(i=0; i<pki_conf.get_conf().get_groups().size(); i++)
	{
		// Get the associated group from the request
		for(j=0; j<body.get_groups().size(); j++)
		{
			// We found the group
			if(pki_conf.get_conf().get_groups()[i].get_serial() == body.get_groups()[j].get_serial())
			{
				//Empty it
				pki_conf.get_conf().get_groups()[i].set_usersSerial(body.get_groups()[j].get_usersSerial());
				break;
			}
		}
	}

	if(!UpdateEntityConfs(ENTITY_CONF_UPDATE_GROUPS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	if(!WritePersonnalConf())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}
	ConfAccessLock.UnlockWrite();
	return true;
}

bool Entity_PKI::AddGroup(COMMAND_PARAMETERS)
{
	size_t i;
	unsigned long newSerial;
	UsersGroup newGroup;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_USERS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!ResourceOwnerIsMe(PKI_ENTITIES_GROUPS_RESOURCE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	newSerial = 0;
	ConfAccessLock.LockWrite();
	// Get the highest serial, and make sure the group name is unique
	for(i=0; i<pki_conf.get_conf().get_groups().size(); i++)
	{
		if(pki_conf.get_conf().get_groups()[i].get_name() ==  body.get_groupName())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_GROUP_ALREADY_EXISTS);
			ConfAccessLock.UnlockWrite();
			return false;
		}
		if(pki_conf.get_conf().get_groups()[i].get_serial() > newSerial)
			newSerial = pki_conf.get_conf().get_groups()[i].get_serial();
	}
	newSerial++;

	newGroup.set_serial(newSerial);
	newGroup.set_name(body.get_groupName());

	pki_conf.get_conf().get_groups().push_back(newGroup);

	if(!UpdateEntityConfs(ENTITY_CONF_UPDATE_GROUPS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	if(!WritePersonnalConf())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	ConfAccessLock.UnlockWrite();
	return true;
}

bool Entity_PKI::DelGroup(COMMAND_PARAMETERS)
{
	size_t i;
	size_t j;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_USERS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!ResourceOwnerIsMe(PKI_ENTITIES_GROUPS_RESOURCE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockWrite();
	for(i=0; i<pki_conf.get_conf().get_groups().size(); i++)
	{
		//Is it the seeked group ?
		if(pki_conf.get_conf().get_groups()[i].get_serial() == body.get_groupId())
		{
			break;
		}
	}
	if(i == pki_conf.get_conf().get_groups().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_GROUP_UNKNOWN);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	// We found the group, now we need to make sure this group
	// isn't used in an RA configuration for the auto ldap synch
	for(j=0; j<pki_conf.get_confs().size(); j++)
	{
		// Is it a RA
		if(pki_conf.get_confs()[j].get_conf().get_body().get_type() != ENTITY_TYPE_RA)
			continue;

		// Is the LDAP auto synch activated
		if(!ASN1_BIT_STRING_get_bit(pki_conf.get_confs()[j].get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_ALLOW_LDAP_AUTO_SYNCH))
			continue;

		// Is the group the one we delete
		if(pki_conf.get_conf().get_groups()[i].get_serial() == pki_conf.get_confs()[j].get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapProfilesGroup())
		{
			// ERROR !
			NEWPKIerr(PKI_ERROR_TXT, ERROR_GROUP_USED_RA_LDAP_SYNCH);
			ERR_add_error_data(1, pki_conf.get_confs()[j].get_conf().get_name().c_str());
			ConfAccessLock.UnlockWrite();
			return false;
		}

	}

	pki_conf.get_conf().get_groups().erase(pki_conf.get_conf().get_groups().begin() + i);

	if(!UpdateEntityConfs(ENTITY_CONF_UPDATE_GROUPS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}

	if(!WritePersonnalConf())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}
	ConfAccessLock.UnlockWrite();
	return true;
}

bool Entity_PKI::RenameGroup(COMMAND_PARAMETERS)
{
	size_t i;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_MANAGE_USERS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!ResourceOwnerIsMe(PKI_ENTITIES_GROUPS_RESOURCE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockWrite();
	for(i=0; i<pki_conf.get_conf().get_groups().size(); i++)
	{
		//Is it the seeked group ?
		if(pki_conf.get_conf().get_groups()[i].get_serial() == body.get_renameGroup().get_id())
		{
			pki_conf.get_conf().get_groups()[i].set_name(body.get_renameGroup().get_name());

			if(!UpdateEntityConfs(ENTITY_CONF_UPDATE_GROUPS))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}

			if(!WritePersonnalConf())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockWrite();
				return false;
			}
			ConfAccessLock.UnlockWrite();
			return true;
		}
		else
		{
			// A group already has this name ?
			if(pki_conf.get_conf().get_groups()[i].get_name() == body.get_renameGroup().get_name())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_GROUP_ALREADY_EXISTS);
				ConfAccessLock.UnlockWrite();
				return false;
			}
		}
	}
	ConfAccessLock.UnlockWrite();

	NEWPKIerr(PKI_ERROR_TXT, ERROR_GROUP_UNKNOWN);
	return false;
}

bool Entity_PKI::EXPORTED_PKI_CONF_get_conf(const X509 * cert, EntityConfCrypted & conf)
{
	size_t i;
	for(i=0; i<pki_conf.get_cryptConfs().get_confs().get_allConfs().size(); i++)
	{

		if(cert->cert_info->key->public_key->length ==
			pki_conf.get_cryptConfs().get_confs().get_allConfs()[i].get_recipient()->public_key->length &&
			memcmp(cert->cert_info->key->public_key->data,
			pki_conf.get_cryptConfs().get_confs().get_allConfs()[i].get_recipient()->public_key->data, 
			pki_conf.get_cryptConfs().get_confs().get_allConfs()[i].get_recipient()->public_key->length) == 0)
		{
			if(! (conf = pki_conf.get_cryptConfs().get_confs().get_allConfs()[i]) )
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			return true;
		}
	}
	NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
	return false;
}

StoredEntityConf & Entity_PKI::STORED_ENTITY_CONF_get(mVector< StoredEntityConf > & entities, const PKI_CERT & cert)
{
	size_t i;

	for(i=0; i < entities.size(); i++)
	{
		if(entities[i].get_certificate() == cert)
			return entities[i];
	}

	return StoredEntityConf::EmptyInstance;
}

bool Entity_PKI::CreatePkiUser(const CreatePkiUserRequest & request, CreatePkiUserResponse & response, unsigned long & serial)
{
	PKI_CERT certUser;
	PKI_RSA privKey;
	const X509_PUBKEY * pubkeyUser;
	HashTable_String ParentCerts;
	int What2Update = ENTITY_CONF_UPDATE_ACL;

	if(!request || 
		!request.get_cn().size() ||
		!request.get_email().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	switch(request.get_ukey().get_type())
	{
		case PKI_USER_KEY_TYPE_SOFTKEY:
			if(!request.get_ukey().get_softkey().get_password().size())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
				return false;
			}
			if(!privKey.GenerateKey(request.get_ukey().get_softkey().get_keylen(), m_Engine))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			pubkeyUser = privKey.GetPublicKey();
			if(!pubkeyUser)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case PKI_USER_KEY_TYPE_PUBKEY:
			pubkeyUser = request.get_ukey().get_pubkey();
			if(!pubkeyUser)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
			break;
		default:
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
	}

	// Generate the certificate
	if(!InternalCA_SignForceDn(INTERNAL_CA_TYPE_USER, pubkeyUser, request.get_cn(), request.get_email(), certUser))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	serial = certUser.GetSerial();

	// Create the response
	switch(request.get_ukey().get_type())
	{
		case PKI_USER_KEY_TYPE_SOFTKEY:
			// We must generate a PKCS#12
			if(!response.set_type(PKI_USER_KEY_TYPE_SOFTKEY))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
				RevokePkiUser(serial);
				return false;
			}
			if(!ParentCerts.Add(RootCa->get_CaCert()->GetStringName(), RootCa->get_CaCert()->GetCertPEM().c_str()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
				RevokePkiUser(serial);
				return false;
			}
			if(!ParentCerts.Add(UsersCa->get_CaCert()->GetStringName(), UsersCa->get_CaCert()->GetCertPEM().c_str()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
				RevokePkiUser(serial);
				return false;
			}

			response.get_p12().SetEndUserCert(certUser);
			response.get_p12().SetEndUserKey(privKey);
			response.get_p12().SetParentCerts(ParentCerts);


			if(!response.get_p12().Generate(request.get_ukey().get_softkey().get_password().c_str()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				RevokePkiUser(serial);
				return false;
			}

			break;
		case PKI_USER_KEY_TYPE_PUBKEY:
			// We must generate a PKCS#7
			if(!response.set_type(PKI_USER_KEY_TYPE_PUBKEY))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
				RevokePkiUser(serial);
				return false;
			}
			if(!response.get_p7b().AddCert(RootCa->get_CaCert()->GetX509()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				RevokePkiUser(serial);
				return false;
			}
			if(!response.get_p7b().AddCert(UsersCa->get_CaCert()->GetX509()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				RevokePkiUser(serial);
				return false;
			}
			if(!response.get_p7b().AddCert(certUser.GetX509()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				RevokePkiUser(serial);
				return false;
			}
			if(!response.get_p7b().Generate())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				RevokePkiUser(serial);
				return false;
			}
			break;
	}


	ConfAccessLock.LockWrite();
	if(request.get_pkiadmin())
	{
		if(!DeclarePkiAdminCert(certUser, request.get_email()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			RevokePkiUser(serial);
			ConfAccessLock.UnlockWrite();
			return false;
		}
		//Update my own ACL handler
		if(!AclValidator.SetACL(pki_conf.get_conf().get_acls()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			RevokePkiUser(serial);
			ConfAccessLock.UnlockWrite();
			return false;
		}

		//Yell to other entities the new admin
		What2Update |= ENTITY_CONF_UPDATE_ADMINS;
	}
	ConfAccessLock.UnlockWrite();


	//Yell to other entities the new ACL
	if(!UpdateEntityConfs(What2Update))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		RevokePkiUser(serial);
		return false;
	}

	if(!WritePersonnalConf())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		RevokePkiUser(serial);
		return false;
	}

	return true;
}

bool Entity_PKI::RevokePkiUser(unsigned long serial)
{
	size_t i;
	int What2Update = ENTITY_CONF_UPDATE_ACL | ENTITY_CONF_UPDATE_CRL | ENTITY_CONF_UPDATE_GROUPS;

	if(AclValidator.IsPkiAdministrator(serial))
	{
		//We remove it from the list
		ConfAccessLock.LockWrite();
		for(i=0; i < pki_conf.get_conf().get_acls().get_adminserials().size() ; i++)
		{
			if(pki_conf.get_conf().get_acls().get_adminserials()[i].get_serial() == serial)
			{
				pki_conf.get_conf().get_acls().get_adminserials().erase(pki_conf.get_conf().get_acls().get_adminserials().begin() + i);
				break;
			}
		}
		ConfAccessLock.UnlockWrite();
		What2Update |= ENTITY_CONF_UPDATE_ADMINS;
	}

	if(!InternalCA_Revoke(INTERNAL_CA_TYPE_USER, serial))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Yell to other entities the new ACL and the new CRL
	if(!UpdateEntityConfs(What2Update))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	//Remove user from the ACLs
	if(!WritePersonnalConf())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity_PKI::PrepareConfToWrite()
{
	EXPORTED_PKI_CONF_BODY * lBody = NULL;
	pki_conf.get_cryptConfs().get_confs().set_version(pki_conf.get_cryptConfs().get_confs().get_version() + 1);

	if(!pki_conf.get_cryptConfs().get_confs().give_Datas(&lBody))
	{
		if(lBody)
			ASN1_item_free((ASN1_VALUE*)lBody, ExportedPkiConfBody::get_ASN1_ITEM());
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Signing the global conf
	//Ok, we have crypted and sign every conf
	//let's sign the whole thing with the PKI Entity Key
	if(ASN1_item_sign(ExportedPkiConfBody::get_ASN1_ITEM(), pki_conf.get_cryptConfs().get_sig()->algor, NULL, pki_conf.get_cryptConfs().get_sig()->digest, (char*)lBody, (EVP_PKEY*)m_EntityKey.GetRsaKey(), EVP_sha1()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)lBody, ExportedPkiConfBody::get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)lBody, ExportedPkiConfBody::get_ASN1_ITEM());

	if(!pki_conf.get_conf().get_body().get_pkiConf().PKI_CONF_PTR.get_offline())
	{
		if(!m_Jobs.SetPkiConf(pki_conf.get_cryptConfs()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	return true;
}

void Entity_PKI::PrintInfo(FILE *out)
{

}

bool Entity_PKI::SynchronizeEeOnRa(EntityConf & eeConf, const EntityConf & raConf)
{
	size_t linkIndex, i, j;


	eeConf.get_body().get_eeConf().EE_CONF_PTR.set_dnSpecs(
		raConf.get_body().get_raConf().RA_CONF_PTR.get_dnspecs());
	eeConf.get_body().get_eeConf().EE_CONF_PTR.set_policies(
		raConf.get_body().get_raConf().RA_CONF_PTR.get_policies());

	eeConf.get_body().get_eeConf().EE_CONF_PTR.set_minkeylen(raConf.get_body().get_raConf().RA_CONF_PTR.get_minkeylen());
	eeConf.get_body().get_eeConf().EE_CONF_PTR.set_minpwdlen(raConf.get_body().get_raConf().RA_CONF_PTR.get_minpwdlen());

	if(ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)raConf.get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_ALLOW_SERVER_SIDE_KEY_GEN))
		ASN1_BIT_STRING_set_bit(eeConf.get_body().get_eeConf().EE_CONF_PTR.get_flags(), EE_ALLOW_SERVER_SIDE_KEY_GEN, 1);
	else
		ASN1_BIT_STRING_set_bit(eeConf.get_body().get_eeConf().EE_CONF_PTR.get_flags(), EE_ALLOW_SERVER_SIDE_KEY_GEN, 0);

	if(ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)raConf.get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_ALLOW_PKCS10_IMPORT))
		ASN1_BIT_STRING_set_bit(eeConf.get_body().get_eeConf().EE_CONF_PTR.get_flags(), EE_ALLOW_PKCS10_IMPORT, 1);
	else
		ASN1_BIT_STRING_set_bit(eeConf.get_body().get_eeConf().EE_CONF_PTR.get_flags(), EE_ALLOW_PKCS10_IMPORT, 0);

	if(ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)raConf.get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_ALLOW_OP_SC_KEY_GEN))
		ASN1_BIT_STRING_set_bit(eeConf.get_body().get_eeConf().EE_CONF_PTR.get_flags(), EE_ALLOW_SC_KEY_GEN, 1);
	else
		ASN1_BIT_STRING_set_bit(eeConf.get_body().get_eeConf().EE_CONF_PTR.get_flags(), EE_ALLOW_SC_KEY_GEN, 0);


	eeConf.get_body().get_eeConf().EE_CONF_PTR.get_cas().clear();

	// We get the list of CAs from the attached RA
	for(linkIndex=0; linkIndex < pki_conf.get_links().size(); linkIndex++)
	{
		if(pki_conf.get_links()[linkIndex].get_src().get_name() == raConf.get_name())
		{
			break;
		}
	}
	// We didn't find the RA in the links list
	if(linkIndex != pki_conf.get_links().size())
	{
		// Seek the CAs links
		for(i=0; i < pki_conf.get_links()[linkIndex].get_dsts().size(); i++)
		{
			if(pki_conf.get_links()[linkIndex].get_dsts()[i].get_type() != ENTITY_TYPE_CA)
				continue;

			//Seek corresponding CA info
			for(j=0; j < pki_conf.get_confs().size(); j++)
			{
				//Is it the seeked one ?
				if(!(pki_conf.get_confs()[j].get_conf().get_name() == pki_conf.get_links()[linkIndex].get_dsts()[i].get_name()) ||
					pki_conf.get_confs()[j].get_conf().get_body().get_type() != ENTITY_TYPE_CA)
					continue;

				eeConf.get_body().get_eeConf().EE_CONF_PTR.get_cas().push_back(pki_conf.get_confs()[j].get_conf().get_name());
				break;
			}
		}
	}

	return true;
}
