/** 
 * @file callstack_pool.c
 * Contains return address functions
 *
 * Functions that handle the callstacks are here.
 *
 * Copyright (C) 2001 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details. 
 */
#include <lib/callstack_pool.h>
#include <lib/callstack.h>
#include <lib/stack.h>
#include <lib/table.h>
#include <lib/util.h>
#include <lib/portability.h>
#include <lib/public.h>
#include <lib/njamd.h>
#include <unistd.h>
#ifdef HAVE_EXECINFO_H
# include <execinfo.h>
#endif
#include <string.h>
#include <stdio.h>

static nj_addr_t return_address(int);
static int next_address_valid(nj_addr_t, nj_addr_t);
static int callstack_fill(nj_addr_t *, int, int);
static void callstack_print(nj_addr_t *, int);

/**
 * Intialize a callstack pool
 *
 * Starts the callstack pool up before we know the prefs 
 * 
 * @param cs_pool The callstack pool to init
 */
void __nj_callstack_pool_bootstrap_init(struct nj_callstack_pool *cs_pool)
{
	/* We need a NEW fd for this..
	 * Technically we could do this in the user_init since we don't even use the callstack
	 * until then, but screw it. We want the naming the same as the heap file, 
	 * so we'd have to do the same ifdef. 
	 */
#ifdef HAVE_WORKING_ERRNO 
	snprintf(cs_pool->file, sizeof(cs_pool->file), "./njamd-%d-callstack", 
			getpid());
#else
	strncpy(cs_pool->file, "./njamd-callstack", sizeof(cs_pool->file));
#endif

	if(__nj_table_bootstrap_init(&cs_pool->table, cs_pool->file, 
				NJ_ALLOCS_IN_ADDRESS_SPACE*NJ_CALLSTACK_AVG_PER_ALLOC*sizeof(nj_addr_t), 1, 1))
		__nj_critical_error(__FUNCTION__);

	__nj_stack_bootstrap_init(&cs_pool->free_list);

	/* still not ready yet */
	cs_pool->state = NJ_CALLSTACK_STATE_NOT_READY;

}

/**
 * Initialize the pool for the user's prefs
 *
 * @param cs_pool The pool
 * @param prefs The prefs
 */ 
void __nj_callstack_pool_user_init(struct nj_callstack_pool *cs_pool, 
		struct nj_prefs *prefs)
{
	if(prefs->stat.store_retaddrs == FALSE)
	{
		__nj_callstack_pool_fini(cs_pool);
		cs_pool->state = NJ_CALLSTACK_STATE_DISABLED;
		return;
	}

	__nj_table_user_init(&cs_pool->table, prefs);
	__nj_stack_user_init(&cs_pool->free_list);
	

	if(prefs->stat.callstack_depth)
	{
		if(prefs->stat.callstack_depth > NJ_CALLSTACK_MAX_LEN)
		{
			__nj_eprintf("Callstack length %d must be <= %d, sorry.\n", prefs->stat.callstack_depth, NJ_CALLSTACK_MAX_LEN);
			prefs->stat.callstack_depth = NJ_CALLSTACK_MAX_LEN;
		}
		else if(prefs->stat.callstack_depth < NJ_CALLSTACK_MIN_LEN)
		{
			__nj_eprintf("Callstack length %d must be >= %d\n", prefs->stat.callstack_depth, NJ_CALLSTACK_MIN_LEN);
			prefs->stat.callstack_depth = NJ_CALLSTACK_MIN_LEN;
		}
		cs_pool->max_len = prefs->stat.callstack_depth;
		cs_pool->fixed_len = TRUE;
	}
	else
	{
		cs_pool->max_len = NJ_CALLSTACK_MAX_LEN;
		cs_pool->fixed_len = FALSE;
	}

	/* if the max length is less than the average length, then we don't need 
	 * to base our alloc on a guess of the average. */
	if(2*cs_pool->max_len < NJ_CALLSTACK_AVG_PER_ALLOC)
		__nj_table_trim(&cs_pool->table, NJ_ALLOCS_IN_ADDRESS_SPACE*2*cs_pool->max_len*sizeof(nj_addr_t));

	cs_pool->state = NJ_CALLSTACK_STATE_USER_READY;
	
}

/**
 * Close the pool.
 *
 * @param cs_pool The pool to close
 */ 
void __nj_callstack_pool_fini(struct nj_callstack_pool *cs_pool)
{
	cs_pool->state = NJ_CALLSTACK_STATE_DESTRUCT;
	__nj_table_fini(&cs_pool->table);
	__nj_stack_fini(&cs_pool->free_list);
}

/**
 * Gets a calltrace of specified length.
 * 
 * The actual addresses saved depend on a few flags, as well as the state of
 * the system, and the addresses next_address_valid() deems are terminal. If the
 * user doesn't want to trace libs (default), next_address_valid() is called
 * with #__nj_sbrk0 as the boundary, until it says that the return address is
 * below sbrk0. Otherwise, all addresses are stored until either we fill rets,
 * or next_address_valid() finds something else wrong with them.
 * 
 * @returns An index into the return address table in the form of a nj_callstack struct
 */
struct nj_callstack __nj_callstack_pool_request_index(struct nj_callstack_pool *cs_pool,
		struct nj_dynamic_prefs prefs)
{
	struct nj_stack_item *item = NULL;
	struct nj_callstack callstack;
	nj_addr_t *rets;
	int len;
	
	callstack.len = 0;

#ifdef HAVE_DEADLY_RET_ADDR
	callstack.index = NJ_CALLSTACK_INDEX_NONE;
	return callstack;
#else

	/* Before main not only do we not care, but we haven't initialized the table
	 * yet */
	switch(cs_pool->state)
	{
		case NJ_CALLSTACK_STATE_NOT_READY:
			callstack.index = NJ_CALLSTACK_INDEX_IGNORE;
			return callstack;
		case NJ_CALLSTACK_STATE_DISABLED:
			callstack.index = NJ_CALLSTACK_INDEX_NONE;
			return callstack;
		case NJ_CALLSTACK_STATE_DESTRUCT:
			callstack.index = NJ_CALLSTACK_INDEX_DESTRUCTOR;
			return callstack;
	}
	
	/* Assume that it's the right length for now. We're only gonna worry 
	 * about fixed length callstacks if they wanna free them. */
	if((item = __nj_stack_pop(&cs_pool->free_list)))
	{
		callstack = NJ_UBER_CAST(item->data, struct nj_callstack);
		/* rets = NJ_TABLE_GET_INDEX(callstack_pool, u_long, NJ_CALLSTACK_GET_INDEX(ind));*/
		/* Shortcut: item IS the old callstack, hence the NJ_CALLSTACK_MIN.. */
		rets = (nj_addr_t *)item;
	}
	else
	{
		rets = (nj_addr_t *)__nj_table_request_top(&cs_pool->table, cs_pool->max_len*sizeof(nj_addr_t));
		/* The index of the table is returned inside the data */
		callstack.index = *(u_int *)rets/sizeof(nj_addr_t);
	}

	len = callstack_fill(rets, cs_pool->max_len, prefs.trace_libs);

	callstack.len = len;

	/* If NJ_NO_FREE_INFO is set, we have to use a fixed length for callstacks,
	 * otherwise it's just a big pain in the ass to handle all the different 
	 * size retaddr blocks */
	if(cs_pool->fixed_len)
		len = cs_pool->max_len;

	/* If item is null, it came from the table */
	if(item == NULL)
		__nj_table_release_top(&cs_pool->table, len*sizeof(nj_addr_t));
	
	TR;
	return callstack;
#endif /* not deadly return_address */
}

/**
 * Renew an index without going through the cache
 * 
 * @NOTES If this function is called, we know the callstack length is fixed 
 * to cs_pool->max_len
 */ 
void __nj_callstack_pool_renew_index(struct nj_callstack_pool *cs_pool, 
		struct nj_callstack callstack, struct nj_dynamic_prefs prefs)
{
	/* An array of return addresses */
	nj_addr_t *rets;

	/* Does not need a lock unless there is a user race to free the same 
	 * segment, which we checked for */
	rets = NJ_CALLSTACK_POOL_INDEX_TO_ADDR_ARRAY(*cs_pool, callstack.index);

	callstack.len = callstack_fill(rets, cs_pool->max_len, prefs.trace_libs);

}

/**
 * Return an index back into the callstack pool
 * 
 * Called only if the user has NO_FREE_INFO set.
 * 
 * @param cs_pool The callstack pool
 * @param callstack The index to release
 *
 */
void __nj_callstack_pool_release_index(struct nj_callstack_pool *cs_pool, 
		struct nj_callstack callstack)
{
	struct nj_stack_item *item;

	switch(callstack.index)
	{
		case NJ_CALLSTACK_INDEX_NONE:
		case NJ_CALLSTACK_INDEX_DESTRUCTOR:
		case NJ_CALLSTACK_INDEX_IGNORE:
		case NJ_CALLSTACK_INDEX_NOTFREE:
			return;
	}

	/* No need for a mutex here, because we won't ever race to release the *
	   same segment. Well we will due to user error, but we check for that
	   earlier up */
    /* and YES, we do want the type of nj_addr_t there.. remember what this is a table of */
	item = (struct nj_stack_item *)NJ_CALLSTACK_POOL_INDEX_TO_ADDR_ARRAY(*cs_pool, callstack.index);

	/* The callstack is gauranteed to fit into a nj_generic_t type by autoconf */
	item->data = NJ_UBER_CAST(callstack, nj_generic_t);

	__nj_stack_push(&cs_pool->free_list, item);
}

/**
 * Prints out the stack trace. Resoloves symbols if possible.
 *
 * @param addr_array The call stack trace
 * @param depth The size of the trace
 */
void __nj_callstack_pool_print_index(struct nj_callstack_pool *cs_pool, 
		struct nj_callstack callstack)
{
	switch(callstack.index)
	{
		case NJ_CALLSTACK_INDEX_NONE:
			__nj_eprintf("\tCall stack not saved\n");
			return;
		case NJ_CALLSTACK_INDEX_DESTRUCTOR:
			__nj_eprintf("\tcalled from a destructor during program exit\n");
			return;
		case NJ_CALLSTACK_INDEX_IGNORE:
			return;
	}
	
	callstack_print(NJ_CALLSTACK_POOL_INDEX_TO_ADDR_ARRAY(*cs_pool, callstack.index),
			callstack.len);
}

/**@{ @name Callstack Functions */
/** 
 * Attempts to detect if our execution path has flowed through
 * main yet. Again, we use the sbrk0 trick to determine if we have entered
 * statically compiled code. The problem is we don't know when to stop the
 * backtrace if we AREN'T past main.  Right now we stop when we find a
 * function that is below this library yet above the user segment, the
 * assumpion being that LD_PRELOADED and/or explicitly linked libs are the
 * lowest in the address space. This is about where __libc_start_main lies..
 * very hackish to say the least. ANy other ideas?
 *
 * This check must be done in the first place because most systems don't set
 * up the environment until just before main is called, so we need to check
 * for entrence into main before calling __nj_secondary_init()
 *
 * @returns True if we have crossed into statically linked code (ie most
 * likely have entered main, and have an environment)
 */
int __nj_callstack_crossed_main()
{
#ifndef HAVE_DEADLY_RET_ADDR
    int i=1; /* Start at the return address of the caller of this function */
    nj_addr_t raddr;

    TR;
    
    for(; i<100; i++)
    {
		raddr = return_address(i);

		if(__nj_sbrk0 < raddr && raddr < (nj_addr_t)__nj_public_end)
            return FALSE;
        if(raddr < __nj_sbrk0)
            return TRUE;
    }
    TR;
    return FALSE;
#else
    return TRUE; /* __bultin_return_address is deadly. Punt and cross fingers */
#endif
}

  


/**
 * Dumps the callstack for the current frame using temporary storage.
 *
 * @param start The number of stack frames above the current to start.
 */
void __nj_callstack_dump(void)
{
	nj_addr_t rets[NJ_CALLSTACK_MAX_LEN];
	int len;

	/** @FIXME use output object */
	if(__NJAMD__.state <= NJ_STATE_USER_READY)
	{
		__nj_eprintf("\tcalled from a system function before main\n");
		return;
	}

	TR;

	if(__NJAMD__.state == NJ_STATE_DESTRUCT)
	{
		__nj_eprintf("\tcalled from a destructor during program exit\n");
		return;
	}

	/** @FIXME Ich.. I don't like using this global var.. */
	if(__NJAMD__.prefs.stat.store_retaddrs)
	{
		__nj_eprintf("\tReturn address information turned off\n");
		return;
	}

	len = callstack_fill(rets, NJ_CALLSTACK_MAX_LEN, __NJAMD__.prefs.dyn.trace_libs);

	callstack_print(rets, len);
}
/*@}*/

/**@{ @name Private Callstack functions */
/**
 * Print the callstack
 *
 * @param addr_array The callstack in array form
 * @param depth The length of the array.
 * 
 * @FIXME Use the output object and get rid of that global __NJAMD__ ref
 */ 
static void callstack_print(nj_addr_t *addr_array, int depth)
{
	int i;

	for(i=0; i < depth; i++)
	{
		if(addr_array[i] == 0)
			return;
		/** @FIXME Backtrace symbols fd is broken and useless now.. */
#ifdef HAVE_BACKTRACE_SYMBOLS_FD__BROKEN
		__nj_eprintf("\tcalled from ");
		backtrace_symbols_fd((void **)&addr_array[i], 1, __NJAMD__.output.fd);
#else
		__nj_eprintf("\tcalled from 0x%lx\n", addr_array[i]);
#endif
		
	}
}

/* We must check the frame address as a hack... Sometimes high optimizations  
 * kill the frame pointer even if you don't specify -fomit-frame-pointer..
 * We wouldn't have to do this if the gcc people just followed their
 * documentation and returned NULL instead of segfaulting. I've reported this
 * bug and got no replies. *shrug*
 */
#if defined(USE_FRAME_ADDR_HACK)
# define RA(x)	case x: return (nj_addr_t)(__builtin_frame_address(x) ? __builtin_return_address(x) : 0)
#elif !defined(HAVE_DEADLY_RET_ADDR)
# define RA(x)	case x: return (nj_addr_t)__builtin_return_address(x)
#else /* __builtin_return_address is not only uselesss, its deadly */
# define RA(x)  case x: return NULL
#endif
/**
 * Gets a return address in the stack trace of the calling function.
 *
 * @param frame The number of function calls to traverse up the stack.
 * @returns A return address (aka calling address)
 */ 
static nj_addr_t return_address(int frame)
{
	TR;
	switch(frame+1)
	{
       RA(1);RA(2);RA(3);RA(4);RA(5);RA(6);RA(7);RA(8);RA(9);
	   RA(10);RA(11);RA(12);RA(13);RA(14);RA(15);RA(16);RA(17);RA(18);RA(19);
	   RA(20);RA(21);RA(22);RA(23);RA(24);RA(25);RA(26);RA(27);RA(28);RA(29);
	   RA(30);RA(31);RA(32);RA(33);RA(34);RA(35);RA(36);RA(37);RA(38);RA(39);
	   RA(40);RA(41);RA(42);RA(43);RA(44);RA(45);RA(46);RA(47);RA(48);RA(49);
	   RA(50);RA(51);RA(52);RA(53);RA(54);RA(55);RA(56);RA(57);RA(58);RA(59);
	   RA(60);RA(61);RA(62);RA(63);RA(64);RA(65);RA(66);RA(67);RA(68);RA(69);
	   RA(70);RA(71);RA(72);RA(73);RA(74);RA(75);RA(76);RA(77);RA(78);RA(79);
	   RA(80);RA(81);RA(82);RA(83);RA(84);RA(85);RA(86);RA(87);RA(88);RA(89);
	   RA(90);RA(91);RA(92);RA(93);RA(94);RA(95);RA(96);RA(97);RA(98);RA(99);
	   default:
	       return 0;
	}

}

/**
 * Checks the given return address to see if the next one would possibly run
 * us off the call stack. There are several tricks this function does. First
 * off, the heap starts right above statically linked code in ELF, so the 
 * first sbrk in a program can be used to find out which return addresses are
 * in static user code.
 *
 * Also, we use the stubs in public.c to find out when to stop
 * backtracing once we are in the destructors (ie main would no longer be our
 * refrence point).
 * 
 * @param cmp The address in question (usually passed to free)
 * @param lower_bound Set to sbrk(0) if we want to stop at main, 0
 * 					  otherwise (ie if we know we're in the destructors)
 * @returns True if the next return address is not likely to knock us off the
 * call stack.
 */
static int next_address_valid(nj_addr_t cmp, nj_addr_t lower_bound)
{
	if(!cmp)
		return FALSE;
#ifdef HAVE_SANE_DLOPEN
# ifdef _THREAD_SAFE
	if(__NJAMD__.active_threads)
	{
		/* only continue recursing if we're outside of main AND we're not inside
		 * the NJAMD pthread launching fucntion (which is the end of the call
		 * stack for pthreads */
		if(cmp > lower_bound
				&& !(((nj_addr_t)__nj_threads_launch < cmp && cmp < (nj_addr_t)__nj_threads_launch_end)
					|| ((nj_addr_t)__nj_pthread_exit_start < cmp && cmp < (nj_addr_t)__nj_pthread_exit_end)
					|| ((nj_addr_t)__nj_exit_start < cmp && cmp < (nj_addr_t)__nj_exit_end)))
			return TRUE;
	}
	else
# endif
	{
		/* Only continue recursing if we're outside of main and we're not in 
		 * exit */
		if(cmp > lower_bound
				&& !((nj_addr_t)__nj_exit_start < cmp && cmp < (nj_addr_t)__nj_exit_end))
			return TRUE;
	}
#else
	if(cmp > lower_bound)
		return TRUE;
#endif

	return FALSE;
}

/**
 * Fill a callstack array with a callstack
 * 
 * @param rets The return addresses
 * @param depth The maximum length of the array
 * 
 * @returns The length of the callstack filled
 */
static int callstack_fill(nj_addr_t *rets, int depth, int trace_libs)
{
	int i, k;
	int start = 2;

	/* Find the first return address outside of NJAMD */
	for(rets[0] = return_address(start); 
			rets[0] < (nj_addr_t)__nj_public_init
			|| rets[0] > (nj_addr_t)__nj_public_end; start++)
	{
		rets[0] = return_address(start);
	}
	
	rets[0] = return_address(start);

#ifdef DEBUG
	__nj_eprintf(__FUNCTION__"/DEBUG: start=%d\n", start);
#endif
	

	TR;
	
	if(trace_libs)
	{
		/* Only record return addresses where the next one is valid (as in
		 * from library space) */
		for(i = 1; i < depth; i++)
		{
#ifdef HAVE_WORKING_RET_ADDR
			if(next_address_valid(rets[i-1], 0))
#else
			if(next_address_valid(rets[i-1], __nj_sbrk0))
#endif
				rets[i] = return_address(i+start);
			else
				break;
		}
	}
	else
	{
		/* Find the first address below sbrk0 (ie static code) and trace from 
		 * there. This isn't inside the ifdef because it stops at sbrk0. */
		for(k=start+1; next_address_valid(rets[0], __nj_sbrk0); k++)
			rets[0] = return_address(k);

#ifdef HAVE_WORKING_RET_ADDR
		/* fill in all return addresses with valid ones from usercode */
		for(i = 1; i < depth; i++)
		{
			if(next_address_valid(rets[i-1], 0))
			{
				rets[i] = return_address(k);
				k++;
			}
			else
				break;
		}
#else
		/* Only one ret address */
		i = 1;
#endif
	}

	return i;
}
/*@}*/
// vim:ts=4
