/** 
 * @file prefs.c initialization routines.
 *
 * Copyright (C) 2001 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details.
 */
#include <lib/prefs.h>
#include <lib/memory_pool.h>
#include <lib/signals.h>
#include <lib/allocator.h>
#include <lib/portability.h>
#include <lib/output.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

/**
 * Sets up default prefs
 * @param prefs The prefs objet
 */
void __nj_prefs_bootstrap_init(struct nj_prefs *prefs)
{

	prefs->dyn.trace_libs = FALSE;
	prefs->dyn.info_on_free = TRUE;
	
	prefs->dyn.alloc_type = NJ_ALLOCATOR_BOOTSTRAP_ALLOC_TYPE;
	prefs->dyn.free_type = NJ_ALLOCATOR_BOOTSTRAP_FREE_TYPE;

	prefs->dyn.align = NJ_ALIGNMENT;
	prefs->stat.store_retaddrs = TRUE;

	prefs->stat.dump_stats = FALSE;
	prefs->stat.dump_leaks = FALSE;
	prefs->stat.persist = FALSE;
	prefs->stat.allow_read = FALSE;
	prefs->stat.allow_free_0 = FALSE;
	prefs->stat.allow_malloc_0 = FALSE;
	prefs->stat.core_dump_type = NJ_DUMP_SOFT_CORE;
	prefs->stat.mutable_alloc = FALSE;
	prefs->stat.callstack_depth = 0; /* No fixed size */

	/** @FIXME If we use a file for prefs init, I think we don't need 
	 * the secondary init stage at ALL! Actually, wait, maybe pthreads */
}

/**
 * Load user prefs in
 */
void __nj_prefs_user_init(struct nj_prefs *prefs)
{
	/** TODO: Have other ways of reading these in, and establish a precidence on existance */
	__nj_prefs_load_from_env(prefs);
}

/**
 * Does nothing for now..
 *
 * Perhaps save user prefs in the future?
 */
void __nj_prefs_fini(struct nj_prefs *prefs)
{
	return;
}

/** 
 * Set the dynamic prefs to the new instance in a threadsafe way
 *
 * @param Prefs The global prefs
 * @param new_prefs The new prefs (is only an int wide, thats why we use the whole struct)
 */
void __nj_prefs_set(struct nj_prefs *prefs, struct nj_dynamic_prefs new_prefs)
{
#ifdef _THREAD_SAFE
	pthread_mutex_lock(&prefs->set_lock);
#endif

	prefs->dyn = new_prefs;
	
#ifdef _THREAD_SAFE
	pthread_mutex_unlock(&prefs->set_lock);
#endif
}

/**
 * Gets the dynamic prefs in a threadsafe way
 *
 * @param prefs The prefs object
 * @returns A dynamic prefs object
 */
struct nj_dynamic_prefs __nj_prefs_get(struct nj_prefs *prefs)
{
	struct nj_dynamic_prefs dyn_prefs;
#ifdef _THREAD_SAFE
	pthread_mutex_lock(&prefs->set_lock);
#endif
	
	dyn_prefs = prefs->dyn;
	
#ifdef _THREAD_SAFE
	pthread_mutex_unlock(&prefs->set_lock);
#endif

	return dyn_prefs;
}

/**
 * Load the prefs from the environment 
 *
 * Backwards compatible with old NJAMD versions
 * @param prefs The old prefs
 */
void __nj_prefs_load_from_env(struct nj_prefs *prefs)
{
	struct rlimit rcore = {4<<NJ_SHIFT_MEGABYTE, 4<<NJ_SHIFT_MEGABYTE};
	char *env_tmp;
	/* Was set to segv in before.. But we don't want that as the default, 
	 * cause NONE is MUCH faster */

	prefs->dyn.free_type = NJ_CHK_FREE_NONE;

	if((env_tmp = getenv("NJAMD_PROT")) != NULL)
	{
		if(strncmp(env_tmp, "over",4) == 0)
			prefs->dyn.alloc_type = NJ_PROT_OVER;
		else if(strncmp(env_tmp, "under", 5) == 0)
			prefs->dyn.alloc_type = NJ_PROT_UNDER;
		else if(strncmp(env_tmp, "strict", 6) == 0)
			prefs->dyn.alloc_type = NJ_PROT_SUNDER;
		else if(strncmp(env_tmp, "none", 4) == 0)
		{
#ifndef HAVE_SANE_DLOPEN
			__nj_eprintf("NJAMD: Error, you don't have a working dlopen on this sys.\nYou can't chose PROT=NONE\n");
			__nj_njamd_fini();
			_exit();
#endif
			prefs->dyn.alloc_type = NJ_PROT_NONE;
		}
		else
			__nj_eprintf("NJAMD: Invalid malloc checking: %s\n", env_tmp);
	}

	TR;
	if((env_tmp = getenv("NJAMD_ALIGN")) != NULL)
	{
		TR;
		errno=0;
		prefs->dyn.align = strtol(env_tmp, NULL, 0);
		TR;
		if(errno)
		{
			TR;
			__nj_perror("NJAMD: invalid alignment");
			prefs->dyn.align = NJ_ALIGNMENT;
		}
	}
	
	if((env_tmp = getenv("NJAMD_CHK_FREE")) != NULL)
	{
		if(strcmp(env_tmp, "error") == 0)
			prefs->dyn.free_type = NJ_CHK_FREE_ERROR;
		else if(strcmp(env_tmp, "none") == 0)
			prefs->dyn.free_type = NJ_CHK_FREE_NONE;
		else if(strcmp(env_tmp, "segv") == 0) 
			prefs->dyn.free_type = NJ_CHK_FREE_SEGV;
		else if(strcmp(env_tmp, "nofree") == 0)
			prefs->dyn.free_type = NJ_CHK_FREE_NOFREE;
		else
			__nj_eprintf("NJAMD: Invalid free checking: %s\n", env_tmp);
	}

	if((env_tmp = getenv("NJAMD_CALLSTACK_LIMIT")))
	{
		errno=0;
		prefs->stat.callstack_depth = strtol(env_tmp, NULL, 0);
		TR;
		if(errno)
		{
			TR;
			__nj_perror("NJAMD: invalid depth");
			prefs->stat.callstack_depth = 0;
		}
	}
	
	if((env_tmp = getenv("NJAMD_DUMP_LEAKS_ON_EXIT")) != NULL)
	{
		prefs->stat.dump_stats = TRUE;
		prefs->stat.dump_leaks = TRUE;

	}
	else if(getenv("NJAMD_DUMP_STATS_ON_EXIT") != NULL || getenv("NJAMD_DUMP_STATS") != NULL)
		prefs->stat.dump_stats = TRUE;
	
	TR;
	/** @FIXME make a core dump size option for soft core */
	if((env_tmp = getenv("NJAMD_DUMP_CORE")) != NULL)
	{
		if(strcmp(env_tmp, "soft") == 0)
			prefs->stat.core_dump_type = NJ_DUMP_SOFT_CORE;
		else if(strcmp(env_tmp, "hard") == 0)
		{
			setrlimit(RLIMIT_CORE, &rcore);
			prefs->stat.core_dump_type = NJ_DUMP_HARD_CORE;
		}
		else if(strcmp(env_tmp, "none") == 0)
			prefs->stat.core_dump_type = NJ_DUMP_HARD_CORE;
		else 
			__nj_eprintf("NJAMD: Invalid porno^H^H^Hcoredump setting: %s\n", env_tmp);
	}

	if((getenv("NJAMD_ALLOW_READ") != NULL))
		prefs->stat.allow_read = TRUE;

	TR;

	/* Unlink the heap if the user didn't explicity say for it to persist */
	/* Stupid me an my lousy speeling. NJAMD < 0.7.1 spelled persistent
	 * wrong, so now we must maintain backward compatibility */
	if(getenv("NJAMD_PERSISTANT_HEAP") || getenv("NJAMD_PERSISTENT_HEAP"))
		prefs->stat.persist = TRUE;

	/** @FIXME Stop testing these vars just for existance and get a boolean value */
	if(getenv("NJAMD_STORE_RETADDRS"))
		prefs->stat.store_retaddrs = TRUE;

	if(getenv("NJAMD_NO_TRACE"))
		prefs->stat.store_retaddrs = FALSE;

	if(getenv("NJAMD_ALLOW_FREE_0"))
		prefs->stat.allow_free_0 = TRUE;

	if(getenv("NJAMD_ALLOW_MALLOC_0"))
		prefs->stat.allow_malloc_0 = TRUE;

	if(getenv("NJAMD_TRACE_LIBS"))
		prefs->dyn.trace_libs = TRUE;

	if(getenv("NJAMD_MUTABLE_ALLOC"))
		prefs->stat.mutable_alloc = TRUE;

	if((env_tmp = getenv("NJAMD_NO_FREE_INFO")))
	{
		if(!prefs->stat.callstack_depth)
			__nj_eprintf("NJAMD: Error, in order to set no_free_info you must chose a fixed callstack depth (NJ_CALLSTACK_DEPTH)\n");
		else
			prefs->dyn.info_on_free = FALSE;
	}
}

// vim:ts=4
