/* $Id: host_protocol.h,v 1.54 2004/11/12 23:19:24 graziano Exp $ */

#ifndef HOST_PROTOCOL_H
#define HOST_PROTOCOL_H

#include "register.h"
#include "hosts.h"
#include "experiments.h"
#include "nws_api.h"

#ifdef __cplusplus
extern "C" {
#endif


/*
** Returns a default port number for hosts of type #hostType#.
*/
unsigned short
DefaultHostPort(HostTypes hostType);


/*
** Tears down the connection contained in #host_c#.
*/
void
DisconnectHost(struct host_cookie *host_c);


/*
 * Create an object suitable to be registered as host object. The Object
 * has to be freed by the caller. 
 */
Object
CreateHostObject(	const char *myName,
			HostTypes hostType,
			IPAddress *addresses,
			unsigned int addressesCount,
			unsigned short port);
			
/*
 * Takes several steps to establish an NWS host:
 *   1) Attaches port #port# in order to listen for messages.
 *   2) Sets up a listening function to handle host messages.  This handler
 *      will call #exitFunction# when a HOST_DIE message accompanied by a
 *      string that matches #password# is received.  If #exitFunction# is NULL
 *      or returns 1 when called, the handler will terminate via exit(2).  The
 *      handler uses a cached copy of #nameServer# to service HOST_GET_NS and
 *      HOST_TEST messages, and to process calls to HostRegister() and
 *      HostHealthy().  Samee for #memoryServer#: it's used to handle the
 *      HOST_GET_MEMORY message. Uses #registration#, #hostType#, and the
 *      #addressCount#- long array #addresses# to register the host with
 *      the name server.
 * NOTE: #addresses# could also be used to select interfaces of interest.  This
 *       is not presently implemented; hosts will receive messages that arrive
 *       on any interface.
 * NOTE1: the order of #addresses# is important: it defined what the
 *        sensor believes is the main interfaces (important when you have
 *        multiple NICs, some of them private).
 * NOTE2: we messed with signals (SIGTERM are caught to try a graceful stop)
 *
 */
int
EstablishHost(const char *registration,
              HostTypes hostType,
              IPAddress *addresses,
              unsigned int addressesCount,
              unsigned short port,
              const char *password,
              struct host_cookie *nameServer,
              struct host_cookie *memoryServer,
              int (*exitFunction)(void));

/**
 * returns the best IPAddress we have (that is the first passed in
 * EstablishHost). Could be important on multihomed hosts.
 */
IPAddress
PreferredInterface();


/**
 * Returns 1 or 0 depending on whether or not this host is receiving messages
 * sent to #address#:#port#.
 */
int
EstablishedInterface(IPAddress address,
                     unsigned short port);


/**
 * Returns the registration name previously passed to EstablishHost().
 */
const char *
EstablishedRegistration(void);

/**
 * Submit a query to our nameserver(s). Returns 1 if something has been
 * returned (and the results is in #whereTo#), 0 otherwise.
 */
int
RetrieveFromMyNameserver(	const char *filter,
				ObjectSet *whereTo);


/**
 * Returns 1 or 0 depending on whether or not the information stored with the
 * name server for this host is current.
 */
int
HostHealthy(void);

/**
 * Register the experiment. Children of the sensors (which takes the real
 * experiments) should use this function to save the data into the
 * memory. 
 * #experiment# is the exeriment we need to store, and #registration# is
 * the series (as object to be registered) related to this experiment.
 * We take care of registering the series whenever we are succesful on
 * saving the values. We keep some backup of old values we couldn't store
 * into the memory.
 *
 * Returns 1 if #experiment# was saved the backup list, 0 otherwise.
 */
int
RegisterExperiment(	char *registration,
			NWSAPI_Measurement *experiment);

/**
 * Re-registers all host objects, good for the next #timeOut# seconds. 
 * It keeps track of all the nameservers which are working in mirroring
 * and switch to the most appropriate one. 
 * Returns 1 upon success 0 if registration failed.
 */
int
RegisterHost(unsigned long timeOut);


/**
 * Immediately registers #object# with the name server and adds it to the set
 * of objects to be re-registered during a beat. If #object# is already
 * registered, unregister the old one and override with the new one.
 */
void
RegisterObject(const Object object);


/** 
 * This function register #object# with the first available nameserver
 * for #howLong# but doesn't put the object into the set of our
 * registrations (that is it won't be in RegistrationSet() nor won't be
 * registered with RegisterHost()). Returns 0 on failure.
 */
int
RegisterWithNameServer(	ObjectSet object,
			unsigned long timeout);


/**
 * Returns 1 or 0 depending on whether or not #left# and #right# refer to the
 * same host. The be 'same host' #left# and #right# have to met the
 * following:
 * 	- they are listening to the same port;
 * 	- one of the following:
 *		+ they have the same name
 *		+ one of their inet address is the same (only one is
 *		  needed).
 *
 */
int
SameHost(const struct host_cookie *left,
         const struct host_cookie *right);


/*
** Removes the object with a name attribute #objectName# from the set of
** objects to be re-registered during a beat.
*/
void
UnregisterObject(const char *objectName);

/**
 * Returns a copy the set of objects to be re-registered during a beat.
 * You need to free the returned memory. Can be NULL in case of error.
 */
ObjectSet
RegistrationSet(void);

/**
 * Query the host in #cookie# with a HOST_TEST message and return the
 * result in #info#. Returns 1 if  successfull  within #tiemout# 0 otherwise.
 */
int
GetHostInfo(	struct host_cookie *cookie,
		HostInfo *info,
		double timeout);


/**
 * Query the host in #cookie# with a HOST_GET_NS message and return the
 * result in #info#. Returns 1 if  successfull  within #tiemout# 0 otherwise.
 */
int
HostNameServerInfo(	struct host_cookie *cookie,
			NsInfo *info,
			double timeout);


/**
 * Query the host in #cookie# with a HOST_GET_MEMORY message and return the
 * result in #info#. Returns 1 if  successfull  within #tiemout# 0 otherwise.
 */
int
HostMemoryInfo(	struct host_cookie *cookie,
		NsInfo *info,
		double timeout);

/**
 * Send a HOST_DIE message to host in #cookie#. #passord# may be required
 * if the host has been started with the passord option. Returns 1 if
 * succesful within #timeout#, 0 otherwise.
 */
int
HostHalt(	struct host_cookie *cookie,
		const char *password,
		double timeout);


/**
 * Send a HOST_REGISTER (asking the host to change its primary nameserver
 * to #nsName#).  Returns 1 if succesfull within #timeout#, 0 otherwise.
 */
int
HostChangeNameServer(	struct host_cookie *cookie,
			const char *nsName,
			double timeout);


/**
 * Send a HOST_CHANGE_MEMORY (asking the host to change its primary
 * memory to #memory#).  Returns 1 if succesfull within #timeout#, 0
 * otherwise.
 */
int
HostChangeMemory(	struct host_cookie *cookie,
			const char *memory,
			double timeout);


/**
 * Send a HOST_REGISTRATION to the nws host in #cookie# and returns the
 * sensor's registrations in #objs#. Returns 1 if succesful within
 * #timeout#, 0 otherwise.
 */
int
HostGetRegistration(	struct host_cookie *cookie,
			ObjectSet *objs,
			double timeout);

/**
 * Implements the "error" and "log" commands by sending a
 * HOST_DIAGNOSTICS message, accompanied by #diagnostics#, to the
 * host connected to #sd#.  Returns 1 if successful within #timeOut#
 * seconds, else 0.
 * The supported values for the #diagnostic# parameter are defined below.
 * */
#define ALL_ERRORS      -1
#define ALL_LOGS        -2
#define ALL_DIAGNOSTICS -3
                                                                               
int
HostChangeDiagnostic(	struct host_cookie *cookie,
			int diagnostic,
			double timeout);


#ifdef __cplusplus
}
#endif

#endif
