/* $Id: nws_extract.c,v 1.103 2005/07/06 18:47:46 graziano Exp $ */

#include "config_nws.h"
#include <ctype.h>       /* isalpha() isspace() */
#include <stdio.h>       /* printf() */
#include <unistd.h>      /* getopt() sleep() */
#include <stdlib.h>
#include <sys/types.h>   /* size_t */
#include <stdarg.h>      /* Variable parameter stuff. */
#include <string.h>      /* strchr(), strncasecmp() strstr() */
#include <strings.h>     /* strncasecmp() on aix */
#include <time.h>        /* time() */

#include "diagnostic.h" 
#include "strutil.h" 
#define NWSAPI_SHORTNAMES
#include "nws_api.h"     /* NWS programming interface */


/*
** This program allows command-line extraction and display of the measurements
** and forecasts generated by the NWS.  See the user's guide for a description
** of the command-line options.
*/

static const char *FIELD_NAMES[] = {"destination", "mae_error",
	"mae_forecast", "mae_method", "measurement", "mse_error",
	"mse_forecast", "mse_method", "resource", "source", "time", "option"};
typedef enum { DESTINATION, MAE_ERROR, MAE_PREDICTION, MAE_METHOD,
	MEASUREMENT, MSE_ERROR, MSE_PREDICTION, MSE_METHOD, RESOURCE,
	SOURCE, TIME, OPTION
} Fields;
#define FORECAST_FIELD_COUNT (OPTION + 1)
typedef enum { FIELD_NAME, FIELD_VALUE, FIELD_WIDTH
} FieldNwsAttributes;


#define SAFE_STRCPY(to, from) \
  do {strncpy(to, from, sizeof(to)); to[sizeof(to) - 1] = '\0'; if (1) break;} while (0)

#define NUMERIC_WIDTH 12

/*
** Returns #str# trimmed down to at most #len# characters.  The trimming is
** done by removing all trailing characters of words that begin with a capital
** letter, as well as any excess whitespace.  If this is not sufficient,
** trailing characters are truncated.
*/
static const char *
Abbreviate(const char *str,
           size_t len) {

  static char returnValue[255 + 1];
  char *end;
  const char *from;
  int leftToTrim;
  char *to;

	leftToTrim = strlen(str) - len;
	if(leftToTrim <= 0) {
		strcpy(returnValue, str);
		return returnValue; /* It's already short enough. */
	}

  end = &returnValue[len];
  from = str;
  to = returnValue;

  while((to < end) && (leftToTrim > 0)) {
    if(isupper((int)*from)) {
      /* Trim everything to the end of the word except the initial capital. */
      *to++ = *from++;
      while((*from != '\0') && !isspace((int)*from)) {
        from++;
        leftToTrim--;
      }
    }
    else {
      /* No initial capital; copy the whole word and a trailing space. */
      while((to < end) && !isspace((int)*from))
        *to++ = *from++;
      if(to < end)
        *to++ = *from++;
    }
    /* Eliminate any excess whitespace. */
    while(isspace((int)*from) && (leftToTrim > 0)) {
      from++;
      leftToTrim--;
    }
  }

  /* Add a terminating nul, plus any trailing characters we have room for */
  strcpy(to, (leftToTrim <= 0) ? from : "");
  return returnValue;

}


/*
** Appends the #count# strings passed after the third parameter to the
** #len#-long string #dest#.  Terminates #dest# with a nul character.  Returns
** the number of characters appended.
*/
static int
MultiCat(char *dest,
         size_t len,
         int count,
         ...) {

  va_list paramList;
  int i;
  const char *source;
  char *end = dest + len - 1;
  char *next;

  next = dest + strlen(dest);
  va_start(paramList, count);
  for (i = 0; i < count; i++) {
    for (source = va_arg(paramList, const char*);
         (next < end) && (*source != '\0'); next++, source++)
      *next = *source;
  }
  *next = '\0';
  va_end(paramList);
  return next - dest;

}


/*
** Returns #str# padded on #where# to a total length (not including the
** terminating nul) of #len#.
*/
typedef enum {PAD_CENTER, PAD_LEFT, PAD_RIGHT} PadTypes;
static const char *
PadString(const char *str,
          size_t len,
          PadTypes where) {

  static char returnValue[255 + 1];
  size_t padding;
  size_t strSize = strlen(str);

  if(strSize > len)
    return str;  /* No room for any padding. */

  memset(returnValue, ' ', sizeof(returnValue));
  padding = len - strSize;
  strncpy(returnValue + ((where == PAD_CENTER) ? (padding / 2) :
                         (where == PAD_LEFT) ? padding : 0), str, strSize);
  returnValue[len] = '\0';
  return returnValue;

}


/*
** Returns #str# with all occurrences of #toReplace# changed to #replacement#.
*/
static const char *
ReplaceAll(const char *str,
           char toReplace,
           char replacement) {
  char *c;
  static char returnValue[255 + 1];
  SAFE_STRCPY(returnValue, str);
  for(c = returnValue; *c != '\0'; c++) {
    if(*c == toReplace)
      *c = replacement;
  }
  return returnValue;
}


/*
** Returns #str# with any trailing whitespace removed.
*/
static const char *
TrimString(const char *str) {
  static char returnValue[255 + 1];
  char *c;
  SAFE_STRCPY(returnValue, str);
  for(c = returnValue + strlen(returnValue) - 1;
      (c >= returnValue) && isspace((int)*c); c--)
    ; /* Nothing more to do. */
  *(c + 1) = '\0';
  return returnValue;
}


/*
** Info about series used to compute a running forecast and map series names
** back to host and resource names.
*/
typedef struct {
  char name[255 + 1];
  char opts[255 + 1];
  SeriesSpec series;
  ForecastState *forecast;
} SeriesInfo;


/**
 * Look through the #howManySeries#-long array #series# to make sure every
 * source/destination pair of the #howManySources#-long array #sources# and the
 * #howManyDests#-long array #dests# has an entry.  (#howManyDests# may be
 * zero, in which case only source coverage is checked.)  Prints a message to
 * stdout warning the user about any missing series.
 */
static void
CheckSeriesCoverage(	const SeriesInfo *series,
			size_t howManySeries,
			const char **sources,
			size_t howManySources,
			const char **dests,
			size_t howManyDests,
			const char *resourceName) {
	char covered[255];
	int coveredIndex;
	const char **currentDest;
	const SeriesInfo *currentSeries;
	const char **currentSource;
	int destIndex;
	const char **endOfDests;
	const SeriesInfo *endOfSeries;
	const char **endOfSources;
	int sourceIndex;

	endOfDests = (howManyDests == 0) ? NULL : (dests + howManyDests);
	endOfSeries = series + howManySeries;
	endOfSources = sources + howManySources;
	memset(covered, 0, sizeof(covered));

	for(currentSeries = series; currentSeries < endOfSeries; currentSeries++) {
		for(currentSource=sources; currentSource < endOfSources; currentSource++) {
			if(strncmp(currentSeries->series.sourceMachine, *currentSource, strlen(*currentSource)) == 0) {
				if(howManyDests == 0) {
					coveredIndex = currentSource - sources;
					if (covered[coveredIndex])
						; /* TBD Multiple series mapping to one source. */
					covered[coveredIndex] = 1;
					break;
				} else {
					for(currentDest = dests; currentDest < endOfDests; currentDest++) {
						if (strncmp(currentSeries->series.destinationMachine, *currentDest, strlen(*currentDest)) == 0) {
							coveredIndex = (currentSource - sources) * howManySources + currentDest - dests;
							if (covered[coveredIndex])
								; /* TBD Multiple series mapping to one source/dest pair. */
							covered[coveredIndex] = 1;
							break;
						}
					}
					if (currentDest < endOfDests)
						break;
				}
			}
		}
	}

	for(coveredIndex = (howManyDests == 0) ?  (howManySources - 1) : (howManySources * howManyDests - 1); coveredIndex >= 0; coveredIndex--) {
		if (!covered[coveredIndex]) {
			if (howManyDests == 0) {
				ERROR2("No %s series available from %s; ignored\n", resourceName, sources[coveredIndex]);
			} else {
				sourceIndex = coveredIndex / howManyDests;
				destIndex = coveredIndex % howManyDests;
				if (strcmp(sources[sourceIndex], dests[destIndex]) != 0)
					ERROR3("No %s series available from %s to %s; ignored\n", resourceName, sources[sourceIndex], dests[destIndex]);
			}
		}
	}
}


#define SIGNIFICANT_HISTORY 1000


/*
** Fetches measurements for #seriesName# taken since #sinceWhen# and uses them
** to compute in #stateToUse# and return up to #atMost# forecasts in #whereTo#.
** If successful, returns 1 and sets #numberReturned# to the number of
** forecasts copied into #where#; otherwise, returns 0.
*/
int
ExtractForecasts(const char *seriesName,
                 double sinceWhen,
                 ForecastState *stateToUse,
                 NWSAPI_ForecastCollection *whereTo,
                 size_t atMost,
                 unsigned int *numberReturned) {

  size_t historySize;
  static Measurement *measurements = NULL;
  unsigned int howMany;

  /*
  ** Retrieve enough history to generate reasonably accurate forecasts even if
  ** the user wants only a few measurements,
  */
  historySize = (SIGNIFICANT_HISTORY > atMost) ? SIGNIFICANT_HISTORY : atMost;

  if(measurements == NULL) {
    measurements = (Measurement *)MALLOC(historySize * sizeof(Measurement), 0);
    if(measurements == NULL) {
      ERROR("ExtractForecasts: malloc failed\n");
      return 0;
    }
  }

  if(!GetMeasurements(seriesName,
                      sinceWhen,
                      measurements,
                      historySize,
                      &howMany)) {
    ERROR1("ExtractForecasts: unable to retrieve measurements for series %s\n", seriesName);
    return 0;
  }


  UpdateForecastState(stateToUse, measurements, howMany, whereTo, atMost);
  *numberReturned = (atMost < howMany) ? atMost : howMany;
  return 1;

}


/*
 * Formats the fields of #collection# and #series# listed in the
 * #howManyFields#-long array #fieldsDesired# and returns the result.  The
 * #howManyFields#-long array #whichNwsAttributes# specifies, for each field,
 * whether the field value, padding blanks, or the field name should be
 * included in the image.
 */
static const char *
ForecastCollectionImage(	const SeriesInfo *series,
				const ForecastCollection *collection,
				const Fields *fieldsDesired,
				const FieldNwsAttributes *whichNwsAttributes,
				size_t howManyFields) {
	char fieldValue[127 + 1];
	int i;
	char *nextChar;
	Fields printField;
	static char returnValue[255 + 1];
	static const size_t FIELD_WIDTHS[FORECAST_FIELD_COUNT] = {32, NUMERIC_WIDTH, NUMERIC_WIDTH, 15, NUMERIC_WIDTH, NUMERIC_WIDTH, NUMERIC_WIDTH, 15, 20, 32, 11, 32}; 
	static const char* HEADERS[FORECAST_FIELD_COUNT] = {"Destination", "MAE Err", "MAE Fore", "MAE Method", "Measure", "MSE Err", "MSE Fore", "MSE Method", "Resource", "Source", "Time", "Options"};

	nextChar = returnValue;
	for(i = 0; i < howManyFields; i++) {
		printField = fieldsDesired[i];
		if(whichNwsAttributes[i] == FIELD_WIDTH) {
			fieldValue[0] = '\0';
		} else if(whichNwsAttributes[i] == FIELD_NAME) {
			SAFE_STRCPY(fieldValue, HEADERS[printField]);
		} else if(printField == DESTINATION) {
			SAFE_STRCPY(fieldValue, series->series.destinationMachine); 
		} else if(printField == MAE_ERROR) {
			sprintf(fieldValue, "%.*f", NUMERIC_WIDTH, collection->forecasts[MAE_FORECAST].MAE_error);
		} else if(printField == MAE_PREDICTION) {
			sprintf(fieldValue, "%.*f", NUMERIC_WIDTH, collection->forecasts[MAE_FORECAST].forecast);
		} else if(printField == MAE_METHOD) {
			SAFE_STRCPY(fieldValue, MethodName(collection->forecasts[MAE_FORECAST].methodUsed));
		} else if(printField == MEASUREMENT) {
			sprintf(fieldValue, "%.*f", NUMERIC_WIDTH, collection->measurement.measurement);
		} else if(printField == MSE_ERROR) {
			sprintf(fieldValue, "%.*f", NUMERIC_WIDTH, collection->forecasts[MSE_FORECAST].error);
		} else if(printField == MSE_PREDICTION) {
			sprintf(fieldValue, "%.*f", NUMERIC_WIDTH, collection->forecasts[MSE_FORECAST].forecast);
		} else if(printField == MSE_METHOD) {
			SAFE_STRCPY(fieldValue, MethodName(collection->forecasts[MSE_FORECAST].methodUsed));
		} else if(printField == RESOURCE) {
			SAFE_STRCPY(fieldValue, series->series.resourceName);
		} else if(printField == SOURCE) {
			SAFE_STRCPY(fieldValue, series->series.sourceMachine);
		} else if(printField == TIME) {
			sprintf(fieldValue, "%d", (int)collection->measurement.timeStamp);
		} else if(printField == OPTION) {
			SAFE_STRCPY(fieldValue, series->opts);
		}
		strcpy(fieldValue, Abbreviate(fieldValue, FIELD_WIDTHS[printField] - 1));
		strcpy(fieldValue, ReplaceAll(fieldValue, ' ', '_'));
		strcpy(nextChar, PadString(fieldValue,FIELD_WIDTHS[printField],PAD_RIGHT));
		nextChar += FIELD_WIDTHS[printField];
	}
	return TrimString(returnValue);
}


/*
 * Prints those fields of #series# and #forecast# included in the
 * #howManyFields#-long array *fieldsToPrint# to stdout.  #headerFrequency#
 * indicates how often header information should be interspersed among field
 * values.
 */
static void
PrintForecastCollection(int headerFrequency,
                        const SeriesInfo *series,
                        const ForecastCollection *forecast,
                        const Fields *fieldsToPrint,
                        size_t howManyFields) {
	int i, src = -1, dst = -1;
	FieldNwsAttributes printControl[FORECAST_FIELD_COUNT];
	static SeriesSpec priorSeries = {"", "", ""};
	static int callsUntilHeader = 0;

	if((headerFrequency > 0) && (callsUntilHeader == 0)) {
		for(i = 0; i < howManyFields; i++) {
			printControl[i] = FIELD_NAME;
		}
		printf("%s\n",
		ForecastCollectionImage(NULL,
                                   NULL,
                                   fieldsToPrint,
                                   printControl,
                                   howManyFields));
		for(i = 0; i < howManyFields; i++) {
			printControl[i] = FIELD_VALUE;
		}
		callsUntilHeader = headerFrequency - 1;
	} else {
		if(headerFrequency > 0) {
			callsUntilHeader--;
		}
		for(i = 0; i < howManyFields; i++) {
			if((fieldsToPrint[i] == RESOURCE) && (strcmp(series->series.resourceName, priorSeries.resourceName) == 0)) {
				printControl[i] = FIELD_WIDTH;
			} else { 
				if (fieldsToPrint[i] == DESTINATION) 
					dst = i;
				else if (fieldsToPrint[i] == SOURCE) 
					src = i;
				printControl[i] = FIELD_VALUE;
			}
		}
	}

	/* let's check if we already printed our source/destination: we
	 * have to be careful at the paired resource to consider
	 * source/destination as a single entity */
  	if (src == -1) {
		if (dst != -1 && (strcmp(series->series.destinationMachine, 
				priorSeries.destinationMachine) == 0)) {
		        printControl[dst] = FIELD_WIDTH;
		}
	} else if(strcmp(series->series.sourceMachine,priorSeries.sourceMachine)==0) {
		if (dst == -1) {
		        printControl[src] = FIELD_WIDTH;
		} else if (strcmp(series->series.destinationMachine, 
				priorSeries.destinationMachine) == 0) {
		        printControl[src] = FIELD_WIDTH;
		        printControl[dst] = FIELD_WIDTH;
		}
	}
	printf("%s\n", ForecastCollectionImage(series,
                                 forecast,
                                 fieldsToPrint,
                                 printControl,
                                 howManyFields));
	fflush(stdout);
	priorSeries = series->series;
}


/*
** Retrieves from the name server registrations for all series matching
** #filter# that have a source equal to any element of the #sourceCount#-long
** array #sources# and their source and a destination equal to any element of
** the #destCount#-long array #dests#.  #destCount# may be zero, in which case
** #dests# is ignored.  Returns an array of series info for the retrieved
** registration and sets #seriesCount# to its length.  Returns NULL on error.
*/
static SeriesInfo *
RetrieveSeriesInfo(	const char **sources,
			int sourceCount,
			const char **dests,
			int destCount,
			const char *filter,
			int *seriesCount) {
	char *attr, *value;
	const char *s;
	SeriesInfo currentSeries;
	char fullFilter[4095 + 1] = "";
	int hostIndex;
	SeriesInfo *returnValue;
	Object seriesObject;
	ObjectSet seriesObjectSet;

	/* Combine #filter#, #sources# and, optionally, #dests# into a
	 * name server retrieval filter:
	 * (&(objectclass=nwsSeries)<filter>(|(source=<source>)...)(|(dest=<dest)...))
	 * The host names appear in the registration as <machine>:<port>;
	 * we allow the user to drop the port and the full qualification
	 * of the machine. */
	MultiCat(fullFilter, sizeof(fullFilter), 4, "(&", SERIES, filter, "(|");
	for(hostIndex = 0; hostIndex < sourceCount; hostIndex++) {
		MultiCat(fullFilter, sizeof(fullFilter), 5,
			"(", HOST_ATTR, "=", sources[hostIndex],
			(strchr(sources[hostIndex], ':') != NULL) ? ")" :
			(strchr(sources[hostIndex], '.') != NULL) ? ":*)" : "*)");
	}
	if(destCount != 0) {
		MultiCat(fullFilter, sizeof(fullFilter), 2, ")", "(|");
		for(hostIndex = 0; hostIndex < destCount; hostIndex++) {
			MultiCat(fullFilter, sizeof(fullFilter), 5,
				"(", TARGET_ATTR, "=", dests[hostIndex],
				(strchr(dests[hostIndex], ':') != NULL) ? ")" :
				(strchr(dests[hostIndex], '.') != NULL) ? ":*)" : "*)");
		}
	}
	strcat(fullFilter, "))");

	if(!GetObjects(fullFilter, &seriesObjectSet)) {
		ERROR("RetrieveSeriesInfo: series retrieval failed\n");
		return NULL;
	}

	*seriesCount = 0;
	ForEachObject(seriesObjectSet, seriesObject) {
		(*seriesCount)++;
	}

	returnValue = MALLOC(*seriesCount * sizeof(SeriesInfo), 0);
	if(returnValue == NULL) {
		FreeObjectSet(&seriesObjectSet);
		ERROR("RetrieveSeriesInfo: malloc failed\n");
		return NULL;
	}

	/* Parse each retrieved series into a SeriesInfo struct. */
	*seriesCount = 0;
	ForEachObject(seriesObjectSet, seriesObject) {
		SAFE_STRCPY(currentSeries.series.sourceMachine, "");
		SAFE_STRCPY(currentSeries.series.destinationMachine, "");
		SAFE_STRCPY(currentSeries.series.resourceName, "");
		SAFE_STRCPY(currentSeries.name, "");
		SAFE_STRCPY(currentSeries.opts, "");
		currentSeries.forecast = NewForecastState();
		if (currentSeries.forecast == NULL) {
			ABORT("out of memory\n");
		}

		/* let's get the single attributes */
		value = NwsAttributeValue_r(NWSAPI_FindNwsAttribute(seriesObject, TARGET_ATTR));
		if (value) {
			SAFE_STRCPY(currentSeries.series.destinationMachine, value);
			free(value);
		}
		value = NwsAttributeValue_r(NWSAPI_FindNwsAttribute(seriesObject, NAME_ATTR));
		if (value) {
			SAFE_STRCPY(currentSeries.name, value);
			free(value);
		}
		value = NwsAttributeValue_r(NWSAPI_FindNwsAttribute(seriesObject, RESOURCE_ATTR));
		if (value) {
			SAFE_STRCPY(currentSeries.series.resourceName, value);
			free(value);
		}
		value = NwsAttributeValue_r(NWSAPI_FindNwsAttribute(seriesObject, HOST_ATTR));
		if (value) {
			SAFE_STRCPY(currentSeries.series.sourceMachine, value);
			free(value);
		}
		value = NwsAttributeValue_r(NWSAPI_FindNwsAttribute(seriesObject, OPTION_ATTR));
		if (value) {
			for(s = value; GETTOK(fullFilter, s, ",", &s); ) {
				attr = NwsAttributeValue_r(NWSAPI_FindNwsAttribute(seriesObject, fullFilter));
				if (attr == NULL) {
					continue;
				}
				if (currentSeries.opts[0] != '\0') {
					strcat(currentSeries.opts, ",");
				}
				strcat(currentSeries.opts, fullFilter);
				strcat(currentSeries.opts, attr);
				free(attr);
			}
		}
		free(value);

		returnValue[(*seriesCount)++] = currentSeries;
	}

	FreeObjectSet(&seriesObjectSet);
	return returnValue;
}



/* Defaults for command-line values and NWS host ports. */
#define DEFAULT_DISPLAY_SIZE 20
#define DEFAULT_HEADER_FREQ 20
#define DEFAULT_FIELDS "time,measurement,mae_forecast,mae_error,mse_forecast,mse_error,source,destination,resource"
static const char *DEFAULT_RESOURCES[] = {
	DEFAULT_BANDWIDTH_RESOURCE, DEFAULT_AVAILABLE_CPU_RESOURCE,
	DEFAULT_CURRENT_CPU_RESOURCE, DEFAULT_LATENCY_RESOURCE,
	DEFAULT_MEMORY_RESOURCE
};
#define DEFAULT_RESOURCE_COUNT \
        (sizeof(DEFAULT_RESOURCES) / sizeof(DEFAULT_RESOURCES[0]))

#define SWITCHES "af:h:M:n:N:t:wv:VS"
void usage() {
	printf("\nUsage: nws_extract [OPTIONS] resource [filter] host [...]\n");
	printf("extraction utility for the Network Weather Service\n");
	printf("\nOPTIONS can be:\n");
	printf("\t-a                  treat all listed machines as experiment  sources.\n");
	printf("\t-f fieldlist        list of fields you would like to display\n");
	printf("\t-n #                get only # measurements/forecasts\n");
	printf("\t-h #                specify header frequency\n");
	printf("\t-t #                gets only measurements newer then #\n");
	printf("\t-M memory           use this NWS memory\n");
	printf("\t-N nameserver       use this NWS nameserver\n");
	printf("\t-w                  continuosly display informations\n");
	printf("\t-S                  there is no resource/filter/host but only series name\n");
	printf("\t-v level            verbose level (up to 5)\n");
	printf("\t-V                  print version\n");
	printf("\nresource is the resource you are interested in (use nws_search(1)\n to find out available resources on your NWS installation\n");
	printf("\nfilter is to be used in conjunction with the -N options. It specifies a\nfilter (as defined in nws_search(1))\n"); 
	printf("\nhost [...]  these  are  the  lists of hosts for which you are interested\nin getting the measurements.\n");
	printf("Report bugs to <nws@nws.cs.ucsb.edu>.\n\n");
}

int
main(		int argc,
		char *argv[]) {
	extern char *optarg;
	extern int optind;
	/* User-settable parameters. */
	int autoFetch = 0;
	int extractSingle = 1;
	int seriesNameOnly = 0;
	int headerFrequency = DEFAULT_HEADER_FREQ;
	size_t initialDisplaySize = DEFAULT_DISPLAY_SIZE;
	char printOrder[255 + 1];
	double sinceWhen = BEGINNING_OF_TIME;

	/* Other local variables. */
	Measurement meas;
	ForecastCollection currentForecast, *forecasts;
	char autoName[127 + 1];
	int extractCount;
	SeriesInfo *extractSeries;
	char *fieldBegin;
	char *fieldEnd;
	char filter[1023 + 1];
	const char **firstHost;
	HostSpec host;
	int hostCount, nameServer;
	int i, j;
	int interMachine = 0;
	int lookupSeries = 1;
	int opt;
	Fields printFields[FORECAST_FIELD_COUNT];
	size_t printFieldsLen;
	const char *resourceName;
	unsigned int returnedCount;
  	int verbose;

	verbose = 1;
	resourceName = NULL;
	nameServer = 0;

	fieldEnd = EnvironmentValue_r("EXTRACT_FIELDS", DEFAULT_FIELDS);
	if (fieldEnd == NULL) {
		ABORT("out of memory\n");
	}
	SAFE_STRCPY(printOrder, fieldEnd);
	FREE(fieldEnd);


	while((opt = getopt(argc, argv, SWITCHES)) != EOF) {
		switch(opt) {
		case 'a':
			/* extract all known series amongst the hosts */
			extractSingle = 0;
			break;

		case 'f':
			SAFE_STRCPY(printOrder, optarg);
			break;

		case 'h':
			headerFrequency = (int)strtol(optarg, NULL, 10);
			break;

		case 'S':
			seriesNameOnly = 1;
			break;

		case 'M':
			host = *MakeHostSpec (optarg, GetDefaultPort(NWSAPI_MEMORY_HOST));
			if(!UseMemory(&host)) {
				ERROR2("Unable to contact memory %s:%d\n", host.machineName, host.machinePort);
				exit(1);
			}
			lookupSeries = 0;
			break;

		case 'n':
			initialDisplaySize = (int)strtol(optarg, NULL, 10);
			break;

		case 'N':
			host = *MakeHostSpec (optarg, GetDefaultPort(NWSAPI_NAME_SERVER_HOST));
			if(!UseNameServer(&host)) {
				WARN2("Unable to contact name server %s:%d\n", host.machineName, host.machinePort);
			} else {
				nameServer = 1;
			}
			break;

		case 't':
			sinceWhen = time(NULL) - strtol(optarg, NULL, 0);
			break;

		case 'w':
			autoFetch = 1;
			break;

		case 'v':
			verbose = (unsigned short)atol(optarg);
			break;

		case 'V':
			printf("nws_extract for NWS version %s", VERSION);
#ifdef HAVE_PTHREAD_H
			printf(", with thread support");
#endif
#ifdef WITH_DEBUG
			printf(", with debug support");
#endif
			printf("\n\n");
			exit(0);
			break;

		default:
			usage();
			exit(1);
			break;

		}
	}

	/* we need -M with -S */
	if (seriesNameOnly && lookupSeries) {
		printf("You need to specify the memory (-M) with -S\n");
		exit(1);
	}

	/* let's set the verbose level  and open up files (if needed) */
	/* fatal errors are always on */
	SetDiagnosticLevel(verbose, stderr, stderr);

	/* warning about something the user might not want */
	if (nameServer && !lookupSeries) {
		WARN("You used both -N and -M switch: -N won't be used\n");
	}

	/* let's see if we have enough commandline options */
	if ((optind + (seriesNameOnly ? 0 : 1)) >= argc) {
		usage();
		exit(1);
	}

	/* if we have specified -S, we don't have any host/resource just
	 * series name */
	if (!seriesNameOnly) {
		/*
		 * Determine which resource we're being asked to extract
		 * and see if the user has specified the optional filter
		 * to distinguish between multiple series.
		 */
		resourceName = argv[optind++];
		opt = strlen(resourceName);
		for(i = 0; i < DEFAULT_RESOURCE_COUNT; i++) {
			if(strncasecmp(resourceName, DEFAULT_RESOURCES[i], opt) == 0) {
				resourceName = DEFAULT_RESOURCES[i];
				break;
			}
		}

		/* this is the extra filter specified by the user */
		filter[0] = '\0';
		if (*argv[optind] == '(') {
			MultiCat(filter, sizeof(filter), 7, "(&", argv[optind], "(", RESOURCE_ATTR, "=", resourceName, "))");
			if (optind++ >= argc) {
				usage();
				exit(1);
			}
		} else if (*argv[optind] == '&' || *argv[optind] == '|') {
			MultiCat(filter, sizeof(filter), 7, "(&(", argv[optind], "(", RESOURCE_ATTR, "=", resourceName, ")))");
			if (optind++ >= argc) {
				usage();
				exit(1);
			}
		} else {
			MultiCat(filter, sizeof(filter), 5, "(", RESOURCE_ATTR, "=", resourceName, ")");
		}

		/* check if we have enough hosts for the experimente we've been
		 * asked */
		interMachine = IntermachineResource(resourceName);
		if (interMachine && ((optind + 1) >= argc)) {
			ABORT("You must specify at least one destination machine\n");
		}
	}

	/* Determine the field display order. */
	for(printFieldsLen = 0, fieldBegin = printOrder; ; fieldBegin = fieldEnd + 1) {
		fieldEnd = strchr(fieldBegin, ',');
		if(fieldEnd == NULL)
			fieldEnd = fieldBegin + strlen(fieldBegin);
		for(i = 0; i < FORECAST_FIELD_COUNT; i++) {
			if(strncasecmp(fieldBegin, FIELD_NAMES[i], fieldEnd - fieldBegin) == 0)
				break;
		}
		if(i == FORECAST_FIELD_COUNT) {
			ERROR1("Unknown field name %s\n", fieldBegin);
			exit(1);
		}
		/* Ignore destination for single-machine resources. */
		if(interMachine || ((Fields)i != DESTINATION))
			printFields[printFieldsLen++] = (Fields)i;
		if(*fieldEnd == '\0')
			break;
	}

	/* Get the information about the specified series */
	hostCount = argc - optind;
	firstHost = (const char **)&argv[optind];

	if (!nameServer && !seriesNameOnly && lookupSeries) {
		/* we need to ask the sensor for the nameserver */
		HostSpec sensorSpec;

		sensorSpec = *MakeHostSpec(*firstHost, GetDefaultPort(NWSAPI_SENSOR_HOST));

		if (GetNameServer(&sensorSpec, &host)) {
			UseNameServer(&host);
		}
	}

	/* let's get the series */
	if (lookupSeries) {
		/* look for the series registered with the nameserver */
		extractSeries = RetrieveSeriesInfo(firstHost,
                                       (!interMachine || !extractSingle) ?
                                        hostCount : 1,
                                       firstHost + extractSingle,
                                       interMachine ? hostCount - extractSingle  : 0,
                                       filter,
                                       &extractCount);
		if(extractSeries == NULL) {
			exit(1); 
		}
		CheckSeriesCoverage(extractSeries,
				extractCount,
				firstHost,
				(!interMachine || !extractSingle) ?  hostCount : 1,
				firstHost + extractSingle,
				interMachine ? hostCount - extractSingle : 0,
				resourceName);
	} else if (seriesNameOnly) {
		/* we have a list of series */
		extractCount = hostCount;
		extractSeries = (SeriesInfo *)MALLOC(extractCount * sizeof(SeriesInfo), 1);
		for(i = 0; i < extractCount; i++) {
			extractSeries[i].series.sourceMachine[0] = '\0';
			extractSeries[i].series.destinationMachine[0] = '\0';
			extractSeries[i].series.resourceName[0] = '\0';
			extractSeries[i].opts[0] = '\0';
			strcpy(extractSeries[i].name, firstHost[i]);
			extractSeries[i].forecast = NewForecastState();
			if (extractSeries[i].forecast == NULL) {
				ABORT("out of memory\n");
			}
		}
	} else {
		extractCount = !interMachine ? hostCount : !extractSingle ? (hostCount * (hostCount - 1)) : (hostCount - 1);
		extractSeries = (SeriesInfo *)MALLOC(extractCount * sizeof(SeriesInfo), 1);

		/* let's generate the series specs */
		for (i = 0; i < extractCount; i++) {
			extractSeries[i].opts[0] = '\0';
			if (!interMachine) {
				strcpy(extractSeries[i].series.sourceMachine, firstHost[i]);
				strcpy(extractSeries[i].series.destinationMachine, "");
			} else {
				/* let's get the first host */
				j = extractCount / hostCount;

				/* the second host is in i % hostCount */
				if (j == i % hostCount && !extractSingle) {
					continue;
				}

				strcpy(extractSeries[i].series.sourceMachine, firstHost[j]);
				strcpy(extractSeries[i].series.destinationMachine, firstHost[(i + extractSingle) % hostCount]);
			}
			/* let's get the basics */
			strcpy(extractSeries[i].series.resourceName, resourceName);
			strcpy(extractSeries[i].name, SeriesName(&extractSeries[i].series));
			extractSeries[i].forecast = NewForecastState();
			if (extractSeries[i].forecast == NULL) {
				ABORT("out of memory\n");
			}

		}
	}

	forecasts = (ForecastCollection *) MALLOC(initialDisplaySize * sizeof(ForecastCollection), 1);

	/* Compute and print the initial forecast set for each series. */
	opt = 1;		/* failed by default */
	for(i = 0; i < extractCount; i++) {
		if(ExtractForecasts(extractSeries[i].name,
					sinceWhen,
					extractSeries[i].forecast,
					forecasts,
					initialDisplaySize,
					&returnedCount)) {
			if(returnedCount > 0) {
				for(j = returnedCount -1; j >= 0; j--) {
					currentForecast = forecasts[j];
					PrintForecastCollection(headerFrequency,
							&extractSeries[i],
							&currentForecast,
							printFields,
							printFieldsLen);
				}
				opt = 0;	/* we got at least 1 series */
			} else {
				/* there is no data here, so we try to
				 * print a sensible message depending on
				 * the conditions */
				if (sinceWhen != BEGINNING_OF_TIME) {
					/* the user specified she doesn't
					 * want old data */
					LOG1("Only old data for %s\n", extractSeries[i].series.sourceMachine);
				} else {
					/* let's tell the user we didn't
					 * find anything useful */
					ERROR1("No measurements for %s\n", extractSeries[i].name);
					if (!lookupSeries) {
						ERROR("Nothing found: try to use the -N option.\n");
					}
				}
			}
		}
		if (autoFetch) {
			(void)AutoFetchBegin(extractSeries[i].name);
		}

		fflush(stdout);
	}

	free(forecasts);

	/*  If requested, continue to display new measurements and
	 *  forecasts based on them as they come in. */
	while (autoFetch) {
		if (AutoFetchCheck(autoName, sizeof(autoName), &meas, 120)) {
			for(i = 0; i < extractCount; i++) {
				if (!strcmp(autoName, extractSeries[i].name)) {
					break;
				}
			}
			if(i == extractCount) {
				continue; /* Unknown series? */
			}

			UpdateForecastState(extractSeries[i].forecast,
					&meas,
					1,
					&currentForecast,
					1);
			PrintForecastCollection(headerFrequency,
					&extractSeries[i],
					&currentForecast,
					printFields,
					printFieldsLen);
		} else if (autoName[0] == '\0') {
			/* no message in the timeout */
			INFO("no message received\n");
		} else if (strchr(autoName, '\t')) {
			/* the memory connection failed: let's redo
			 * everything */
			for(i = 0; i < extractCount; i++) {
				(void)AutoFetchEnd(extractSeries[i].name);
			}
			for(i = 0; i < extractCount; i++) {
				(void)AutoFetchBegin(extractSeries[i].name);
			}
		} else {
			/* we got a problem on a single series */
			for(i = 0; i < extractCount; i++) {
				if (!strcmp(extractSeries[i].name, autoName)) {
					continue;
				}

				INFO1("restarting autofetch (%s)\n", autoName);
				(void)AutoFetchEnd(extractSeries[i].name);
				(void)AutoFetchBegin(extractSeries[i].name);
				break;
			}
		}
	}

	/* Tidy up. */
	for(i = 0; i < extractCount; i++) {
		FreeForecastState(&extractSeries[i].forecast);
	}
	free(extractSeries);

	return opt;
}
