/* $Id: osutil.c,v 1.59 2005/04/19 22:38:02 graziano Exp $ */

#include "config_portability.h"

#include <ctype.h>
#include <errno.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#ifdef HAVE_SYS_WAIT_H
#	include <sys/wait.h> 
#endif
#ifdef HAVE_SYS_UTSNAME_H
#	include <sys/utsname.h>
#endif

#include "osutil.h"

#ifdef HAVE_PTHREAD_H
#	include <pthread.h>
#endif

#include "diagnostic.h"

/* lock for the module */
static void *lock = NULL;


/* global variable of the module */
static char *systemName = NULL;
static char *releaseName = NULL;
static char *machineName = NULL;

/* internal function that fills the data coming from uname. Returns 1 on
 * success, 0 on failure */
static int
CallUname() {
#ifdef HAVE_UNAME
	static struct utsname uts;
	int ret;

	GetNWSLock(&lock);
	ret = uname(&uts);
	if (ret != -1) {
		/* let's fill the fields */
		systemName = uts.sysname;
		releaseName = uts.release;
		machineName = uts.machine;
	}
	ReleaseNWSLock(&lock);

	return (ret != -1);
#else
	return 0;
#endif
}

const char *
GetSystemName() {
	if (systemName == NULL) {
		if (!CallUname()) {
			return NULL;
		}
	}

	return systemName;
}

const char *
GetRelease() {
	if (systemName == NULL) {
		if (!CallUname()) {
			return NULL;
		}
	}

	return releaseName;
}

const char *
GetMachine() {
	if (systemName == NULL) {
		if (!CallUname()) {
			return NULL;
		}
	}

	return machineName;
}


/* 
 * These are defined here only becasue they are not found (sometimes) in
 * the headers but only in libraries. To avoid annoying compilation warnings
 */
#ifdef HAVE_SIGHOLD
int sighold(int sig);
#endif
#ifdef HAVE_SIGRELSE
int sigrelse(int sig);
#endif


int
CPUCount( ) {
	long sysconfCount = 1;
#ifdef HAVE_SYSCONF
#	ifdef SYSCONF_PROCESSOR_COUNT_PARAMETER 
#		undef SYSCONF_PROCESSOR_COUNT_PARAMETER
#	endif
#	ifdef _SC_CRAY_NCPU
#		define SYSCONF_PROCESSOR_COUNT_PARAMETER _SC_CRAY_NCPU
#	elif defined(_SC_NPROC_CONF)
#		define SYSCONF_PROCESSOR_COUNT_PARAMETER _SC_NPROC_CONF
#	elif defined(_SC_NPROCESSORS_CONF)
#		define SYSCONF_PROCESSOR_COUNT_PARAMETER _SC_NPROCESSORS_CONF
#	endif
#	ifdef SYSCONF_PROCESSOR_COUNT_PARAMETER 
	sysconfCount = sysconf(SYSCONF_PROCESSOR_COUNT_PARAMETER);
	if (sysconfCount <= 0) {
		/* remove error and non-sense */
		sysconfCount = 1;
	}
#	endif
#endif
	return (int) sysconfCount;
}

double
InstalledMemory() {
	double mem = -1;
#if defined(HAVE_SYSCONF) && defined(_SC_PHYS_PAGES)
#	ifdef _SC_PAGE_SIZE
	mem = sysconf(_SC_PAGE_SIZE);
#	elif defined(PAGE_SIZE)
	mem = sysconf(PAGE_SIZE);
#	endif
	if (mem > 0) {
		mem *= sysconf(_SC_PHYS_PAGES);
		if (mem < 0) {
			mem = -1;
		}
	}
#endif
	if (mem > 0) {
		mem /= (1024*1024);
	}
	return mem;
}

double
AvailableMemory() {
	double mem = -1;
#if defined(HAVE_SYSCONF) && defined(_SC_AVPHYS_PAGES)
#	ifdef _SC_PAGE_SIZE
	mem = sysconf(_SC_PAGE_SIZE);
#	elif defined(PAGE_SIZE)
	mem = sysconf(PAGE_SIZE);
#	endif
	if (mem > 0) {
		mem *= sysconf(_SC_AVPHYS_PAGES);
		if (mem < 0) {
			mem = -1;
		}
	}
#endif
	if (mem > 0) {
		mem /= (1024*1024);
	}
	return mem;
}


/* It should be thread safe (time should be thread safe) */
double
CurrentTime(void) {
	return((double)time(NULL));
}


char *
GetEnvironmentValue(	const char *name,
			const char *rcDirs,
			const char *rcName,
			const char *defaultValue) {
	const char *dirEnd;
	const char *dirStart;
	const char *homeDir;
	char *from, *to;
	size_t nameLen;
	FILE *rcFile;
	char rcLine[255 + 1], rcPath[255 + 1];
	char *returnValue;
	int i, len;

	/* sanity check */
	if (name == NULL) {
		if (defaultValue != NULL) {
			return strdup(defaultValue);
		} else {
			return NULL;
		}
	}

	returnValue = getenv(name);
	if(returnValue != NULL) {
		/* easy way out: we got the environmental variable */
		return strdup(returnValue);
	}

	if (rcName != NULL && *rcName != '\0') {
		/* we need rcDirs of we are here */
		if (rcDirs == NULL) {
			return NULL;
		}

		nameLen = strlen(name);
		len = sizeof(rcPath);

		for(dirStart = rcDirs, dirEnd = dirStart; dirEnd != NULL; dirStart = dirEnd + 1) {
			/* Construct a file path from the next dir in
			 * rcDirs and rcName. */
			dirEnd = strchr(dirStart, ':');
			if (dirEnd == NULL) {
				i = strlen(dirStart);
			} else {
				i = dirEnd -dirStart;
			}
			if (i >= (len - strlen(rcName) - 1)) {
				/* we wouldn't fit the name anyway */
				continue;
			}
			strncpy(rcPath, dirStart, i);
			rcPath[i] = '\0';

			if ((strcmp(rcPath, "~") == 0) || (strcmp(rcPath, "~/") == 0)) {
				homeDir = getenv("HOME");
				if (homeDir != NULL) {
					strncpy(rcPath, homeDir, len);
					rcPath[len - 1] = '\0';
				}
			}
			strcat(rcPath, "/");
			strcat(rcPath, rcName);
			rcFile = fopen(rcPath, "r");

			if (rcFile == NULL) {
				/* no luck, try the next one */
				continue;
			}

			while(fgets(rcLine, sizeof(rcLine), rcFile) != NULL) {
				/* Test against pattern " *#name# =". */
				for(from = rcLine; (*from != '\0') && isspace((int)*from); from++)
					; /* Nothing more to do. */
				if(strncmp(from, name, nameLen) != 0) {
					continue;
				}
				for(from += nameLen; (*from != '\0') && isspace((int)*from); from++)
					; /* Nothing more to do. */
				if(*from != '=') {
					continue;
				}

				/* We found a line that sets the variable. */
				(void)fclose(rcFile);
				for(from++; (*from != '\0') && isspace((int)*from); from++)
					; /* Nothing more to do. */

				/* Return a single word to allow for
				 * future free-format input. */
				if(*from == '"') {
					returnValue = from + 1;
					for(from++, to = from; (*from != '\0') && (*from != '"'); from++, to++) {
						if(*from == '\\') {
							from++;
							switch(*from) {
							case 'b':
								*to = '\b';
								break;
							case 'f':
								*to = '\f';
								break;
							case 'n':
								*to = '\n';
								break;
							case 'r':
								*to = '\r';
								break;
							case 't':
								*to = '\t';
								break;
							case 'v':
								*to = '\v';
								break;
							default:
								*to = *from;
								break;
							}
						} else {
							*to = *from;
						}
					}
				} else {
					returnValue = from;
					for(to = from; (*to != '\0') && !isspace((int)*to); to++)
						; /* Nothing more to do. */
				}
				*to = '\0';

				if (returnValue != NULL) 
					return strdup(returnValue);
				else 
					break;
			}
		(void)fclose(rcFile);
		}
	}

	/* we didn't find anything: return the default value */
	if (defaultValue != NULL) {
		returnValue = strdup(defaultValue);
	} else {
		returnValue = NULL;
	}

	return returnValue;
}

int
GetUserName(	char *name,
		size_t length) {
	struct passwd *myPasswd;

	GetNWSLock(&lock);
	myPasswd = getpwuid(getuid());
	if(myPasswd != NULL) {
		strncpy(name, myPasswd->pw_name, length);
		name[length - 1] = '\0';
	}
	endpwent();
	ReleaseNWSLock(&lock);

	return(myPasswd != NULL);
}

void
HoldSignal(int sig) {
#ifdef HAVE_SIGHOLD
	sighold(sig);
#else
	sigset_t set, oset;
	sigemptyset(&set);
	sigaddset(&set, sig);
	sigprocmask(SIG_BLOCK, &set, &oset);
#endif
}


int
MakeDirectory(	const char *path,
		mode_t mode,
		int makeEntirePath) {
	const char *endSubPath;
	struct stat pathStat;
	char *subPath;

	if(makeEntirePath) {
		endSubPath = path;  /* This will ignore leading '/' */
		while((endSubPath = strchr(endSubPath + 1, '/')) != NULL) {
			subPath = (char *)MALLOC(endSubPath - path + 1, 0);
			if (subPath == NULL) {
				return 0;
			}
			strncpy(subPath, path, endSubPath - path);
			subPath[endSubPath - path] = '\0';
			if((stat(subPath, &pathStat) == -1) && (errno == ENOENT)) {
				if(mkdir(subPath, mode) == -1) {
					FREE(subPath);
					return 0;
				}
			} else if(!S_ISDIR(pathStat.st_mode)) {
				FREE(subPath);
				return 0;
			}
			FREE(subPath);
		}
	}

	if((stat(path, &pathStat) == -1) && (errno == ENOENT)) {
		return(mkdir(path, mode) != -1);
	} else {
		return(S_ISDIR(pathStat.st_mode));
	}
}


#ifdef HAVE_PTHREAD_H
typedef struct {
	pthread_mutex_t lock;
	pthread_cond_t cond;
	int value;
} nwsLockType;
#endif

int
GetNWSLock(void **Lock) {
#ifdef HAVE_PTHREAD_H
	/* this is the lock for this module. Every other module will use
	 * a void * as a variable as mutex. */
	int ret;
	static pthread_mutex_t nwsLock = PTHREAD_MUTEX_INITIALIZER;
	nwsLockType *thisLock;

	if (Lock == NULL) {
		return 0;
	}
	if (*Lock == NULL) {
		/* let's play it safe: let's do one mutex at the time (in
		 * case multiple threads are running on the same lock */
	 	if (pthread_mutex_lock(&nwsLock) != 0) {
			return 0;
		}
		thisLock = MALLOC(sizeof(nwsLockType), 0);
		if (thisLock == NULL) {
			fprintf(stderr, "GetNWSLock: out of memory\n");
			return 0;
		}
		*Lock = thisLock;		/* save the lock structure */
		pthread_mutex_init(&thisLock->lock, NULL);
		pthread_cond_init(&thisLock->cond, NULL);
		thisLock->value = 0;

	 	if (pthread_mutex_unlock(&nwsLock) != 0) {
			return 0;
		}
	}
	thisLock = *Lock;
	ret = pthread_mutex_lock(&thisLock->lock);
	if (ret != 0) {
		fprintf(stderr, "GetNWSLock: Unable to lock (errno = %d)!\n", ret);
		return 0;
	}
	while (thisLock->value == 1) {
		pthread_cond_wait(&thisLock->cond, &thisLock->lock);
	}
	thisLock->value = 1;
	ret = pthread_mutex_unlock(&thisLock->lock);
	if  (ret != 0) {
		fprintf(stderr, "GetNWSLock: Unable to unlock (errno = %d)!\n", ret);
		return 0;
	}
#endif
	return 1;
}

int 
ReleaseNWSLock(void **Lock) {
#ifdef HAVE_PTHREAD_H
	nwsLockType *thisLock;
	int ret;

	if (Lock == NULL) {
		/* if we don't have lock, there's a problem */
		fprintf(stderr, "ReleaseNWSLock: non-existing lock!\n");
		return 0;
	}
	thisLock = *Lock;
	if (pthread_mutex_lock(&thisLock->lock) != 0) {
		fprintf(stderr, "ReleaseNWSLock: Unable to lock (errno = %d)!\n", errno);
		return 0;
	}
	thisLock->value = 0;
	pthread_cond_signal(&thisLock->cond);
	ret = pthread_mutex_unlock(&thisLock->lock);
	if (ret != 0) {
		fprintf(stderr, "ReleaseNWSLock: Unable to unlock (errno = %d)!\n", ret);
		return 0;
	}
#endif
	return 1;
}

double
MicroTime(void) {
	struct timeval tv;
	double tmp;


	if (gettimeofday(&tv, NULL) == -1) {
		fprintf(stderr, "MicroTime: gettimeofday failed!\n");
		tmp = -1;
	} else {
		tmp = (double)(tv.tv_sec) * 1000000 + (double)tv.tv_usec;
	}

	return tmp;
}


void
ReleaseSignal(int sig) {
#ifdef HAVE_SIGRELSE
	sigrelse(sig);
#else
	sigset_t set, oset;
	sigemptyset(&set);
	sigaddset(&set, sig);
	sigprocmask(SIG_UNBLOCK, &set, &oset);
#endif
}


#ifdef USE_ALARM_SIGNAL
RETSIGTYPE
OsutilTimeOut(int sig) {
	WARN("Send/Receive timeout\n");
}
#endif


/* It should be thread safe (alarm is thread safe) */
void
SetRealTimer(unsigned int numberOfSecs) {
#ifdef USE_ALARM_SIGNAL
	static RETSIGTYPE (*was)(int) = NULL;
	RETSIGTYPE (*tmp)(int);

	if (numberOfSecs > 0) {
#ifdef HAVE_SIGINTERRUPT
		/* just interrupt a system call upon receipt of interrupt */
		siginterrupt(SIGALRM, 1);
#endif
		/* set the signal */
		tmp = signal(SIGALRM, OsutilTimeOut);
		if  (tmp == SIG_ERR) {
			WARN("SetRealTimer: call to signale failed!\n");
		} else {
			GetNWSLock(&lock);
			/* save the old pointer */
			was = tmp;
			ReleaseNWSLock(&lock);
		}
	} else {
		/* reset the signal */
		GetNWSLock(&lock);
		tmp = was;
		was = NULL;
		ReleaseNWSLock(&lock);
		if (tmp != NULL) {
			if (signal(SIGALRM, tmp) == SIG_ERR) {
				WARN("SetRealTimer: call to signale failed!\n");
			}
		}
	}

	alarm(numberOfSecs);
#endif
}

void *
PortabilityMalloc(size_t size, int quit) {
	void *tmp;

	tmp = malloc((size) == 0 ? 1 : (size));
	if (tmp == NULL && quit) {
		ABORT("out of memory\n");
	}

	return tmp;
}

void *
PortabilityRealloc(void *ptr, size_t size, int quit) {
	void *tmp;

	tmp = NULL;	/* make some compiler happy */

	/* all of this to be sure that realloc won't choke on NULL or
	 * zero pointers */
	if (size > 0) {
		if (ptr == NULL) {
			/* normal malloc */
			tmp =  PortabilityMalloc(size, quit);
		} else {
			/* normal realloc */
#ifdef HAVE_REALLOC
			tmp =  realloc(ptr, size);
			if (tmp == NULL && quit) {
				ABORT("out of memory\n");
			}
#else
			tmp =  PortabilityMalloc(size, quit);
			memcpy(tmp, ptr, size);
			free(ptr);
#endif

		}
	} else if (size == 0) {
		/* size is 0: free the memory */
		if (ptr != NULL) {
			free(ptr);
		}
		tmp = NULL;
	}

	return tmp;
}

pid_t
WaitChild() {
	pid_t pid = 0;
	int done = 0;

#ifdef HAVE_WAITPID
	do {
		pid = waitpid(-1, NULL, WNOHANG);
		if (pid != -1 || errno != EINTR) {
			done = 1;
		}
	} while (!done);
	
	/* the function ignores the error: we'll have few zombies */
	if (pid < 0) {
		pid = 0;
	}
#else 
	ERROR("WaitChild: waitpid non available on this system!\n");
#endif

	return pid;
}


int
PortabilitySelect(	int n,
			fd_set *readfs,
			fd_set *writefs,
			fd_set *exceptfs,
			struct timeval *timeout,
			int *err) {
	struct timeval tout, *p;
	int ret, tmp;
	double when, now;

#ifndef HAVE_SELECT
	/* what am I supposed to do?? */
	fprintf(stderr, "This system doesn't have select available??\n");
	exit(1);
#endif

	now = MicroTime();
	if (timeout != NULL)  {
		/* save the timeout (at least linux complain that you
		 * cannot trust tout after select returns) */
		when = now + timeout->tv_sec*1000000 + timeout->tv_usec;
		p = &tout;
	} else {
		when = 0;
		p = NULL;
	}

	/* loop through to avoid EINTR */
	do {
		if (timeout != NULL) {
			if (when < now) {
				/* ran out of time */
				ret = 0;
				tmp = ETIMEDOUT;
				break;
			}

			/* set the timeout */
			tout.tv_sec = (unsigned long)(when - now)/1000000;
			tout.tv_usec = (unsigned long)(when - now - (double)tout.tv_sec*1000000);
		}

		ret = select(n, readfs, writefs, exceptfs, p);
		tmp = errno;

		if (ret != -1 || (ret == -1 && tmp != EINTR)) {
			when = -1;
		}
		
		/* new time */
		now = MicroTime();
	} while (when >= now);

	if (err != NULL) {
		*err = tmp;
	}

	return ret;
}

