/*
   Copyright 2003-2006 The Savonet Team
   Some parts from oggenc/oggdec, (c) 2000-2002 Michael Smith
   Some parts from ices2 (c) 2001 Michael Smith <msmith@labyrinth.net.au>
   See COPYRIGHT file for details.

   This file is part of Ocaml-vorbis.

   Ocaml-vorbis is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   Ocaml-vorbis is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Ocaml-vorbis; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
   */

/**
 * Libvorbis bindings for OCaml.
 *
 * @author Samuel Mimram, Julien Cristau, David Baelde
 */

/* $Id: vorbis_stubs.c 2693 2006-10-09 17:04:50Z smimram $ */

/*
TODO:
- handle errors
- advanced encoder options
*/

#define CAML_NAME_SPACE
#include <caml/alloc.h>
#include <caml/callback.h>
#include <caml/custom.h>
#include <caml/fail.h>
#include <caml/memory.h>
#include <caml/misc.h>
#include <caml/mlvalues.h>
#include <caml/signals.h>

#include <vorbis/vorbisfile.h>
#include <vorbis/vorbisenc.h>
#include <vorbis/codec.h>

#include <string.h>
#include <stdio.h>
#include <time.h>

#include "utf8.h"

static void raise_err(int ret)
{
  switch(ret)
  {
    case OV_HOLE:
      caml_raise_constant(*caml_named_value("vorbis_exn_hole_in_data"));

    case OV_EREAD:
      caml_raise_constant(*caml_named_value("vorbis_exn_read_error"));

    case OV_EFAULT:
      caml_raise_constant(*caml_named_value("vorbis_exn_internal_fault"));

    case OV_ENOTVORBIS:
      caml_raise_constant(*caml_named_value("vorbis_exn_not_vorbis"));

    case OV_EBADHEADER:
      caml_raise_constant(*caml_named_value("vorbis_exn_bad_header"));

    case OV_EVERSION:
      caml_raise_constant(*caml_named_value("vorbis_exn_version_mismatch"));

    case OV_EBADLINK:
      caml_raise_constant(*caml_named_value("vorbis_exn_bad_link"));

    default:
      caml_raise_constant(*caml_named_value("vorbis_exn_unknown_error"));
  }
}

static void check_err(int ret)
{
  if (ret < 0)
    raise_err(ret);
}

typedef struct myvorbis__encoder_t
{
  vorbis_info vi;
  vorbis_dsp_state vd;
  vorbis_block vb;
  ogg_stream_state os;
  ogg_page og;
  ogg_packet op;

  int channels;
  int in_channels;
  int samplefreq;
  int in_samplefreq;
  int in_samplesize;
  int in_bigendian;
  int serial;
} myvorbis_encoder_t;

#define Encoder_val(v) (*((myvorbis_encoder_t**)Data_custom_val(v)))

static void finalize_myvorbis_encoder(value v_e)
{
  myvorbis_encoder_t *ve = Encoder_val(v_e);

  ogg_stream_clear(&ve->os);
  vorbis_block_clear(&ve->vb);
  vorbis_dsp_clear(&ve->vd);

  vorbis_info_clear(&ve->vi);
  free(ve);
}

static struct custom_operations encoder_ops =
{
  "ocaml_vorbis_encoder",
  finalize_myvorbis_encoder,
  custom_compare_default,
  custom_hash_default,
  custom_serialize_default,
  custom_deserialize_default
};

inline static int int_of_option(value v)
{
  if(Is_block(v))
    return Int_val(Field(v, 0));
  else
    return -1;
}

inline static char* string_of_option(value v)
{
  if(Is_block(v))
    return String_val(Field(v, 0));
  else
    return NULL;
}

#define copy_buffer(dst, b, len) \
        dst = caml_alloc_string(len); \
        memmove(String_val(dst), b, len);

inline static char* strncat_safe(char* dest, char* src, int pos, int len)
{
  dest = (char*)realloc((void*)dest, sizeof(char)*(pos+len));
  memcpy(dest+pos, src, len);
  return dest;
}

static void add_tag(vorbis_comment *vc, char *name, char *val)
{
  char *utf8;
  if(utf8_encode(val, &utf8) >= 0)
  {
    if(name == NULL)
      vorbis_comment_add(vc, utf8);
    else
      vorbis_comment_add_tag(vc, name, utf8);
    free(utf8);
  }
  else
    caml_raise_with_arg(*caml_named_value("vorbis_exn_utf8_failure"), caml_copy_string(val));
}

static void build_comments(vorbis_comment *vc,
    char *artist, char *album, char *title, char *tracknum, char* comment,
    char *date, char *genre)
{
  vorbis_comment_init(vc);

  if (title)
    add_tag(vc, "title", title);
  if (artist)
    add_tag(vc, "artist", artist);
  if (genre)
    add_tag(vc, "genre", genre);
  if (date)
    add_tag(vc, "date", date);
  if (album)
    add_tag(vc, "album", album);
  if (tracknum)
    add_tag(vc, "tracknumber", tracknum);
  if (comment)
    add_tag(vc, "comment", comment);
  add_tag(vc, "encoder", "ocaml-vorbis by the savonet team (http://savonet.sf.net/)");
}


static value ocaml_vorbis_get_header_frame(value encoder, value title, value artist, value genre, value date, value album, value tracknum, value comment)
{
  value s;
  char *ret = NULL;
  int ret_len = 0;
  myvorbis_encoder_t *ve = Encoder_val(encoder);

  /* Now, build the three header packets and send through to the stream
     output stage (but defer actual file output until the main encode loop). */
  {
    ogg_packet header_main;
    ogg_packet header_comments;
    ogg_packet header_codebooks;
    vorbis_comment comments;

    build_comments(&comments, string_of_option(artist), string_of_option(album), string_of_option(title), string_of_option(tracknum), string_of_option(comment), string_of_option(date), string_of_option(genre));
    comments.vendor = malloc(strlen("ocaml-vorbis") + 1);
    strcpy(comments.vendor, "ocaml-vorbis");

    /* Build the packets */
    vorbis_analysis_headerout(&ve->vd, &comments,
        &header_main, &header_comments, &header_codebooks);

    /* And stream them out */
    ogg_stream_packetin(&ve->os, &header_main);
    ogg_stream_packetin(&ve->os, &header_comments);
    ogg_stream_packetin(&ve->os, &header_codebooks);

    while(ogg_stream_flush(&ve->os, &ve->og))
    {
      ret = (char*)realloc(ret, ret_len + (&ve->og)->header_len + (&ve->og)->body_len);
      memcpy(ret + ret_len, (&ve->og)->header, (&ve->og)->header_len);
      memcpy(ret + ret_len + (&ve->og)->header_len, (&ve->og)->body, (&ve->og)->body_len);
      ret_len += (&ve->og)->header_len + (&ve->og)->body_len;
    }

    vorbis_comment_clear(&comments);
  }

  copy_buffer(s, ret, ret_len);
  free(ret);

  return s;
}

CAMLprim value ocaml_vorbis_create_encoder(value params, value title, value artist, value genre, value date, value album, value tracknum, value comment)
{
  CAMLparam5(params, title, artist, genre, date);
  CAMLxparam3(album, tracknum, comment);
  CAMLlocal2(encoder, block);

  myvorbis_encoder_t *ve;
  int quality_set = 0;

  int bitrate = int_of_option(Field(params, 0));
  int min_bitrate = int_of_option(Field(params, 1));
  int max_bitrate = int_of_option(Field(params, 2));
  float quality = Double_val(Field(params, 3));
  int channels = Int_val(Field(params, 4));
  int sample_freq = int_of_option(Field(params, 5));
  int managed = Bool_val(Field(params, 6));

  int in_channels = Int_val(Field(params, 7));
  int in_samplefreq = Int_val(Field(params, 8));
  int in_samplesize = Int_val(Field(params, 9));
  int in_bigendian = Int_val(Field(params, 10));

  /* TODO: in argument? */
  int serialno;
  srand(time(NULL));
  serialno = rand();

  /* Frequency conversion is performed only for these simple cases */
  if(in_samplefreq % sample_freq != 0)
    caml_raise_constant(*caml_named_value("vorbis_exn_invalid_sample_freq"));
  if((channels != 1 && channels != 2) || (in_channels != 1 && in_channels != 2))
    caml_raise_constant(*caml_named_value("vorbis_exn_invalid_channels"));
  if(in_samplesize != 8 && in_samplesize != 16)
    caml_raise_constant(*caml_named_value("vorbis_exn_invalid_samplesize"));

  /* We need a pointer to a myvorbis_encoder_t which won't move since vorbis seems to want to update some of the fields */
  encoder = caml_alloc_custom(&encoder_ops, sizeof(myvorbis_encoder_t*), 0, 1);
  ve = malloc(sizeof(myvorbis_encoder_t));
  Encoder_val(encoder) = ve;

  ve->channels = channels;
  ve->in_channels = in_channels;
  ve->in_samplesize = in_samplesize;
  ve->samplefreq = sample_freq;
  ve->in_samplefreq = in_samplefreq;
  ve->in_bigendian = in_bigendian;
  ve->serial = serialno;

  /* If we had no quality or bitrate spec at all from the user, use
     the default quality with no management */
  if (bitrate < 0 && min_bitrate < 0 && max_bitrate < 0)
    quality_set = 1;

  /* Have vorbisenc choose a mode for us */
  vorbis_info_init(&ve->vi);
  if(quality_set > 0)
  {
    if (vorbis_encode_setup_vbr(&ve->vi, channels, sample_freq, quality))
    {
      vorbis_info_clear(&ve->vi);
      caml_raise_constant(*caml_named_value("vorbis_exn_invalid_quality"));
    }

    /* Do we have optional hard quality restrictions? */
    if (max_bitrate > 0 || min_bitrate > 0)
    {
      struct ovectl_ratemanage_arg ai;

      vorbis_encode_ctl(&ve->vi, OV_ECTL_RATEMANAGE_GET, &ai);
      ai.bitrate_hard_min = min_bitrate * 1000;
      ai.bitrate_hard_max = max_bitrate * 1000;
      ai.management_active = 1;
      vorbis_encode_ctl(&ve->vi, OV_ECTL_RATEMANAGE_SET, &ai);
    }
  }
  else
  {
    if (vorbis_encode_setup_managed(&ve->vi, channels, sample_freq,
          max_bitrate>0?max_bitrate*1000:-1,
          bitrate*1000,
          min_bitrate>0?min_bitrate*1000:-1))
    {
      vorbis_info_clear(&ve->vi);
      caml_raise_constant(*caml_named_value("vorbis_exn_invalid_parameters"));
    }
  }

  if (managed && bitrate < 0)
  {
    vorbis_encode_ctl(&ve->vi, OV_ECTL_RATEMANAGE_AVG, NULL);
  }
  else if(!managed)
  {
    /* Turn off management entirely (if it was turned on). */
    vorbis_encode_ctl(&ve->vi, OV_ECTL_RATEMANAGE_SET, NULL);
  }

  /* TODO: advanced encoder options */

  vorbis_encode_setup_init(&ve->vi);
  vorbis_analysis_init(&ve->vd, &ve->vi);
  vorbis_block_init(&ve->vd, &ve->vb);

  ogg_stream_init(&ve->os, serialno);

  block = caml_alloc_tuple(2);
  Store_field(block, 0, encoder);
  /* TODO : clean this */
  Store_field(block, 1, ocaml_vorbis_get_header_frame(encoder, title, artist, genre, date, album, tracknum, comment));

  CAMLreturn(block);
}

CAMLprim value ocaml_vorbis_create_encoder_byte(value* argv, int argc)
{
  return ocaml_vorbis_create_encoder(argv[0], argv[1], argv[2], argv[3], argv[4], argv[5], argv[6], argv[7]);
}

/* Encode audio from buf, return the available ogg/vorbis data */
CAMLprim value ocaml_vorbis_encode_buffer(value v_e, value _buf, value _ofs, value _len)
{
  CAMLparam2(v_e, _buf);
  value s;
  myvorbis_encoder_t *ve = Encoder_val(v_e);
  int len = Int_val(_len);
  int ofs = Int_val(_ofs);
  char* in = (char*)malloc(len*sizeof(char));

  if (ofs + len > caml_string_length(_buf))
    caml_invalid_argument("buffer too small (Vorbis.encode_buffer)");
  memcpy(in, String_val(_buf)+ofs, len*sizeof(char));

  /* Ask vorbis the adress where to put the new data.
   * TODO some data can be lost when downsampling: with input at 22050Hz
   * and output at 44100Hz, one sample is lost if the input contains an odd
   * number of samples. It shouldn't hurt too much. */
  int realsamples = (len * 8 * ve->samplefreq) / (ve->in_samplesize * ve->in_channels * ve->in_samplefreq);
  float** buffer = vorbis_analysis_buffer(&ve->vd, realsamples);
  int i, j, c;
  int downsample = ve->in_samplefreq / ve->samplefreq ;

  /* stereo2mono : shall we mix two channels in one ?
   * Set to 2 if doing so, otherwise it's set to 1. */
  int s2m = (ve->in_channels==2 && ve->channels==1?2:1);

  /* Then we'll fill [ret] with the new ogg/vorbis data */
  char *ret = NULL;
  int ret_len = 0;
  int eos = 0;

  /* For now, we only work on the C side, and let other Caml threads run */
  caml_enter_blocking_section();

  /* Sample conversion, using the assertion size == 8 or 16.
   * At the same time we perform downsampling, and channel conversions.
   * Stereo2mono is made in two passes: on the first one (c=0), we SET
   * the buffer to half the value of the first channel, on the second
   * one (c=1) we add half of the second channel.
   * SELECT(j) is [c] if [s2m], otherwise it's [j], or 0 if [j] is out
   * of bounds. This last case happens when encoding mono to stereo. */
#define SET(A,B) (s2m==1?(A=B):(c==0?(A=B/2):(A+=B/2)))
#define SELECT(A) (s2m==2?c:((A)<ve->in_channels?(A):0))
  for(c=0;c<s2m;c++)
    if(ve->in_samplesize == 8)
    {
      unsigned char *bufu = (unsigned char *)in;
      for(i = 0; i < realsamples; i++)
        for(j = 0; j < ve->channels; j++)
          SET(buffer[j][i],
              ((int)(bufu[i * ve->in_channels * downsample
                     + SELECT(j)]) - 128)
              / 128.0f);
    }
    else
      if(!ve->in_bigendian)
        for(i = 0; i < realsamples; i++)
          for(j = 0; j < ve->channels; j++)
            SET(buffer[j][i],
                ((in[i * 2 * ve->in_channels * downsample
                  + 2 * SELECT(j) + 1] << 8) |
                 (in[i * 2 * ve->in_channels * downsample
                  + 2 * SELECT(j)] & 0xff)) / 32768.0f);
      else
        for(i = 0; i < realsamples; i++)
          for(j = 0; j < ve->channels; j++)
            SET(buffer[j][i],
                ((in[i * 2 * ve->in_channels * downsample
                  + 2 * SELECT(j)] << 8) |
                 (in[i * 2 * ve->in_channels * downsample
                  + 2 * SELECT(j) + 1] & 0xff)) / 32768.0f);

  /* Tell the library how many samples (per channel) we wrote
     into the supplied buffer */
  vorbis_analysis_wrote(&ve->vd, realsamples);

  /* While we can get enough data from the library to analyse, one
     block at a time... */
  while(vorbis_analysis_blockout(&ve->vd, &ve->vb) == 1)
  {
    /* Do the main analysis, creating a packet */
    vorbis_analysis(&ve->vb, NULL);
    vorbis_bitrate_addblock(&ve->vb);

    while(vorbis_bitrate_flushpacket(&ve->vd, &ve->op))
    {
      /* Add packet to bitstream */
      ogg_stream_packetin(&ve->os, &ve->op);

      /* If we've gone over a page boundary, we can do actual output,
         so do so (for however many pages are available) */
      while(!eos)
      {
        if(!ogg_stream_pageout(&ve->os, &ve->og)) break;

        ret = (char*)realloc(ret, ret_len + (&ve->og)->header_len + (&ve->og)->body_len);
        memcpy(ret + ret_len, (&ve->og)->header, (&ve->og)->header_len);
        memcpy(ret + ret_len + (&ve->og)->header_len, (&ve->og)->body, (&ve->og)->body_len);
        ret_len += (&ve->og)->header_len + (&ve->og)->body_len;

        if(ogg_page_eos(&ve->og)) eos = 1;
      }
    }
  }

  /* Back to the Caml world */
  caml_leave_blocking_section();

  copy_buffer(s, ret, ret_len);
  free(ret);
  free(in);

  CAMLreturn(s);
}


/* This should be malloced since we might want to register *_func as global root. */
typedef struct myvorbis__dec_file_t
{
  OggVorbis_File *ovf;
  int sample_size;
  int big_endian;
  int sign;
  int bitstream;
  value read_func;
  value seek_func;
  value close_func;
  value tell_func;
} myvorbis_dec_file_t;

#define Decfile_val(v) (*((myvorbis_dec_file_t**)Data_custom_val(v)))

static void finalize_dec_file(value df)
{
  free(Decfile_val(df));
}

static struct custom_operations decfile_ops =
{
  "ocaml_vorbis_decfile",
  finalize_dec_file,
  custom_compare_default,
  custom_hash_default,
  custom_serialize_default,
  custom_deserialize_default
};

CAMLprim value ocaml_vorbis_open_dec_file(value fd, value params)
{
  CAMLparam1(params);
  value block;
  int ret;
  FILE *in = NULL;
  myvorbis_dec_file_t *df = malloc(sizeof(myvorbis_dec_file_t));

  if(!(in = fdopen(Int_val(fd), "rb")))
    caml_raise_constant(*caml_named_value("vorbis_exn_could_not_open_file"));
  df->ovf = (OggVorbis_File*)malloc(sizeof(OggVorbis_File));
  df->sample_size = Int_val(Field(params, 0));
  df->big_endian = Bool_val(Field(params, 1));
  df->sign = Bool_val(Field(params, 2));
  df->bitstream = 0;
  df->read_func = (value)NULL;
  df->seek_func = (value)NULL;
  df->close_func = (value)NULL;
  df->tell_func = (value)NULL;

  caml_enter_blocking_section();
  ret = ov_open(in, df->ovf, NULL, 0);
  caml_leave_blocking_section();

  if (ret < 0)
  {
    fclose(in);
    free(df->ovf);
    raise_err(ret);
  }

  block = caml_alloc_custom(&decfile_ops, sizeof(myvorbis_dec_file_t*), 0, 1);
  Decfile_val(block) = df;

  CAMLreturn(block);
}

static size_t read_func_cb(void *ptr, size_t size, size_t nmemb, void *datasource)
{
  myvorbis_dec_file_t *df = datasource;
  value buf;
  int len;
  caml_leave_blocking_section();
  buf = caml_callback(df->read_func, Val_int(size*nmemb));
  len = caml_string_length(buf);
  memcpy(ptr, String_val(buf), len);
  caml_enter_blocking_section();
  return len;
}

static int seek_func_cb(void *datasource, ogg_int64_t offset, int whence)
{
  myvorbis_dec_file_t *df = datasource;
  int ret;
  caml_leave_blocking_section();
  ret = Int_val(caml_callback(df->seek_func, Val_int(offset)));
  caml_enter_blocking_section();
  return ret;
}

static int close_func_cb(void *datasource)
{
  myvorbis_dec_file_t *df = datasource;
  caml_leave_blocking_section();
  caml_callback(df->close_func, Val_unit);
  caml_enter_blocking_section();
  return 0;
}

static long tell_func_cb(void *datasource)
{
  myvorbis_dec_file_t *df = datasource;
  int ret;
  caml_leave_blocking_section();
  ret = Int_val(caml_callback(df->tell_func, Val_unit));
  caml_enter_blocking_section();
  return ret;
}

static ov_callbacks callbacks =
{
  .read_func = read_func_cb,
  .seek_func = seek_func_cb,
  .close_func = close_func_cb,
  .tell_func = tell_func_cb
};

CAMLprim value ocaml_vorbis_open_dec_stream(value read_func, value seek_func, value close_func, value tell_func, value params)
{
  CAMLparam5(read_func, seek_func, close_func, tell_func, params);
  CAMLlocal1(block);
  int ret = 0;
  myvorbis_dec_file_t *df;

  block = caml_alloc_custom(&decfile_ops, sizeof(myvorbis_dec_file_t*), 0, 1);
  df = malloc(sizeof(myvorbis_dec_file_t));
  Decfile_val(block) = df;

  df->ovf = (OggVorbis_File*)malloc(sizeof(OggVorbis_File));
  df->sample_size = Int_val(Field(params, 0));
  df->big_endian = Bool_val(Field(params, 1));
  df->sign = Bool_val(Field(params, 2));
  df->bitstream = 0;
  df->read_func = read_func;
  caml_register_global_root(&df->read_func);
  df->seek_func = seek_func;
  caml_register_global_root(&df->seek_func);
  df->close_func = close_func;
  caml_register_global_root(&df->close_func);
  df->tell_func = tell_func;
  caml_register_global_root(&df->tell_func);
  caml_enter_blocking_section();
  ret = ov_open_callbacks(df, df->ovf, NULL, 0, callbacks);
  caml_leave_blocking_section();

  if(ret < 0)
  {
    free(df->ovf);
    raise_err(ret);
  }

  CAMLreturn(block);
}

/* TODO: convert file format? */
CAMLprim value ocaml_vorbis_decode(value d_f, value buf_, value ofs_, value len_)
{
  CAMLparam2(d_f, buf_);

  myvorbis_dec_file_t *df = Decfile_val(d_f);
  int ret = 0;
  int ofs = Int_val(ofs_);
  int len = Int_val(len_);
  char* buf = malloc(len);

  if (ofs + len > caml_string_length(buf_))
    caml_raise_constant(*caml_named_value("vorbis_exn_invalid_parameters"));

  // We have to make sure that when a callback is called, the ocaml master lock
  // has been released.  Callbacks are responsible for taking it back if they
  // need to call ocaml code.
  caml_enter_blocking_section();
  ret = ov_read(df->ovf, buf, len, df->big_endian, df->sample_size / 8, df->sign, &df->bitstream);
  caml_leave_blocking_section();

  if (ret == 0)
  {
    free(buf);
    caml_raise_end_of_file();
  }
  if (ret < 0)
  {
    free(buf);
    raise_err(ret);
  }
  memcpy(String_val(buf_) + ofs, buf, len);
  free(buf);

  CAMLreturn(Val_int(ret));
}


CAMLprim value ocaml_vorbis_close_dec_file(value d_f)
{
  CAMLparam1(d_f);
  myvorbis_dec_file_t* df = Decfile_val(d_f);

  if(df->read_func)
  {
    caml_remove_global_root(&df->read_func);
    caml_remove_global_root(&df->seek_func);
    caml_remove_global_root(&df->close_func);
    caml_remove_global_root(&df->tell_func);
  }
  caml_enter_blocking_section();
  ov_clear(df->ovf);
  caml_leave_blocking_section();
  free(df->ovf);

  CAMLreturn(Val_unit);
}

CAMLprim value ocaml_vorbis_utf8_set_charset(value charset)
{
  convert_set_charset(String_val(charset));
  return Val_unit;
}

CAMLprim value ocaml_vorbis_utf8_decode(value string)
{
  value ret;
  char* utf8;
  if(utf8_decode(String_val(string), &utf8) >= 0)
  {
    ret = caml_copy_string(utf8);
    free(utf8);
    return ret;
  }
  caml_raise_with_arg(*caml_named_value("vorbis_exn_utf8_failure"), string);
}

CAMLprim value ocaml_vorbis_utf8_encode(value string)
{
  value ret;
  char* nonutf8;
  if(utf8_encode(String_val(string), &nonutf8) >= 0)
  {
    ret = caml_copy_string(nonutf8);
    free(nonutf8);
    return ret;
  }
  caml_raise_with_arg(*caml_named_value("vorbis_exn_utf8_failure"), string);
}

CAMLprim value ocaml_vorbis_encoder_reset(value v_e, value title, value artist, value genre, value date, value album, value tracknum, value comment)
{
  CAMLparam5(v_e, title, artist, genre, date);
  CAMLxparam3(album, tracknum, comment);

  myvorbis_encoder_t *s = Encoder_val(v_e);

  ogg_packet op;
  ogg_page og;
  CAMLlocal1(ans);
  char* ret = NULL;
  int size = 0;

  /* tell libvorbis that this is the end of the stream */
  vorbis_analysis_wrote(&s->vd, 0);

  while(vorbis_analysis_blockout(&s->vd, &s->vb)==1)
  {
    vorbis_analysis(&s->vb, NULL);
    vorbis_bitrate_addblock(&s->vb);
    while(vorbis_bitrate_flushpacket(&s->vd, &op))
      ogg_stream_packetin(&s->os, &op);
  }

  /* flush the last pages in a buffer */
  while(ogg_stream_pageout(&s->os, &og)<=0?0:1) {
    ret = strncat_safe(ret, (char*)og.header, size, og.header_len);
    size += og.header_len;
    ret = strncat_safe(ret, (char*)og.body, size, og.body_len);
    size += og.body_len;
  }

  /* clear the encoder */
  ogg_stream_clear(&s->os);
  vorbis_block_clear(&s->vb);
  vorbis_dsp_clear(&s->vd);

  /* build the new vorbis comments */
  do {
    vorbis_comment comments;
    ogg_packet h1,h2,h3;

    build_comments(&comments, string_of_option(artist), string_of_option(album), string_of_option(title), string_of_option(tracknum), string_of_option(comment), string_of_option(date), string_of_option(genre));

    comments.vendor = "ocaml-vorbis";

    vorbis_analysis_init(&s->vd, &s->vi);
    vorbis_block_init(&s->vd, &s->vb);

    ogg_stream_init(&s->os, ++(s->serial));
    vorbis_analysis_headerout(&s->vd, &comments, &h1,&h2,&h3);
    ogg_stream_packetin(&s->os, &h1);
    ogg_stream_packetin(&s->os, &h2);
    ogg_stream_packetin(&s->os, &h3);

  } while(0);

  /* add the headers to the returned buffer */
  while(ogg_stream_flush(&s->os, &s->og))
  {
    ret = (char*)realloc(ret, size + (&s->og)->header_len + (&s->og)->body_len);
    memcpy(ret + size, (&s->og)->header, (&s->og)->header_len);
    memcpy(ret + size + (&s->og)->header_len, (&s->og)->body, (&s->og)->body_len);
    size += (&s->og)->header_len + (&s->og)->body_len;
  }

  copy_buffer(ans, ret, size);
  free(ret);
  CAMLreturn(ans);
}

value ocaml_vorbis_encoder_reset_byte(value *argv, int argc)
{
  return ocaml_vorbis_encoder_reset(argv[0], argv[1], argv[2], argv[3], argv[4], argv[5], argv[6], argv[7]);
}

CAMLprim value ocaml_vorbis_get_dec_file_bitstream(value d_f)
{
  myvorbis_dec_file_t *df = Decfile_val(d_f);
  return Val_int(df->bitstream);
}

CAMLprim value ocaml_vorbis_get_dec_file_info(value d_f)
{
  CAMLparam1(d_f);
  CAMLlocal1(ans);
  myvorbis_dec_file_t *df = Decfile_val(d_f);
  vorbis_info *vi = df->ovf->vi;

  ans = caml_alloc_tuple(9);
  Store_field(ans, 0, Val_int(vi->version));
  Store_field(ans, 1, Val_int(vi->channels));
  Store_field(ans, 2, Val_int(vi->rate));
  /* TODO */
  Store_field(ans, 3, Val_int(0));
  Store_field(ans, 4, Val_int(0));
  Store_field(ans, 5, Val_int(0));
  Store_field(ans, 6, Val_int(0));
  Store_field(ans, 7, Val_int(0));
  Store_field(ans, 8, Val_int(0));

  CAMLreturn(ans);
}

CAMLprim value ocaml_vorbis_get_dec_file_comments(value d_f, value link_)
{
  CAMLparam2(d_f, link_);
  CAMLlocal2(ans, cmts);

  myvorbis_dec_file_t *df;
  int link;
  int i;
  vorbis_comment *vc;

  df = Decfile_val(d_f);
  if (Is_block(link_))
    link = Int_val(Field(link_, 0));
  else
    link = -1;
  caml_enter_blocking_section();
  vc = ov_comment(df->ovf, link);
  caml_leave_blocking_section();

  if (!vc)
  {
    /* TODO: better error */
    caml_raise_constant(*caml_named_value("vorbis_exn_unknown_error"));
  }

  cmts = caml_alloc_tuple(vc->comments);
  for (i = 0; i < vc->comments; i++)
  {
    Store_field(cmts, i, caml_copy_string(vc->user_comments[i]));
  }
  ans = caml_alloc_tuple(2);
  Store_field(ans, 0, caml_copy_string(vc->vendor));
  Store_field(ans, 1, cmts);

  CAMLreturn(ans);
}
