// Copyright 2007 Deutsches Forschungszentrum fuer Kuenstliche Intelligenz
// or its licensors, as applicable.
//
// You may not use this file except under the terms of the accompanying license.
//
// Licensed under the Apache License, Version 2.0 (the "License"); you
// may not use this file except in compliance with the License. You may
// obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Project:
// File: logging.cc
// Purpose: logging in HTML file different kind of information
// Responsible: rangoni
// Reviewer: rangoni
// Primary Repository:
// Web Sites: www.iupr.org, www.dfki.de, www.ocropus.org

#include <stdarg.h>
#include <time.h>
#include "colib.h"
#include "imgio.h"
#include "narray-io.h"
#include "logger.h"
#include "imglib.h"
#include "ocr-utils.h"
#include "sysutil.h"
#include "ocr-segmentations.h"


using namespace colib;
using namespace iulib;
using namespace ocropus;

param_float graph_scale("graph_scale",1.,"scale factor for graph ploting if error_rates is enabled");
param_bool merge_lr("merge_lr",true,"plot learning rate with error rates, otherwise in another plot");

namespace {
    struct Log {
        narray<strbuf> enabled_logs;
        strbuf dir;
        stdio file;
    };

    // sorry for all that.
    Log *the_log = NULL;
    Log *get_log() {
        if(!the_log)
            the_log = new Log();
        return the_log;
    }

    struct DeleteLog {
        ~DeleteLog() {
            if(the_log)
                delete the_log;
        }
    } delete_log;

    int indent_level = 0;
    int image_counter = 0;
    bool self_logging;

    /// Returns true if the given specification chunk turns on a log with the given name.
    /// The simplest way would be to use !strcmp, but we have an extension:
    /// a spec "X" would turn on a log named "X.Y".
    bool turns_on(const char *spec, const char *name) {
        int spec_len = strlen(spec);
        int name_len = strlen(spec);
        if(spec_len > name_len)
            return false; // then spec is too long to be a prefix of name
        if(strncmp(spec, name, spec_len))
            return false; // then spec can't be a prefix of name
        if(name_len == spec_len)
            return true; // spec == name
        return name[spec_len] == '.';
    }

    void draw_line(intarray &a, int y, int color) {
        if(y < 0 || y >= a.dim(1)) return;
        for(int x = 0; x < a.dim(0); x++)
            a(x, y) = color;
    }

};


namespace ocropus {
    bool Logger::turned_on(const char *name) {
        for(int i = 0; i < get_log()->enabled_logs.length(); i++)
            if(turns_on(get_log()->enabled_logs[i], name))
                return true;
        return false;
    }

    void Logger::init_logging() {
        char *ocrolog = getenv("ocrolog");
        if(!ocrolog)
            return;
        split_string(get_log()->enabled_logs, ocrolog, ":;");
        self_logging = turned_on("logger");
    }

    void Logger::start_logging() {
        if(!!get_log()->file)
            return;
        const char *ocrologdir = getenv("ocrologdir");
        if(!ocrologdir) {
            ocrologdir = "log.ocropus";
        }
        set_logger_directory(ocrologdir);

        fprintf(get_log()->file, "logging turned on for the following loggers:<BR /><UL>\n");
        for(int i = 0; i < get_log()->enabled_logs.length(); i++)
            fprintf(get_log()->file, "    <LI>%s</LI>\n", (char *) get_log()->enabled_logs[i]);
        fprintf(get_log()->file, "</UL>\n");

        time_t rawtime;
        time (&rawtime);
        fprintf(get_log()->file, "Started %s <BR /><BR />\n", ctime (&rawtime));
        fflush(get_log()->file);
    }

    void Logger::putIndent() {
        fprintf(get_log()->file, "[%s] ", (char *) name);
        for(int i = 0; i < indent_level; i++) {
            fprintf(get_log()->file, "&nbsp;&nbsp;");
        }
    }

    stdio Logger::logImage(const char *description) {
        char buf[strlen(get_log()->dir) + 100];
        sprintf(buf, "%s/ocropus-log-%d.png", (char *) get_log()->dir, image_counter);
        putIndent();
        fprintf(get_log()->file, "%s: <IMG SRC=\"ocropus-log-%d.png\"><BR>\n",
                description, image_counter);
        fflush(get_log()->file);
        image_counter++;
        return stdio(buf, "wb");
    }
    stdio Logger::logImageHtml() {
        char buf[strlen(get_log()->dir) + 100];
        sprintf(buf, "%s/ocropus-log-%d.png", (char *) get_log()->dir, image_counter);
        //putIndent();
        fprintf(get_log()->file, "<IMG SRC=\"ocropus-log-%d.png\">",image_counter);
        fflush(get_log()->file);
        image_counter++;
        return stdio(buf, "wb");
    }
    stdio Logger::logImageHtmlBorder() {
        char buf[strlen(get_log()->dir) + 100];
        sprintf(buf, "%s/ocropus-log-%d.png", (char *) get_log()->dir, image_counter);
        //putIndent();
        fprintf(get_log()->file, "<IMG hspace=\"4\" vspace=\"4\" style=\"border-color:#8888FF\" SRC=\"ocropus-log-%d.png\" border=\"1\">",image_counter);
        fflush(get_log()->file);
        image_counter++;
        return stdio(buf, "wb");
    }

    stdio Logger::logText(const char *description) {
        char buf[strlen(get_log()->dir) + 100];
        sprintf(buf, "%s/ocropus-log-%d.txt", (char *) get_log()->dir, image_counter);
        putIndent();
        fprintf(get_log()->file, "<A HREF=\"ocropus-log-%d.txt\">%s</A><BR>\n",
                image_counter, description);
        fflush(get_log()->file);
        image_counter++;
        return stdio(buf, "w");
    }


    Logger::Logger(const char *name) {
        this->name = name;

        if(!get_log()->enabled_logs.length()) {
            init_logging();
        }

        enabled = false;
        for(int i = 0; i < get_log()->enabled_logs.length(); i++) {
            if(turns_on(get_log()->enabled_logs[i], name)) {
                enabled = true;
                break;
            }
        }

        if(enabled || self_logging)
            start_logging();

        // trying to handle gracefully the situation when the log file couldn't be opened
        if(!get_log()->file) {
            enabled = false;
            return;
        }

        if(self_logging)
            fprintf(get_log()->file, "[logger] `%s': %s<BR />\n", (char *) name, enabled ? "enabled": "disabled");
    }

    void Logger::format(const char *format, ...) {
        if(!enabled) return;
        va_list va;
        va_start(va, format);
        putIndent();
        vfprintf(get_log()->file, format, va);
        fprintf(get_log()->file, "<BR />\n");
        va_end(va);
        fflush(get_log()->file);
    }

    void Logger::operator()(const char *s) {
        if(!enabled) return;
        putIndent();
        fprintf(get_log()->file, "%s<BR>\n", s);
        fflush(get_log()->file);
    }
    void Logger::operator()(const char *message, bool val) {
        if(!enabled) return;
        putIndent();
        fprintf(get_log()->file, "%s: %s<BR>\n", message, val ? "true" : "false");
        fflush(get_log()->file);
    }
    void Logger::operator()(const char *message, int val) {
        if(!enabled) return;
        putIndent();
        fprintf(get_log()->file, "%s: %d<BR>\n", message, val);
        fflush(get_log()->file);
    }
    void Logger::operator()(const char *message, double val) {
        if(!enabled) return;
        putIndent();
        fprintf(get_log()->file, "%s: %lf<BR>\n", message, val);
        fflush(get_log()->file);
    }
    void Logger::operator()(const char *message, const char *val) {
        if(!enabled) return;
        putIndent();
        fprintf(get_log()->file, "%s: \"%s\"<BR>\n", message, val);
        fflush(get_log()->file);
    }
    void Logger::operator()(const char *message, nuchar val) {
        if(!enabled) return;
        putIndent();
        fprintf(get_log()->file, "%s: \'%lc\' (hex %x, dec %x)<BR>\n",
                message, val.ord(), val.ord(), val.ord());
        fflush(get_log()->file);
    }
    void Logger::operator()(const char *description, colib::intarray &a) {
        if(!enabled) return;
        if(a.rank() == 2) {
            stdio f = logImage(description);
            write_image_packed(f,a,"png");
        } else {
            stdio f = logText(description);
            text_write(f, a);
        }
    }
    void Logger::recolor(const char *description, colib::intarray &a) {
        if(!enabled) return;
        if(a.rank() == 2) {
            stdio f = logImage(description);
            intarray tmp;
            copy(tmp, a);
            simple_recolor(tmp);
            write_image_packed(f,tmp,"png");
        } else {
            stdio f = logText(description);
            text_write(f, a);
        }
    }
    void Logger::operator()(const char *description, colib::bytearray &a) {
        if(!enabled) return;
        if(a.rank() == 2) {
            stdio f = logImage(description);
            write_png(f, a);
        } else {
            stdio f = logText(description);
            text_write(f, a);
        }
    }
    void Logger::html(colib::bytearray &a) {
        if(!enabled) return;
        if(a.rank() == 2) {
            stdio f = logImageHtml();
            write_png(f, a);
        } else {
            stdio f = logText("error");
            text_write(f, a);
        }
    }
    void Logger::html(const char* s) {
        fprintf(get_log()->file, "%s\n", s);
    }
    void Logger::html_border(colib::bytearray &img) {
        if(!enabled) return;
        if(img.rank() == 2) {
            stdio f = logImageHtmlBorder();
            write_png(f, img);
        } else {
            stdio f = logText("error");
            text_write(f, img);
        }
    }

    void Logger::operator()(const char *description, colib::floatarray &a) {
        if(!enabled) return;
        stdio f = logText(description);
        text_write(f, a);
    }
    void Logger::operator()(const char *message, colib::nustring &val) {
        if(!enabled) return;
        char *buf = val.newUtf8Encode();
        putIndent();
        fprintf(get_log()->file, "%s: nustring(\"%s\")<BR>\n", message, buf);
        fflush(get_log()->file);
        delete[] buf;
    }
    void Logger::html(colib::nustring &val) {
        char *buf = val.newUtf8Encode();
        fprintf(get_log()->file, "%s", buf);
        fflush(get_log()->file);
        delete[] buf;
    }
    void Logger::operator()(const char *description, colib::rectangle &val) {
        if(!enabled) return;
        putIndent();
        fprintf(get_log()->file, "%s: rectangle(%d,%d,%d,%d)<BR>\n",
                description, val.x0, val.y0, val.x1, val.y1);
        fflush(get_log()->file);
    }

    void Logger::operator()(const char *message, colib::IGenericFst &val) {
        if(!enabled) return;
        nustring s;
        val.bestpath(s);
        char *buf = s.newUtf8Encode();
        putIndent();
        fprintf(get_log()->file, "%s: ICharLattice(bestpath: \"%s\")<BR>\n", message, buf);
        fflush(get_log()->file);
        delete[] buf;
    }

    void Logger::operator()(const char *description, void *ptr) {
        if(!enabled) return;
        putIndent();
        fprintf(get_log()->file, "%s: pointer(%p)<BR>\n", description, ptr);
        fflush(get_log()->file);
    }

    void Logger::indent() {
        if(enabled)
            indent_level++;
    }
    void Logger::dedent() {
        if(enabled)
            indent_level--;
    }
    void Logger::operator()(const char *description, bytearray &line_image,
             int baseline, int xheight, int ascender, int descender) {
        if(!enabled) return;
        intarray a;
        copy(a, line_image);
        draw_line(a, ascender, 0x77FF00);
        draw_line(a, descender, 0x00FF00);
        draw_line(a, baseline, 0xFF0000);
        draw_line(a, xheight, 0xFF7700);
        (*this)(description, a);
    }
    void Logger::operator()(const char *description, intarray &line_image,
             int baseline, int xheight, int ascender, int descender) {
        if(!enabled) return;
        intarray a;
        copy(a, line_image);
        ocropus::make_line_segmentation_white(a);
        draw_line(a, ascender, 0x77FF00);
        draw_line(a, descender, 0x00FF00);
        draw_line(a, baseline, 0xFF0000);
        draw_line(a, xheight, 0xFF7700);
        (*this)(description, a);
    }

    const char *get_logger_directory() {
        return (char *) get_log()->dir;
    }

    void set_logger_directory(const char *path) {
        if(!!get_log()->file) {
            fprintf(get_log()->file,
                    "log finished; switching to directory %s\n", path);
        }
        mkdir_if_necessary(path);
        strbuf old_dir;
        if(get_log()->dir)
            old_dir = get_log()->dir;
        get_log()->dir = path;
        strbuf html;
        html = path;
        html += "/index.html";
        get_log()->file = fopen(html,"wt");
        if(!get_log()->file) {
            fprintf(stderr, "unable to open log file `%s' for writing\n",
                    (char *) html);
        }
        fprintf(get_log()->file, "<HTML>\n<HEAD>\n<meta http-equiv=\"content-type\" "
                        "content=\"text/html\"; charset=UTF-8\">\n</HEAD>\n<BODY>\n");
        if(old_dir) {
            fprintf(get_log()->file, "Log continued from %s<P>\n", (char *) old_dir);
        }
    }

    void Logger::display_one_line_graph(int d_d_trans,
            int size, int indent, intarray &cuts, floatarray &cost_t,
            intarray &id_t, intarray &prev_t, intarray &trans_t,
            intarray &charcode_t, objlist<bytearray> &subimages) {
        int n = cuts.dim(0);
        int max_trans=max(trans_t);
        char temp[640];
        fprintf(get_log()->file, "<tr>\n");

        for(int i=0; i<indent; i++) {           // empty cells for no transition
            fprintf(get_log()->file, "<td></td>");
        }
        for(int trans=indent; trans<=max_trans-size; trans+=size) {
            bool find = false;
            for(int i=0; i<n; i++) {                // for all the transitions
                if(cuts(i) == 0) {
                    // look for consecutive transition of the same length and start
                    if(((trans_t(i)-prev_t(i)) == size) &&
                        (prev_t(i)%size == indent) && (trans==prev_t(i))) {
                        sprintf(temp,"<td valign=\"top\" colspan=%d><center>",size);
                        fprintf(get_log()->file, temp);
                        if(id_t(i) == 0) {                              // special case: id=0 means white space or \n
                            fprintf(get_log()->file, "<br>White Space");
                        } else {
                            html(subimages(id_t(i)-1));                 // otherwise just put the picture
                        }
                        fprintf(get_log()->file, "<br><table border>");
                        for(int j=0;(d_d_trans==-1)||(j<d_d_trans);j++) {       // do not display all the possible labels
                            fprintf(get_log()->file, "<tr>");                   // make a table
                            nustring char_text;
                            char_text.resize(1);
                            char_text[0] = nuchar(charcode_t(i+j));             // with the label
                            char *buf = char_text.newUtf8Encode();              // and associated cost
                            sprintf(temp, "<td><font size=\"-1\">%s</font></td><td>"
                                "<font size=\"-1\">%.3f</font></td></tr>", buf, cost_t(i+j));
                            delete[] buf;
                            fprintf(get_log()->file, temp);
                            if ((i+j+1>=n) || (cuts(i+j+1) == 0)) {
                                break;                  // cannot display more than possible number of transitions
                            }
                        }
                        fprintf(get_log()->file, "</table>");
                        //sprintf(temp, "<font size=\"-2\">%d --> %d</font>", prev_t(i), trans_t(i));
                        //fprintf(file, temp);          // having the 'coordinates' of the transition
                        fprintf(get_log()->file, "</center></td>");
                        find = true;
                    }
                }
            }
            if(find == false) {                             // if the transition is not found
                for(int i=0; i<size; i++) {     // display empty cell in compensation
                    fprintf(get_log()->file, "<td></td>");
                }
            }
        }
        fprintf(get_log()->file, "\n</tr>\n");
        fflush(get_log()->file);
    }

    void Logger::transitions_in_graph(int maxcomp, int debug_display_transitions,
            intarray &cuts, floatarray &cost_t,
            intarray &id_t, intarray &prev_t, intarray &trans_t,
            intarray &charcode_t, objlist<bytearray> &subimages) {
        if(!enabled) return;
        putIndent();
        fprintf(get_log()->file, "<table border>\n");
        for(int i=1; i <= maxcomp; i++) {       // diplay all possible combinations
            for(int j=0; j<i; j++) {
                display_one_line_graph(debug_display_transitions, i, j, cuts, cost_t, id_t, prev_t, trans_t, charcode_t, subimages);
            }
        }
        fprintf(get_log()->file, "</table>\n");
        fflush(get_log()->file);
    }

    void Logger::transitions_in_list(int maxcomp, int n_disp_trans, intarray &cuts, floatarray &cost_t,
                intarray &id_t, intarray &prev_t, intarray &trans_t,
                intarray &charcode_t, objlist<bytearray> &subimages) {
        if(!enabled) return;
        putIndent();
        char temp[1024];
        int n = cuts.dim(0);
        for(int i=0; i<n; i++) {                        // for all the transitions
            if (cuts(i) == 0) {                             // if a new transition is found
                operator()("");
                if (id_t(i) == 0) {                     // special case: id=0 means white space or \n
                    operator()("picture : white space");
                } else {
                    operator()("picture ", subimages(id_t(i)-1));
                }
            }       // display everything or the nbest
            if ((n_disp_trans == -1) or (cuts(i) < n_disp_trans)) {
                nustring char_text;
                char_text.resize(1);
                char_text[0] = nuchar(charcode_t(i));
                char *buf = char_text.newUtf8Encode();
                sprintf(temp,"transition `%s' (%d) id %d from %d to %d costs %f",
                        buf, charcode_t(i), id_t(i), prev_t(i), trans_t(i), fabs(cost_t(i)));
                delete[] buf;
                operator()(temp);
            }
        }
        fflush(get_log()->file);
    }

    void Logger::confusion(iupr_bpnet::ConfusionMatrix &cm) {
        if(!enabled) return;
        putIndent();
        intarray conf;
        cm.get(conf);
        fprintf(get_log()->file, "<table border>\n<tr>\n<td></td>");
        for(int j=0;j<conf.dim(1);j++) {
            fprintf(get_log()->file,"<td>%d</td>",j);
        }
        fprintf(get_log()->file, "</tr>\n");
        for(int i=0;i<conf.dim(0);i++) {
            fprintf(get_log()->file,"<tr>\n<td>%d</td>",i);
            for(int j=0;j<conf.dim(1);j++) {
                if(conf(i,j)==0) {
                    fprintf(get_log()->file,"<td><font color=\"Gray\">%d</font></td>",conf(i,j));
                } else {
                    fprintf(get_log()->file,"<td>%d</td>",conf(i,j));
                }
            }
            fprintf(get_log()->file,"</tr>\n");
        }
        fprintf(get_log()->file, "</table>\n");
        fflush(get_log()->file);
    }

    void Logger::confusion(iupr_bpnet::ConfusionMatrix &cm,ClassMap &map) {
        if(!enabled) return;
        putIndent();
        intarray conf;
        cm.get(conf);
        fprintf(get_log()->file, "<table border>\n<tr>\n<td></td>");
        for(int j=0;j<conf.dim(1);j++) {
            nuchar c(map.get_ascii(j));
            nustring s(1);
            s[0] = c;
            fprintf(get_log()->file,"<td>");
            html(s);
            fprintf(get_log()->file,"</td>");
        }
        fprintf(get_log()->file,"</tr>\n");
        for(int i=0;i<conf.dim(0);i++) {
            nuchar c(map.get_ascii(i));
            nustring s(1);
            s[0] = c;
            fprintf(get_log()->file,"<td>");
            html(s);
            fprintf(get_log()->file,"</td>");
            for(int j=0;j<conf.dim(1);j++) {
                if(conf(i,j)==0) {
                    fprintf(get_log()->file,"<td><font color=\"Gray\">%d</font></td>",conf(i,j));
                } else {
                    fprintf(get_log()->file,"<td>%d</td>",conf(i,j));
                }
            }
            fprintf(get_log()->file,"</tr>\n");
        }
        fprintf(get_log()->file, "</table>\n");
        fflush(get_log()->file);
    }

    void Logger::print_one_confusion(   ClassMap &map,int i,int j,int v,
                                        const char* prefix) {
        nuchar c1(map.get_ascii(i));
        nustring s1(1);s1[0] = c1;
        nuchar c2(map.get_ascii(j));
        nustring s2(1);s2[0] = c2;
        fprintf(get_log()->file,"<tr align=\"center\"><td>%s</td>"
                                "<td><font color=\"Gray\">%d</font></td>"
                                "<td><font color=\"Gray\">%d</font></td>"
                                "<td><font color=\"Gray\">%d</font></td>"
                                "<td><font color=\"Gray\">%d</font></td>",
                                prefix,i,j, map.get_ascii(i),map.get_ascii(j));
        if (map.get_ascii(i)==172)
            fprintf(get_log()->file,"<td>*G*</td>");
        else {
            fprintf(get_log()->file,"<td>");html(s1);fprintf(get_log()->file,"</td>");
        }
        if (map.get_ascii(j)==172)
            fprintf(get_log()->file,"<td>*G*</td>");
        else {
            fprintf(get_log()->file,"<td>");html(s2);fprintf(get_log()->file,"</td>");
        }
        fprintf(get_log()->file,"<td>%d</td></tr>\n",v);
    }

    void Logger::reduced_confusion(iupr_bpnet::ConfusionMatrix &cm,ClassMap &map) {
        if(!enabled) return;
        putIndent();
        intarray conf;
        cm.confusion_to_list(conf);
        fprintf(get_log()->file,"<table border>");
        fprintf(get_log()->file,"<tr align=\"center\"><td></td><td colspan=\"2\">internal</td>"
                                "<td colspan=\"2\">ascii</td><td colspan=\"2\">utf8</td></tr>\n");
        fprintf(get_log()->file,"<tr align=\"center\"><td></td><td>-t-</td><td>-o-</td>"
                                "<td>-t-</td><td>-o-</td><td>-t-</td><td>-o-</td></tr>\n");
        fprintf(get_log()->file,"<tr></tr>\n");
        for(int i=0;i<conf.dim(0);i++) {
            if (conf(i,2)>0 && (conf(i,0)==conf(i,1))) {
                print_one_confusion(map,conf(i,0),conf(i,1),conf(i,2),"<font color=\"Green\">good</font>");
            }
        }
        fprintf(get_log()->file, "<tr></tr>\n");
        for(int i=0;i<conf.dim(0);i++) {
            if ((conf(i,2)>0) && (conf(i,0)!=conf(i,1))) {
                print_one_confusion(map,conf(i,0),conf(i,1),conf(i,2),"<font color=\"Red\">conf</font>");
            }
        }
        fprintf(get_log()->file, "</table>\n");
        fflush(get_log()->file);
    }

    void write_data_in_R(FILE* f,char* v, floatarray &d, int e) {
        fprintf(f,"%s",v);
        for(int i=0;i<e;i++)
            fprintf(f,"%f,",d[i]);
        fprintf(f,"%f)\n",d[e]);
    }

    void Logger::train_test_curves(
                        floatarray &h_train_error,floatarray &h_train_cls_error,
                        floatarray &h_test_error,floatarray &h_test_cls_error,
                        floatarray &h_lr,int epochs) {
        if(!enabled) return;
        if (epochs == 0) return;
        if (system("R --version > /dev/null") != 0) return;
        if (system("convert > /dev/null") != 0) return;

        float graph_dim_x = 14;
        float graph_dim_y = 8;
        const char* log_dir = (char*)get_log()->dir;
        graph_dim_x *= graph_scale;
        graph_dim_y *= graph_scale;

        strbuf buf_R;
        strbuf buf_PDF;
        strbuf_format(buf_R,"%s/curves-%d-%d.R",log_dir,
                            epochs,image_counter);

        stdio R = stdio(buf_R,"wt");
        write_data_in_R(R,"tr_err<-c(",h_train_error,epochs);
        write_data_in_R(R,"tr_cls_err<-c(",h_train_cls_error,epochs);
        write_data_in_R(R,"te_err<-c(",h_test_error,epochs);
        write_data_in_R(R,"te_cls_err<-c(",h_test_cls_error,epochs);
        write_data_in_R(R,"lr<-c(",h_lr,epochs);

        strbuf_format(buf_PDF,"%s/ocropus-log-%d-",log_dir,image_counter);
        fprintf(R,  "pdf('%str_te_err.png',width=%g,height=%g);\n",
                    (char*)buf_PDF,graph_dim_x,graph_dim_y);
        putIndent();
        if (merge_lr) {
            fprintf(get_log()->file,"Train and test error, learning rate<BR>");
            fprintf(R,"plot.new();\n");
            fprintf(R,  "plot.window(xlim=c(%d,%d),ylim=c(min(tr_err[2:%d],"
                        "te_err[2:%d]),max(tr_err[2:%d],te_err[2:%d])));\n",
                        (epochs==0)?0:1,(epochs==0)?1:epochs,epochs+1,
                        epochs+1,epochs+1,epochs+1);
            if (epochs==1) {
                fprintf(R,"points(c(1),tr_err[2],col='red',lwd=2,lty=1)\n");
                fprintf(R,"points(c(1),te_err[2],col='blue',lwd=2,lty=3);\n");
            } else {
                fprintf(R,  "lines(c(1:%d),tr_err[2:%d],col='red',"
                            "lwd=2,lty=1)\n",epochs,epochs+1);
                fprintf(R,  "lines(c(1:%d),te_err[2:%d],col='blue',"
                            "lwd=2,lty=3);\n",epochs,epochs+1);
            }
            fprintf(R,  "axis(1);\n");
            fprintf(R,  "axis(2);\n");
            fprintf(R,  "legend('topright',c('Train','Test','LR'),"
                        "lwd=2,lty=c(1,3,1),col=c('red','blue','darkgreen'));\n");
            fprintf(R,  "title(xlab='Epochs',ylab='Mean error',"
                        "main='Train error, test error, learning rate');\n");
            fprintf(R,  "plot.window(xlim=c(1,%d),c(min(lr[2:%d]),"
                        "max(lr[2:%d])));\n",epochs,epochs+1,epochs+1);
            if (epochs==1)
                fprintf(R,"points(c(1),lr[2],col='darkgreen',lwd=2,lty=1);\n");
            else
                fprintf(R,"lines(c(1:%d),lr[2:%d],col='darkgreen',lwd=2,lty=1);\n",
                            epochs,epochs+1);
            fprintf(R,"axis(4,col='darkgreen',col.axis='darkgreen');\n");
            fprintf(R,"box();\n");
        } else {
            fprintf(get_log()->file,"Train, test errors<BR>");
            if (epochs==1)
                fprintf(R,  "\tplot(c(1),c(tr_err[2]),col='red',lwd=2,lty=1");
            else
                fprintf(R,  "\tplot(c(1:%d),tr_err[2:%d],type='l',col='red',"
                            "lwd=2,lty=1",epochs,epochs+1);
            fprintf(R,  ",\n\t\tylim=c(min(tr_err[2:%d],te_err[2:%d]),"
                        "max(tr_err[2:%d],te_err[2:%d]))",
                        epochs+1,epochs+1,epochs+1,epochs+1);
            fprintf(R,  ",\n\t\txlab='Epochs',ylab='Mean error'"
                        ",\n\t\tmain='Train and test error');\n");
            if (epochs==1)
                fprintf(R,  "\tpoints(c(1),c(te_err[2]),col='blue',"
                            "lwd=2,lty=3);\n",epochs,epochs+1);
            else
                fprintf(R,  "\tlines(c(1:%d),te_err[2:%d],type='l',col='blue',"
                            "lwd=2,lty=3);\n",epochs,epochs+1);
            fprintf(R,  "\tlegend('topright',c('Train','Test'),lwd=2,lty=c(1,3),"
                        "col=c('red','blue'));\n");
        }
        fprintf(R,"dev.off();\n\n");
        fprintf(get_log()->file,"<IMG SRC=\"ocropus-log-%d-tr_te_err.png\">\n"
                                "<BR>\n",image_counter);

        if (!merge_lr) {
            strbuf_format(buf_PDF,"%s/ocropus-log-%d-",
                        log_dir,image_counter+2);
            fprintf(R,  "pdf(\"%slr.png\",width=%g,height=%g);\n",
                        (char*)buf_PDF,graph_dim_x,graph_dim_y);
            if (epochs==1)
                fprintf(R,  "\tplot(c(1),c(lr[2]),col='darkgreen',lwd=2,lty=1",
                            epochs,epochs+1);
            else
                fprintf(R,  "\tplot(c(1:%d),lr[2:%d],type='l',col='darkgreen',"
                            "lwd=2,lty=1",epochs,epochs+1);
            fprintf(R,  ",\n\tylim=c(min(lr[2:%d]),max(lr[2:%d])),\n\t",
                        epochs+1,epochs+1);
            fprintf(R,"xlab='Epochs',ylab='Value',main='Learning rate');\n");
            fprintf(R,"dev.off();\n\n");
            putIndent();
            fprintf(get_log()->file,"Learning rate<BR>");
            fprintf(get_log()->file,"<IMG SRC=\"ocropus-log-%d-lr.png\">\n<BR>\n",
                                    image_counter+2);
        }

        strbuf_format(buf_PDF,  "%s/ocropus-log-%d-",
                                log_dir,image_counter+1);
        fprintf(R,  "pdf('%str_te_cls_err.png',width=%g,height=%g);\n",
                    (char*)buf_PDF,graph_dim_x,graph_dim_y);
        if (epochs==1)
            fprintf(R,  "\tplot(c(1),c(tr_cls_err[2]),col='red',lwd=2,lty=1");
        else
            fprintf(R,  "\tplot(c(1:%d),tr_cls_err[2:%d],"
                        "type='l',col='red',lwd=2,lty=1",epochs,epochs+1);
        fprintf(R,  ",\n\t\tylim=c(min(tr_cls_err[2:%d],te_cls_err[2:%d]),"
                    "max(tr_cls_err[2:%d],te_cls_err[2:%d]))",
                    epochs+1,epochs+1,epochs+1,epochs+1);
        fprintf(R,  ",\n\t\txlab=\"Epochs\",ylab='Classification %% error',"
                    "\n\t\tmain='Train and test classification error');\n");
        if (epochs==1)
            fprintf(R,  "\tpoints(c(1),c(te_cls_err[2]),col='blue',"
                        "lwd=2,lty=3);\n",epochs,epochs+1);
        else
            fprintf(R,  "\tlines(c(1:%d),te_cls_err[2:%d],col='blue',"
                        "lwd=2,lty=3);\n",epochs,epochs+1);
        fprintf(R,  "\tlegend('topright',c('Train','Test'),lwd=2,"
                    "lty=c(1,3),col=c('red','blue'));\n");
        fprintf(R,"dev.off();\n\n");
        putIndent();
        fprintf(get_log()->file,"Train, test classification error<BR>");
        fprintf(get_log()->file,"<IMG SRC=\"ocropus-log-%d-tr_te_cls_err.png\">"
                                "\n<BR>\n",image_counter+1);
        R.close();
        strbuf cmd;

        strbuf_format(cmd, "(nice -n19 R --quiet --slave CMD BATCH %s ; ",
                            (char*)buf_R);
        strbuf_format(  buf_PDF,"%s/ocropus-log-%d-tr_te_err.png",
                        (char*)get_log()->dir,image_counter);
        strbuf_format(  buf_R,"nice -n19 convert -antialias %s png:%s ; ",
                        (char*)buf_PDF,(char*)buf_PDF);
        cmd += buf_R;

        if (!merge_lr) {
            strbuf_format(buf_PDF,  "%s/ocropus-log-%d-lr.png",
                                    log_dir,image_counter+2);
            strbuf_format(buf_R,    "nice -n19 convert -antialias %s png:%s ; ",
                                    (char*)buf_PDF,(char*)buf_PDF);
            cmd += buf_R;
        }

        strbuf_format(buf_PDF,  "%s/ocropus-log-%d-tr_te_cls_err.png",
                                log_dir,image_counter+1);
        strbuf_format(buf_R,    "nice -n19 convert -antialias %s png:%s) &",
                                (char*)buf_PDF,(char*)buf_PDF);
        cmd += buf_R;
        //printf("%s\n",(char*)cmd);
        system(cmd);

        image_counter += 2+(merge_lr?0:1);
        fflush(get_log()->file);
    }
}
