/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "RenameEvent.h"
#include "Patch.h"
#include "Node.h"
#include "NodeTree.h"
#include "Plugin.h"
#include "InputNode.h"
#include "InputPort.h"
#include "OutputNode.h"
#include "OutputPort.h"
#include "Om.h"
#include "Array.h"
#include "DisconnectNodeEvent.h"
#include "DisconnectPortEvent.h"
#include "OmApp.h"
#include "OSCSender.h"
#include "Maid.h"
#include "JackDriver.h"
#include "SlowEvent.h"
#include "OmPath.h"
#include "ObjectStore.h"
#include "util.h"

namespace Om {


RenameEvent::RenameEvent(Request* request, const string& path, const string& name)
: SlowEvent(request),
  m_old_path(path),
  m_name(name),
  m_parent_patch(NULL),
  m_treenode(NULL),
  m_error(NO_ERROR)
{
	m_new_path = OmPath::parent(m_old_path) +"/"+ m_name;
}


RenameEvent::~RenameEvent()
{
}


void
RenameEvent::prepare()
{
	if (m_name.find("/") != string::npos) {
		m_error = INVALID_NAME;
		SlowEvent::prepare();
		return;
	}

	if (om->object_store()->find_object(m_new_path)) {
		m_error = OBJECT_EXISTS;
		SlowEvent::prepare();
		return;
	}
	
	OmObject* obj = om->object_store()->find_patch(m_old_path);
	if (obj == NULL)
		obj = om->object_store()->find_node(m_old_path);

	if (obj == NULL) {
		m_error = OBJECT_NOT_FOUND;
		SlowEvent::prepare();
		return;
	}
	
	// Renaming only works for Nodes and Patches (which are Nodes)
	if (obj->as_node() == NULL) {
		m_error = OBJECT_NOT_RENAMABLE;
		SlowEvent::prepare();
		return;
	}
	
	if (obj != NULL) {
		m_parent_patch = obj->as_node()->parent_patch();
		if (m_parent_patch != NULL) {
			m_treenode = m_parent_patch->nodes().find_treenode(obj->name());
			assert(m_treenode != NULL);
			assert(m_treenode->node() == obj);
			m_treenode = m_parent_patch->remove_node(m_treenode);
		}
	}
	
	obj->set_name(m_name);
	assert(obj->path() == m_new_path);
	
	if (m_treenode != NULL) {
		assert(m_parent_patch != NULL);
		m_parent_patch->add_node(m_treenode);
	}

	SlowEvent::prepare();
}


void
RenameEvent::execute(uint sample_offset)
{
	//cout << "Executing rename event...";
	SlowEvent::execute(sample_offset);
}


void
RenameEvent::post_process()
{
	string msg = "Unable to rename object - ";
	
	if (m_error == NO_ERROR) {
		if (m_request != NULL)
			m_request->respond_ok();
		om->osc_sender()->send_rename(m_old_path, m_new_path);
	} else {
		if (m_error == OBJECT_EXISTS)
			msg.append("Object already exists at ").append(m_new_path);
		else if (m_error == OBJECT_NOT_FOUND)
			msg.append("Could not find object ").append(m_old_path);
		else if (m_error == OBJECT_NOT_RENAMABLE)
			msg.append(m_old_path).append(" is not renamable");
		else if (m_error == INVALID_NAME)
			msg.append(m_name).append(" is not a valid name");

		if (m_request != NULL)
			m_request->respond_error(msg);
	}
}


} // namespace Om
