(*
 * Shell expressions.
 *
 * ----------------------------------------------------------------
 *
 * @begin[license]
 * Copyright (C) 2004 Mojave Group, Caltech
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Jason Hickey
 * @email{jyh@cs.caltech.edu}
 * @end[license]
 *)
open Lm_location
open Lm_symbol
open Lm_debug

open Omake_env

(*
 * A shell command.
 *)
type cmd =
   { cmd_loc     : loc;
     cmd_env     : (symbol * string) list;
     cmd_exe     : string;
     cmd_argv    : string list;
     cmd_stdin   : string option;
     cmd_stdout  : string option;
     cmd_stderr  : bool;
     cmd_append  : bool;
     cmd_escaped : bool
   }

(*
 * An internal command.
 *)
type apply =
   { apply_loc      : loc;
     apply_name     : symbol;
     apply_fun      : venv -> Unix.file_descr -> Unix.file_descr -> Unix.file_descr -> string list -> int * value;
     apply_args     : string list;
     apply_stdin    : string option;
     apply_stdout   : string option;
     apply_stderr   : bool;
     apply_append   : bool
   }

(*
 * A pipe may have several cmds in sequence.
 *)
type pipe_op =
   PipeAnd
 | PipeOr
 | PipeSequence

(*
 * A pipe with redirection.
 *)
type group =
   { group_stdin    : string option;
     group_stdout   : string option;
     group_stderr   : bool;
     group_append   : bool;
     group_pipe     : pipe
   }

and pipe =
   PipeApply      of loc * apply
 | PipeCommand    of loc * cmd
 | PipeCond       of loc * pipe_op * pipe * pipe
 | PipeCompose    of loc * bool * pipe * pipe
 | PipeGroup      of loc * group
 | PipeBackground of loc * pipe

(*
 * Signals.
 *)
type signal =
   SigAbrt
 | SigAlrm
 | SigFPE
 | SigHup
 | SigIll
 | SigInt
 | SigKill
 | SigPipe
 | SigQuit
 | SigSegv
 | SigTerm
 | SigUsr1
 | SigUsr2
 | SigChld
 | SigCont
 | SigStop
 | SigTstp
 | SigTtin
 | SigTtou
 | SigVTAlrm
 | SigProf
 | SigNum of int

(*
 * Debug flag.
 *)
let debug_shell =
   create_debug (**)
      { debug_name = "shell";
        debug_description = "print debugging information for the shell";
        debug_value = false
      }

(*!
 * @docoff
 *
 * -*-
 * Local Variables:
 * Caml-master: "compile"
 * End:
 * -*-
 *)
