;;!emacs
;;
;; FILE:         br-python-ft.el
;; SUMMARY:      Python OO-Browser class and member functions.
;; USAGE:        GNU Emacs Lisp Library
;; KEYWORDS:     python, oop, tools
;;
;; AUTHOR:       Harri Pasanen / Bob Weiner
;;               based on Smalltalk and C++ OO-Browsers 
;; ORG:          BeOpen.com
;;
;; ORIG-DATE:    5-Apr-96
;; LAST-MOD:     12-Jun-99 at 01:10:06 by Bob Weiner
;;
;; Copyright (C) 1996, 1997, 1998  BeOpen.com
;; See the file BR-COPY for license information.
;;
;; This file is part of the OO-Browser.
;;
;; DESCRIPTION:  
;;    There may still be traces of C++ origin in this file.
;; DESCRIP-END.

;;; ************************************************************************
;;; Other required Elisp libraries
;;; ************************************************************************

(require 'br-python)

;;; ************************************************************************
;;; Public variables
;;; ************************************************************************

(defvar python-import-dirs '("/usr/local/lib/python/")
  "Ordered list of module directories by default searched by python 
interpreter. Each directory must end with a directory separator.")

(defconst python-type-tag-separator "@"
  "String that separates a tag's type from its normalized definition form.
This should be a single character which is unchanged when quoted for use as a
literal in a regular expression.")

(defconst python-tag-fields-regexp
  ;; The \\\\? below is necessary because we sometimes use this expression to
  ;; test against a string that has ben regexp-quoted and some of the
  ;; characters in br-feature-type-regexp will then be preceded by \\.
  (format "^\\([^%s \n]+\\)%s\\\\?\\(%s \\)\\([^%s\n]+\\)%s"
	  python-type-tag-separator python-type-tag-separator br-feature-type-regexp
	  python-type-tag-separator python-type-tag-separator)
 "Regexp matching the fields of a Python feature tag line.
Group 1 is the class of the feature.  Group 2 is the prefix preceding the
feature when displayed within a listing buffer.  Group 3 is the feature name.
The feature definition signature begins at the end of the regexp match,
i.e. (match-end 0), and goes to the end of the string or line.")

;;; ************************************************************************
;;; Public functions
;;; ************************************************************************

(defun python-add-default-classes ()
  (if br-c-tags-flag (c-add-default-classes)
    ;; Add to categorize global functions
    (br-add-default-classes '("[function]"))))

(defun python-feature-implementors (name)
  "Return unsorted list of Python feature tags which implement feature NAME."
  (python-feature-matches (concat "^" (regexp-quote name) "$")))

(defun python-feature-signature-to-name (feature-sig-or-tag &optional with-class for-display)
  "Extracts the feature name from FEATURE-SIG-OR-TAG.
The feature's class name is dropped from feature-sig-or-tag unless optional
WITH-CLASS is non-nil.  If optional FOR-DISPLAY is non-nil, a feature type
character is prepended to the name for display in a browser listing."
  (if (br-feature-tag-p feature-sig-or-tag)
      (br-feature-tag-name feature-sig-or-tag with-class for-display)
    (let ((name))
      (cond
       ;; member
       ((string-match python-tag-fields-regexp feature-sig-or-tag)
	(setq name (substring feature-sig-or-tag
			      (match-beginning (if for-display 2 3))
			      (match-end 3)))
	(if with-class
	    (setq name (concat
			(substring feature-sig-or-tag
				   (match-beginning 1) (match-end 1))
			"::" name)))
	;; Remove any trailing whitespace.
	(br-delete-space name))
       ;;
       ;; unknown
       (t;; Remove any trailing whitespace and add display prefix.
	(setq name (br-delete-space feature-sig-or-tag))
	(if for-display (concat "- " name) name))))))

(defun python-scan-features ()
  "Return reverse ordered list of current buffer's Python function definitions.
Assume point is at the beginning of a widened buffer."
  (save-excursion
    (let ((routines) class name rout)
      (while (re-search-forward python-routine-def nil t)
	(setq class "[function]"
	      name (br-buffer-substring
		    (match-beginning python-feature-name-grpn)
		    (match-end python-feature-name-grpn))
	      rout (python-feature-normalize 
		    (br-buffer-substring (match-beginning 0) (match-end 0))
		    class name)
	      routines (cons rout routines)))
      routines)))

(defun python-to-definition (&optional other-win)
  "If point is on an import statement, look for the module file.
With OTHER-WIN non-nil, show it in another window."
  (interactive)
  (cond
   ((python-import-file other-win))
   (t	(beep)
	(message
	 "(OO-Browser):  Select an import statement to display its source.")
	nil)))

(defun python-store-class-info (entry)
  "Lookup a Python doc-string for a class or a feature."
 (setq python-docstring (python-lookup-docstring entry)))


(defun python-insert-class-info ()
  "Use the info facility to display Python doc-strings"
  (interactive)
  (insert python-docstring))

;;; ************************************************************************
;;; Private functions
;;; ************************************************************************

(defun python-lookup-docstring (entry)
  "Looks up a docstring for any browser listing ENTRY."
  (let ((filename)
	(feature-tag)
	(docstring))
      (cond ((br-find-feature-entry)
	     (setq feature-tag (br-feature-get-tag)
		   filename (br-feature-tag-path feature-tag)))
	    ((and (setq entry (br-find-class-name))
		  (br-class-in-table-p entry))
	     (setq filename (br-class-path entry)))
	    (t (error "(OO-Browser):  Entry referenced but not defined in the Environment.")))
      (if filename
	  (setq docstring
		(python-get-docstring-from-source entry feature-tag filename)))
      (if docstring
	  docstring
	(concat entry " does not have a documentation string."))))

(defun python-get-docstring-from-source (entry feature-tag filename)
  "Scan source for ENTRY's docstring using FEATURE-TAG and FILENAME.
If FEATURE-TAG is non-nil, locate as a feature, otherwise ENTRY is a classname."
  (let ((no-kill (get-file-buffer filename))
	(docstring))
    (if no-kill
	(set-buffer no-kill)
      (br-insert-file-contents filename))
    (save-restriction
      (save-excursion
	(widen)
	(goto-char (point-min))
	(if feature-tag
	    (if (python-feature-locate-p feature-tag)
		(setq docstring (python-extract-docstring)))
	  (if (re-search-forward (python-class-definition-regexp entry) nil t)
	      (setq docstring (python-extract-docstring))))))
    (if (not no-kill)
	(kill-buffer *br-tmp-buffer*))
    docstring))

(defun python-extract-docstring ()
  "Return the documentation string for the class or method at point, or nil if none."
  (search-forward ":" nil t)
  (if (looking-at
       (concat python-empty-line "+[ \t]*" python-string-start))
      (let ((start (match-end 0))
	    (end-quote (buffer-substring
			(match-beginning 4) (match-end 4))))
	(goto-char start)
	(search-forward end-quote nil t)
	(br-buffer-substring start (match-beginning 0)))))

(defconst python-string-start
  (concat 
   "\\("
   "'''"                ; triple single-quoted
   "\\|"		; or
   "\"\"\""	        ; triple double-quoted
   "\\|"		; or
   "'"		        ; single-quoted, not empty
   "\\|"		; or
   "\""	                ; double-quoted, not empty
   "\\)")
  "regexp matching python string literal starting quotes")

(defconst python-empty-line
  "\\(\\([ \t]*\n\\)\\|\\([ \t]*#.*$\\)\\)"
  "regexp matching an empty or python comment line")

(defun python-within-comment-p ()
  "Return non-nil if point is within a Python comment."
  (save-excursion
    (and (re-search-backward "#\\|\n" nil t)
	 (not (looking-at "\n")))))

(defun python-within-string-p ()
  "Return non-nil if point is within a multi-line python string."
  (save-excursion
    (let ((count 0))
      (while (re-search-forward python-multi-line-string-delimiter nil t)
	(setq count (1+ count)))
      (= (% count 2) 1))))

(defun python-feature-map-tags (function regexp)
  "Apply FUNCTION to all current feature tags that match REGEXP and return a list of the results."
  (let ((identifier-chars (concat "[" python-identifier-chars "]*"))
	(case-fold-search))
    ;; Ensure handle "^" and "$" meta-chars.
    (setq regexp
	(concat (format "\\`%s " br-feature-type-regexp)
		(if (equal (substring regexp 0 1) "^")
		    (progn (setq regexp (substring regexp 1)) nil)
		  identifier-chars)
		(if (equal (substring regexp -1) "$")
		    (substring regexp 0 -1)
		  (concat regexp identifier-chars))
		  "\\'"))
    (br-feature-map-tags function regexp)))

(defun python-feature-matches (regexp)
  "Return an unsorted list of feature tags whose names match in part or whole to REGEXP.
^ and $ characters may be used to match to the beginning and end of a feature name,
respectively."
  (python-feature-map-tags 'identity regexp))

(defun python-feature-normalize (routine class name)
  "Return a feature tag based on ROUTINE, CLASS and NAME."
  (setq class (br-delete-space class)
	name (concat "- " name))
  (let* ((len (length routine))
	 (normal-feature (make-string len ?\ ))
	 (n 0) (i 0)
	 (space-regexp "[ \t\n\r]+")
	 (original-syntax-table (syntax-table))
	 chr)
    (unwind-protect
	(progn
	  (set-syntax-table text-mode-syntax-table)
	  (while (< i len)
	    (setq chr (aref routine i)) 
	    (cond
	     ;; Convert sequences of space characters to a single space.
	     ;; GNU Emacs doesn't support optional syntax-table arg to
	     ;; `char-syntax'.
	     ((eq (char-syntax chr) ?\ )
	      (if (string-match space-regexp routine i)
		  (progn (setq i (match-end 0))
			 (if (not (and (> n 0)
				       (eq (aref normal-feature (1- n)) ?\ )))
			     (setq n (1+ n))))
		(setq i (1+ i)
		      n (1+ n))))
	     ;;
	     ;; Remove # comments
	     ((eq chr ?#)
	      (setq i (1+ i))
	      (while (and (< i len) (not (eq (aref routine i) ?\n)))
		(setq i (1+ i))))
	     (t ;; Normal character
	      (aset normal-feature n chr)
	      (setq i (1+ i)
		    n (1+ n)))))
	  (concat class python-type-tag-separator 
		  name python-type-tag-separator 
		  (br-delete-space (substring normal-feature 0 n))))
      (set-syntax-table original-syntax-table))))

(defun python-files-with-source (class)
  "Use CLASS to compute set of files that match to a Python source file regexp.
Return as a list."
  (let ((file (if class (br-class-path class) buffer-file-name)))
    (and file
	 (let* ((src-file-regexp (concat "^" (br-filename-head file)
					 python-code-file-regexp))
		(dir (file-name-directory file))
		(files (directory-files dir nil src-file-regexp)))
	   (mapcar (function (lambda (f) (concat dir f)))
		   files)))))

(defun python-find-ancestors-feature (class-list ftr-pat &optional other-win)
  "Scan ancestors of CLASS-LIST and show routine definition matching FTR-PAT."
  ;; If no class, search for non-member function.
  (or class-list (setq class-list '(nil)))
  (br-feature-display class-list ftr-pat other-win))

(defun python-find-class-name ()
  "Return current word as a potential class name."
  (save-excursion
    (let* ((start)
	   (ignore "\]\[ \t\n\r\f\;,.\(\){}*&-")
	   (pat (concat "^" ignore)))
      (forward-char 1)
      (skip-chars-backward ignore)
      (skip-chars-backward pat)
      (setq start (point))
      (skip-chars-forward (concat pat ":"))
      (buffer-substring start (point)))))

(defun python-output-feature-tags (routine-file routine-tags-list)
  "Write Python ROUTINE-FILE's ROUTINE-TAGS-LIST into `br-feature-tags-file'.
Assume `br-feature-tags-init' has been called."
  (interactive)
  (save-excursion
    (set-buffer (funcall br-find-file-noselect-function br-feature-tags-file))
    (goto-char 1)
    ;; Delete any prior routine tags associated with routine-file
    (if (search-forward routine-file nil 'end)
	(progn (forward-line -1)
	       (let ((start (point)))
		 (search-forward "\^L" nil 'end 2)
		 (backward-char 1)
		 (delete-region start (point)))))
    (if routine-tags-list
	(progn (insert "\^L\n")
	       ;; Quote pathname to avoid read errors on MS OSes.
	       (prin1 routine-file (current-buffer))
	       (insert "\n")
	       (mapcar (function (lambda (tag) (insert tag "\n")))
		       routine-tags-list)))))

(defun python-find-module-name ()
  "Return current word as a potential module name."
  (save-excursion
    (let ((start))
      (forward-char 1)
      (skip-chars-backward python-identifier-chars)
      (setq start (point))
      (skip-chars-forward python-identifier-chars)
      (buffer-substring start (point)))))

(defun python-import-file (&optional other-win)
  "If point is on an import module line, display the module, method or function name at point.
With optional OTHER-WIN non-nil, display it in the other window.

Return non-nil iff point is on an import file line, even if a matching entry
is not found.  When found return the full pathname to the import entry,
otherwise return t.

Look for import files within `python-import-dirs' and any Environment
directory."
  (let ((opoint (point)))
    (beginning-of-line)
    (cond ((looking-at python-import-modules-regexp)
	   (if (< (match-end 0)
		  (max opoint (match-beginning python-import-name-grpn)))
	       ;; Have to avoid selecting anything within a # comment here.
	       (goto-char (match-beginning python-import-name-grpn))
	     (goto-char
	      (max opoint (match-beginning python-import-name-grpn))))
	   (let* ((import-name (python-find-module-name))
		  (path (python-import-pathname import-name)))
	     ;; If found, display file
	     (python-display-module import-name path other-win)
	     (or path t)))
	  ((looking-at python-import-functions-regexp)
	   (if (< (match-end 0)
		  (max opoint (match-beginning python-import-name-grpn)))
	       ;; Have to avoid selecting anything within a # comment here.
	       (goto-char (match-beginning python-import-name-grpn))
	     (goto-char
	      (max opoint (match-beginning python-import-name-grpn))))
	   (setq opoint (point))
	   (let* ((end-module-name (match-end python-import-name-grpn))
		  (module-name 
		   (br-buffer-substring
		    (match-beginning python-import-name-grpn)
		    (match-end python-import-name-grpn)))
		  (import-name (python-find-module-name))
		  (path (python-import-pathname module-name)))
	     ;; If found, display file
	     (if (python-display-module module-name path other-win)
		 (if (or (<= opoint end-module-name)
			 (equal import-name "import")
			 (equal import-name ""))
		     nil
		   (if (re-search-forward
			(concat "^[ \t]*\\(class\\|def\\|\\)"
				"\\(" (regexp-quote import-name) "\\)"
				"[^" python-identifier-chars "]")
			nil t)
		       (goto-char (match-beginning 2))
		     (beep)
		     (message "(OO-Browser):  Found module `%s' but not member `%s'."
			      module-name import-name))))
	     (or path t)))
	  (t (goto-char opoint)
	     nil))))

(defun python-display-module (module-name path other-win)
  "Display file associated with MODULE-NAME and PATH in OTHER-WIN (if non-nil).
Return t if file is displayed, nil otherwise."
  (if path
      (if (file-readable-p path)
	  (progn
	    (funcall br-edit-file-function path other-win)
	    (widen)
	    (goto-char (point-min))
	    (if (not (fboundp 'br-lang-mode))
		(python-mode)
	      (br-major-mode))
	    t)
	(beep)
	(message "(OO-Browser):  Module `%s' is unreadable." path)
	nil)
    (beep)
    (message "(OO-Browser):  Cannot find module `%s'." module-name)
    nil))

(defun python-import-pathname (import-name)
  "Return the full pathname to a Python IMPORT-NAME or nil if none.
Look for import files within `python-import-dirs' and any Environment
directory."
  (if (not (stringp import-name))
      (error "(python-import-pathname): Invalid import name, `%s'" import-name))
  (if (string-match "\\.py\\'" import-name)
      (setq import-name (substring import-name 0 (match-beginning 0))))
  ;; Convert import-name a.b.c to pathname form, a/b/c.
  (setq import-name (hypb:replace-match-string
		     "\\." import-name
		     (file-name-as-directory "/")
		     t))
  (setq import-name (concat import-name ".py"))
  (let ((dir-list (append python-lib-search-dirs python-sys-search-dirs 
			  python-import-dirs))
	(found)
	path)
    (if buffer-file-name
	(setq dir-list (cons (file-name-directory buffer-file-name)
			     dir-list)))
    (while (and (not found) dir-list)
      (setq path (expand-file-name import-name (car dir-list)))
      (or (setq found (file-exists-p path))
	  (setq dir-list (cdr dir-list))))
    ;;
    ;; If not found in normal include dirs, check all Env paths also.
    ;;
    (if (not found)
	(let ((paths (delq nil (hash-map 'cdr br-paths-htable))))
	  (while (and (not found) paths)
	    (setq path (car paths))
	    (if (string-equal (file-name-nondirectory path) import-name)
		(setq found t paths nil)
	      (setq paths (cdr paths))))))
    (if found path)))

(defun python-locate-feature (ftr class ftr-pat &optional other-win)
  ;; `class' may = nil, implying non-member function
  (or class (setq class "[function]"))
  (let ((def-class))
    (if (and ftr-pat
	     (setq def-class
		   (python-find-ancestors-feature (list class)
					       ftr-pat other-win)))
	(progn (if (and class (not (equal class def-class)))
		   (message
		     "Member `%s' of class `%s' inherited from class `%s'."
		     ftr class def-class))
	       t))))

(defun python-scan-ancestors-feature (class-list ftr-pat &optional other-win)
  "Display routine definition derived from CLASS-LIST, matching FTR-PAT.
Scan files with same base name as class file."
  (let  ((classes class-list)
	 (found-ftr)
	 (code-def-files)
	 (file)
	 (ftr-sig-regexp)
	 (class))
    (if (null class-list)
	nil
      (while (and (not found-ftr) classes)
	(setq class (car classes)
	      code-def-files (python-files-with-source class)
	      ftr-sig-regexp (funcall ftr-pat class))
	(while (and (setq file (car code-def-files))
		    (not (setq found-ftr
			       (br-feature-found-p file ftr-sig-regexp
						   nil other-win t))))
	  (setq code-def-files (cdr code-def-files)))
	(setq classes (if found-ftr nil (cdr classes))))
      (if found-ftr
	  (or class t)
	(python-scan-ancestors-feature
	 (apply 'append (mapcar (function (lambda (cl) (br-get-parents cl)))
				class-list))
	 ftr-pat)))))

(defun python-scan-features-in-class (class start end)
  "Return reverse ordered list of Python routine definitions within CLASS def.
START and END give buffer region to search."
  (setq class (br-delete-space class))
  (save-excursion
    (save-restriction
      (narrow-to-region start end)
      (goto-char start)
      (let ((routines) rout name)
	;;
	;; Get member definitions
	;;
	(while (re-search-forward python-routine-def-in-class nil t)
	  (setq start (match-beginning 0)
		name (br-buffer-substring
		      (match-beginning python-feature-name-grpn)
		      (match-end python-feature-name-grpn))
		rout (python-feature-normalize 
		      (br-buffer-substring (match-beginning 0) (match-end 0))
		      class name)
		routines (cons rout routines)))
	routines))))

(defun python-feature-locate-p (feature-tag &optional regexp-flag)
  "Leaves point at the start of FEATURE-TAG's definition in the current buffer.
Assumes caller has moved point to the beginning of the buffer or to the point
of desired search start.
Optional REGEXP-FLAG means FEATURE-TAG is a regular expression."
  (let ((case-fold-search) (start)
	(found t) feature-sig feature-regexp class)
    (if (br-feature-tag-p feature-tag)
	(setq feature-sig (br-feature-tag-signature feature-tag)
	      class (br-feature-tag-class feature-tag))
      (setq feature-sig feature-tag
	    class nil))
    (if regexp-flag
	(if (stringp feature-tag)
	    (setq feature-regexp feature-tag)
	  (error "(python-feature-locate-p): Not a regexp, %s" feature-tag)))
    ;;
    ;; First move to the proper class implementation if feature-tag does not
    ;; include a <class>:: part and this is not a [default-class], so that if
    ;; two classes in the same file have the same feature signature, we still
    ;; end up at the right one.
    (cond (class
	   (if (or (string-match "\\`\\[" class)
		   (and feature-sig (string-match "::" feature-sig)))
	       nil
	     (setq found (re-search-forward
			  (python-class-definition-regexp class nil)
			  nil t)
		   start (match-beginning 0))))
	  ((string-match python-tag-fields-regexp feature-sig)
	   (setq class (substring feature-sig
				  (match-beginning 1) (match-end 1))
		 feature-sig (substring feature-sig (match-end 0)))
	   (if (or (and regexp-flag
			(not (string-match "\\`\\\\\\[\\|::" feature-regexp)))
		   (not (or regexp-flag
			    (string-match "\\`\\[\\|::" feature-tag))))
	       (setq found (re-search-forward
			    (python-class-definition-regexp class regexp-flag)
			    nil t)
		     start (match-beginning 0)))))
    ;;
    ;; If class was searched for and not found, then skip down to code display.
    (if (not found)
	nil
      ;; Otherwise, look for feature expression.
      (setq found nil)
      (or regexp-flag (setq feature-regexp
			    (python-feature-signature-to-regexp feature-sig)))
      (while (and (re-search-forward feature-regexp nil t)
		  (setq start (match-beginning 0))
		  (not (setq found
			     (not (cond ((python-within-comment-p) t)
					((python-within-string-p)
					 (progn (re-search-forward
						 python-multi-line-string-delimiter
						 nil t)
						t)))))))))
    (if found (br-display-code start))))

(defun python-feature-name-to-regexp (name)
  "Converts routine NAME into a regular expression matching the routine's name tag."
  (setq name (python-feature-signature-to-regexp name))
  (aset name (1- (length name)) ?\()  ;; Match only to functions
  name)

(defun python-feature-signature-to-regexp (signature)
  "Given a Python SIGNATURE, return regexp used to match to its definition."
  (setq signature (regexp-quote signature))
  (let ((prefix-info
	 (if (string-match python-tag-fields-regexp signature)
	     (prog1 (substring signature (match-beginning 0) (match-end 0))
	       (setq signature (substring signature (match-end 0)))))))
    (let ((pat) (i 0) (c) (len (length signature)))
      (while (< i len)
	(setq c (aref signature i)
	      pat (cond ((eq c ? )
			 ;; Allow for possible single line comment
			 ;; following any whitespace, e.g. following
			 ;; each routine argument.
			 (concat pat "[ \t\n\r]*\\(#.*\\)?"))
			(t
			 (concat pat (char-to-string c))))
	      i (1+ i)))
      (setq pat (concat prefix-info pat)))))




;;; ************************************************************************
;;; Private variables
;;; ************************************************************************

(defvar python-docstring ""
  "Documentation string for python class, method or function.")

(defconst python-code-file-regexp "\\.py\\'"
  "Regexp which matches a unique part of a Python source (non-header) file name and no others.")

(defconst python-import-modules-regexp
  (concat "^[ \t]*import[ \t]+" python-identifier "[^#\n\r]+")
  "Regexp which matches Python module import statements.
Grouping `python-import-name-grpn' matches the first import module name.")

(defconst python-import-functions-regexp
  (concat "^[ \t]*from[ \t]+" python-identifier "[ \t\n\r]+import[ \t]+[^#\n\r]+")
  "Regexp which matches Python function import statements.
Grouping `python-import-name-grpn' matches the import module name.")

(defconst python-routine-def
  (concat "^def[ \t]+" python-identifier "[ \t\n\r]*[^:]+:")
  "Regexp which matches a global python function definition.
Grouping `python-feature-name-grpn' matches the function name.
After a forward search, point is left in front of the opening parameter list
parenthesis.")

(defconst python-routine-def-in-class
  (concat "^[ \t]+def[ \t]+" python-identifier "[ \t\n\r]*[^:]+:")
  "Regexp which matches a python class method definition.
Grouping `python-feature-name-grpn' matches the function name.
After a forward search, point is left in front of the opening parameter list
parenthesis.")

(defconst python-feature-name-grpn 1
  "Feature name grouping from `python-routine-def' and `python-routine-def-in-class' matches.")

(defconst python-import-name-grpn 1
  "Module name regexp grouping for import statements.")

(defconst python-multi-line-string-delimiter "'''\\|\"\"\""
  "Regexp matching a Python multi-line string Start or end delimiter.")

(provide 'br-python-ft)
