/*
             Common Public License Version 0.5

             THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF
             THIS COMMON PUBLIC LICENSE ("AGREEMENT"). ANY USE,
             REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
             RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.

             1. DEFINITIONS

             "Contribution" means: 
                   a) in the case of the initial Contributor, the
                   initial code and documentation distributed under
                   this Agreement, and 

                   b) in the case of each subsequent Contributor:
                   i) changes to the Program, and
                   ii) additions to the Program;

                   where such changes and/or additions to the Program
                   originate from and are distributed by that
                   particular Contributor. A Contribution 'originates'
                   from a Contributor if it was added to the Program
                   by such Contributor itself or anyone acting on such
                   Contributor's behalf. Contributions do not include
                   additions to the Program which: (i) are separate
                   modules of software distributed in conjunction with
                   the Program under their own license agreement, and
                   (ii) are not derivative works of the Program.


             "Contributor" means any person or entity that distributes
             the Program.

             "Licensed Patents " mean patent claims licensable by a
             Contributor which are necessarily infringed by the use or
             sale of its Contribution alone or when combined with the
             Program. 

             "Program" means the Contributions distributed in
             accordance with this Agreement.

             "Recipient" means anyone who receives the Program under
             this Agreement, including all Contributors.

             2. GRANT OF RIGHTS

                   a) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free copyright
                   license to reproduce, prepare derivative works of,
                   publicly display, publicly perform, distribute and
                   sublicense the Contribution of such Contributor, if
                   any, and such derivative works, in source code and
                   object code form.

                   b) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free patent
                   license under Licensed Patents to make, use, sell,
                   offer to sell, import and otherwise transfer the
                   Contribution of such Contributor, if any, in source
                   code and object code form. This patent license
                   shall apply to the combination of the Contribution
                   and the Program if, at the time the Contribution is
                   added by the Contributor, such addition of the
                   Contribution causes such combination to be covered
                   by the Licensed Patents. The patent license shall
                   not apply to any other combinations which include
                   the Contribution. No hardware per se is licensed
                   hereunder.

                   c) Recipient understands that although each
                   Contributor grants the licenses to its
                   Contributions set forth herein, no assurances are
                   provided by any Contributor that the Program does
                   not infringe the patent or other intellectual
                   property rights of any other entity. Each
                   Contributor disclaims any liability to Recipient
                   for claims brought by any other entity based on
                   infringement of intellectual property rights or
                   otherwise. As a condition to exercising the rights
                   and licenses granted hereunder, each Recipient
                   hereby assumes sole responsibility to secure any
                   other intellectual property rights needed, if any.

                   For example, if a third party patent license is
                   required to allow Recipient to distribute the
                   Program, it is Recipient's responsibility to
                   acquire that license before distributing the
                   Program.

                   d) Each Contributor represents that to its
                   knowledge it has sufficient copyright rights in its
                   Contribution, if any, to grant the copyright
                   license set forth in this Agreement.

             3. REQUIREMENTS

             A Contributor may choose to distribute the Program in
             object code form under its own license agreement, provided
             that:
                   a) it complies with the terms and conditions of
                   this Agreement; and

                   b) its license agreement:
                   i) effectively disclaims on behalf of all
                   Contributors all warranties and conditions, express
                   and implied, including warranties or conditions of
                   title and non-infringement, and implied warranties
                   or conditions of merchantability and fitness for a
                   particular purpose;

                   ii) effectively excludes on behalf of all
                   Contributors all liability for damages, including
                   direct, indirect, special, incidental and
                   consequential damages, such as lost profits;

                   iii) states that any provisions which differ from
                   this Agreement are offered by that Contributor
                   alone and not by any other party; and

                   iv) states that source code for the Program is
                   available from such Contributor, and informs
                   licensees how to obtain it in a reasonable manner
                   on or through a medium customarily used for
                   software exchange.

             When the Program is made available in source code form:
                   a) it must be made available under this Agreement;
                   and
                   b) a copy of this Agreement must be included with
                   each copy of the Program. 

             Contributors may not remove or alter any copyright notices
             contained within the Program.

             Each Contributor must identify itself as the originator of
             its Contribution, if any, in a manner that reasonably
             allows subsequent Recipients to identify the originator of
             the Contribution. 


             4. COMMERCIAL DISTRIBUTION

             Commercial distributors of software may accept certain
             responsibilities with respect to end users, business
             partners and the like. While this license is intended to
             facilitate the commercial use of the Program, the
             Contributor who includes the Program in a commercial
             product offering should do so in a manner which does not
             create potential liability for other Contributors.
             Therefore, if a Contributor includes the Program in a
             commercial product offering, such Contributor ("Commercial
             Contributor") hereby agrees to defend and indemnify every
             other Contributor ("Indemnified Contributor") against any
             losses, damages and costs (collectively "Losses") arising
             from claims, lawsuits and other legal actions brought by a
             third party against the Indemnified Contributor to the
             extent caused by the acts or omissions of such Commercial
             Contributor in connection with its distribution of the
             Program in a commercial product offering. The obligations
             in this section do not apply to any claims or Losses
             relating to any actual or alleged intellectual property
             infringement. In order to qualify, an Indemnified
             Contributor must: a) promptly notify the Commercial
             Contributor in writing of such claim, and b) allow the
             Commercial Contributor to control, and cooperate with the
             Commercial Contributor in, the defense and any related
             settlement negotiations. The Indemnified Contributor may
             participate in any such claim at its own expense.


             For example, a Contributor might include the Program in a
             commercial product offering, Product X. That Contributor
             is then a Commercial Contributor. If that Commercial
             Contributor then makes performance claims, or offers
             warranties related to Product X, those performance claims
             and warranties are such Commercial Contributor's
             responsibility alone. Under this section, the Commercial
             Contributor would have to defend claims against the other
             Contributors related to those performance claims and
             warranties, and if a court requires any other Contributor
             to pay any damages as a result, the Commercial Contributor
             must pay those damages.


             5. NO WARRANTY

             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, THE
             PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT
             WARRANTIES OR CONDITIONS OF ANY KIND, EITHER EXPRESS OR
             IMPLIED INCLUDING, WITHOUT LIMITATION, ANY WARRANTIES OR
             CONDITIONS OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR
             FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is solely
             responsible for determining the appropriateness of using
             and distributing the Program and assumes all risks
             associated with its exercise of rights under this
             Agreement, including but not limited to the risks and
             costs of program errors, compliance with applicable laws,
             damage to or loss of data, programs or equipment, and
             unavailability or interruption of operations. 

             6. DISCLAIMER OF LIABILITY
             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, NEITHER
             RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY
             FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
             OR CONSEQUENTIAL DAMAGES (INCLUDING WITHOUT LIMITATION
             LOST PROFITS), HOWEVER CAUSED AND ON ANY THEORY OF
             LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
             (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
             OF THE USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE
             OF ANY RIGHTS GRANTED HEREUNDER, EVEN IF ADVISED OF THE
             POSSIBILITY OF SUCH DAMAGES.

             7. GENERAL

             If any provision of this Agreement is invalid or
             unenforceable under applicable law, it shall not affect
             the validity or enforceability of the remainder of the
             terms of this Agreement, and without further action by the
             parties hereto, such provision shall be reformed to the
             minimum extent necessary to make such provision valid and
             enforceable.


             If Recipient institutes patent litigation against a
             Contributor with respect to a patent applicable to
             software (including a cross-claim or counterclaim in a
             lawsuit), then any patent licenses granted by that
             Contributor to such Recipient under this Agreement shall
             terminate as of the date such litigation is filed. In
             addition, If Recipient institutes patent litigation
             against any entity (including a cross-claim or
             counterclaim in a lawsuit) alleging that the Program
             itself (excluding combinations of the Program with other
             software or hardware) infringes such Recipient's
             patent(s), then such Recipient's rights granted under
             Section 2(b) shall terminate as of the date such
             litigation is filed.

             All Recipient's rights under this Agreement shall
             terminate if it fails to comply with any of the material
             terms or conditions of this Agreement and does not cure
             such failure in a reasonable period of time after becoming
             aware of such noncompliance. If all Recipient's rights
             under this Agreement terminate, Recipient agrees to cease
             use and distribution of the Program as soon as reasonably
             practicable. However, Recipient's obligations under this
             Agreement and any licenses granted by Recipient relating
             to the Program shall continue and survive. 

             Everyone is permitted to copy and distribute copies of
             this Agreement, but in order to avoid inconsistency the
             Agreement is copyrighted and may only be modified in the
             following manner. The Agreement Steward reserves the right
             to publish new versions (including revisions) of this
             Agreement from time to time. No one other than the
             Agreement Steward has the right to modify this Agreement.

             IBM is the initial Agreement Steward. IBM may assign the
             responsibility to serve as the Agreement Steward to a
             suitable separate entity. Each new version of the
             Agreement will be given a distinguishing version number.
             The Program (including Contributions) may always be
             distributed subject to the version of the Agreement under
             which it was received. In addition, after a new version of
             the Agreement is published, Contributor may elect to
             distribute the Program (including its Contributions) under
             the new version. Except as expressly stated in Sections
             2(a) and 2(b) above, Recipient receives no rights or
             licenses to the intellectual property of any Contributor
             under this Agreement, whether expressly, by implication,
             estoppel or otherwise. All rights in the Program not
             expressly granted under this Agreement are reserved.


             This Agreement is governed by the laws of the State of New
             York and the intellectual property laws of the United
             States of America. No party to this Agreement will bring a
             legal action under this Agreement more than one year after
             the cause of action arose. Each party waives its rights to
             a jury trial in any resulting litigation. 



*/

/* (C) COPYRIGHT International Business Machines Corp. 2001,2002          */


// File:  object.c
//
// Object manager related functions
//
// Functions contained within:
//
//    object_create
//    object_free
//    object_is_modifiable
//    object_is_private
//    object_is_token_object
//    object_is_session_object
//

#include <pthread.h>
#include <stdlib.h>
#include <memory.h>
#include <string.h>

#include "pkcs11types.h"
#include "defs.h"
#include "host_defs.h"
#include "h_extern.h"
#include "tok_spec_struct.h"
#include "pkcs32.h"
#include "trace.h"

// object_create()
//
// Args:   void *  attributes : (INPUT)  pointer to data block containing ATTRIBUTEs
//         OBJECT *       obj : (OUTPUT) destination object
//
// Creates an object with the specified attributes.  Verifies that all required
// attributes are present and adds any missing attributes that have Cryptoki-defined
// default values.  This routine does not check whether the session is authorized
// to create the object.  That is done elsewhere (see object_mgr_create())
//
CK_RV
object_create( CK_ATTRIBUTE  * pTemplate,
               CK_ULONG        ulCount,
               OBJECT       ** obj )
{
   OBJECT        * o           = NULL;
   CK_ATTRIBUTE  * attr        = NULL;
   CK_ATTRIBUTE  * sensitive   = NULL;
   CK_ATTRIBUTE  * extractable = NULL;
   CK_BBOOL        class_given = FALSE;
   CK_BBOOL        subclass_given = FALSE;
   CK_BBOOL        flag;
   CK_ULONG        class = 0xFFFFFFFF, subclass = 0xFFFFFFFF;
   CK_RV           rc;
   unsigned int    i;

   if (!pTemplate){
      TRACE_ERROR("Invalid function arguments.\n");
      return CKR_FUNCTION_FAILED;
   }
   // extract the object class and subclass
   //
   attr = pTemplate;
   for (i=0; i < ulCount; i++, attr++) {
      if (attr->type == CKA_CLASS) {
         class = *(CK_OBJECT_CLASS *)attr->pValue;
         class_given = TRUE;
      }

      if (attr->type == CKA_CERTIFICATE_TYPE) {
         subclass = *(CK_CERTIFICATE_TYPE *)attr->pValue;
         subclass_given = TRUE;
      }

      if (attr->type == CKA_KEY_TYPE) {
         subclass = *(CK_KEY_TYPE *)attr->pValue;
         subclass_given = TRUE;
      }

      if (attr->type == CKA_HW_FEATURE_TYPE) {
	 subclass = *(CK_HW_FEATURE_TYPE *)attr->pValue;
	 subclass_given = TRUE;
      }
   }

   if (class_given == FALSE){
      TRACE_ERROR("%s\n", ock_err(ERR_TEMPLATE_INCOMPLETE));
      return CKR_TEMPLATE_INCOMPLETE;
   }

	// Return CKR_ATTRIBUTE_TYPE_INVALID when trying to create a
	// vendor-defined object.
	if (class >= CKO_VENDOR_DEFINED) {
		TRACE_ERROR("%s\n", ock_err(ERR_ATTRIBUTE_TYPE_INVALID));
		return CKR_ATTRIBUTE_TYPE_INVALID;
	}

   if (class != CKO_DATA && subclass_given != TRUE){
      TRACE_ERROR("%s\n", ock_err(ERR_TEMPLATE_INCOMPLETE));
      return CKR_TEMPLATE_INCOMPLETE;
   }

   rc = object_create_skel( pTemplate, ulCount,
                            MODE_CREATE,
                            class, subclass,
                            &o );
   if (rc != CKR_OK){
      TRACE_DEVEL("object_create_skel failed.\n");
      return rc;
   }
   // for key objects, we need be careful...
   //
   // note:  I would think that keys loaded with C_CreateObject should
   //        have their CKA_NEVER_EXTRACTABLE == FALSE and
   //        CKA_ALWAYS_SENSITIVE == FALSE since the key data was presumably
   //        stored in the clear prior to the call to C_CreateObject.  The
   //        PKCS #11 spec doesn't impose this restriction however.
   //
   if (class == CKO_PRIVATE_KEY || class == CKO_SECRET_KEY) {
      rc = template_attribute_find( o->template, CKA_SENSITIVE, &attr );
      if (rc == FALSE) {
	 TRACE_ERROR("Failed to find CKA_SENSITIVE for the key.\n");
         rc = CKR_FUNCTION_FAILED;
         goto error;
      }

      flag = *(CK_BBOOL *)attr->pValue;

      rc = build_attribute( CKA_ALWAYS_SENSITIVE, &flag, sizeof(CK_BYTE), &sensitive );
      if (rc != CKR_OK){
         TRACE_DEVEL("build_attribute failed.\n");
         goto error;
      }

      rc = template_attribute_find( o->template, CKA_EXTRACTABLE, &attr );
      if (rc == FALSE) {
	 TRACE_ERROR("Failed to find CKA_EXTRACTABLE for the key.\n");
         rc = CKR_FUNCTION_FAILED;
         goto error;
      }

      flag = *(CK_BBOOL *)attr->pValue;
      flag = (~flag) & 0x1;

      rc = build_attribute( CKA_NEVER_EXTRACTABLE, &flag, sizeof(CK_BYTE), &extractable );
      if (rc != CKR_OK){
         TRACE_DEVEL("build attribute failed.\n");
         goto error;
      }
      template_update_attribute( o->template, sensitive );
      template_update_attribute( o->template, extractable );
   }

   *obj = o;

   return CKR_OK;

error:
   if (sensitive)    free( sensitive );
   if (extractable)  free( extractable );

   object_free( o );
   return rc;
}


// object_copy()
//
// Args:   OBJECT *   old_obj : (INPUT)  pointer to the source object
//         void *  attributes : (INPUT)  pointer to data block containing additional ATTRIBUTEs
//         CK_ULONG     count : (INPUT)  number of new attributes
//         OBJECT **  new_obj : (OUTPUT) destination object
//
// Builds a copy of the specified object.  The new object gets the original
// object's attribute template plus any additional attributes that are specified.
// Verifies that all required attributes are present.  This routine does not
// check whether the session is authorized to copy the object -- routines at
// the individual object level don't have the concept of "session".  These checks
// are done by the object manager.
//
CK_RV
object_copy( CK_ATTRIBUTE  * pTemplate,
             CK_ULONG        ulCount,
             OBJECT        * old_obj,
             OBJECT       ** new_obj )
{
   TEMPLATE  * tmpl     = NULL;
   TEMPLATE  * new_tmpl = NULL;
   OBJECT    * o        = NULL;
   CK_BBOOL    found;
   CK_ULONG    class, subclass;
   CK_RV       rc;


   if (!old_obj || !pTemplate || !new_obj){
      TRACE_ERROR("Invalid function arguments.\n");
      return CKR_FUNCTION_FAILED; 
   }
   o        = (OBJECT   *)malloc(sizeof(OBJECT));
   tmpl     = (TEMPLATE *)malloc(sizeof(TEMPLATE));
   new_tmpl = (TEMPLATE *)malloc(sizeof(TEMPLATE));

   if (!o || !tmpl || !new_tmpl) {
      rc = CKR_HOST_MEMORY;
      TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
      if (o)        free(o);
      if (tmpl)     free(tmpl);
      if (new_tmpl) free(new_tmpl);
      return rc; // do not goto done -- memory might not be initialized
   }

   memset( o,        0x0, sizeof(OBJECT) );
   memset( tmpl,     0x0, sizeof(TEMPLATE) );
   memset( new_tmpl, 0x0, sizeof(TEMPLATE) );

   // copy the original object's attribute template
   //
   rc = template_copy( tmpl, old_obj->template );
   if (rc != CKR_OK){
      TRACE_DEVEL("Failed to copy template.\n");
      goto error;
   }
   rc = template_add_attributes( new_tmpl, pTemplate, ulCount );
   if (rc != CKR_OK){
      TRACE_DEVEL("template_add_attributes failed.\n");
      goto error;
   }
   // at this point, the new object has the list of attributes.  we need
   // to do some more checking now:
   //    1) invalid attribute values
   //    2) missing required attributes
   //    3) attributes inappropriate for the object class
   //    4) conflicting attributes/values
   //

   found = template_get_class( tmpl, &class, &subclass );
   if (found == FALSE) {
      TRACE_ERROR("Could not find CKA_CLASS in object's template.\n");
      rc = CKR_TEMPLATE_INCONSISTENT;
      goto error;
   }


   // the user cannot change object classes so we assume the existing
   // object attributes are valid.  we still need to check the new attributes.
   // we cannot merge the new attributes in with the old ones and then check
   // for validity because some attributes are added internally and are not
   // allowed to be specified by the user (ie. CKA_LOCAL for key types) but
   // may still be part of the old template.
   //
   rc = template_validate_attributes( new_tmpl, class, subclass, MODE_COPY );
   if (rc != CKR_OK){
      TRACE_DEVEL("template_validate_attributes failed.\n");
      goto error;
   }
   // merge in the new attributes
   //
   rc = template_merge( tmpl, &new_tmpl );
   if (rc != CKR_OK){
      TRACE_DEVEL("template_merge failed.\n");
      goto error;
   }
   // do we need this?  since an attribute cannot be removed, the original
   // object's template (contained in tmpl) already has the required attributes
   // present
   //
   rc = template_check_required_attributes( tmpl, class, subclass, MODE_COPY );
   if (rc != CKR_OK){
      TRACE_ERROR("template_check_required_attributes failed.\n");
      goto error;
   }
   // at this point, we should have a valid object with correct attributes
   //
   o->template = tmpl;
   *new_obj = o;

   return CKR_OK;

error:
   if (tmpl)     template_free( tmpl );
   if (new_tmpl) template_free( new_tmpl );
   if (o)        object_free( o );

   return rc;
}


// object_flatten() - this is still used when saving token objects
//
CK_RV
object_flatten( OBJECT    * obj,
                CK_BYTE  ** data,
                CK_ULONG  * len )
{
   CK_BYTE    * buf = NULL;
   CK_ULONG     tmpl_len, total_len;
   CK_ULONG     offset;
   CK_ULONG_32     count;
   long         rc;

   if (!obj){
      TRACE_ERROR("Invalid function arguments.\n");
      return CKR_FUNCTION_FAILED;
   }
   count    = template_get_count( obj->template );
   tmpl_len = template_get_compressed_size ( obj->template );

   total_len = tmpl_len + sizeof(CK_OBJECT_CLASS_32) + sizeof(CK_ULONG_32) + 8;

   buf = (CK_BYTE *)malloc(total_len);
   if (!buf){ // SAB  XXX FIXME  This was DATA
      TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
      return CKR_HOST_MEMORY;
   }

   memset( (CK_BYTE *)buf,0x0,total_len);

   offset = 0;

   memcpy( buf + offset, &obj->class, sizeof(CK_OBJECT_CLASS_32) );
   offset += sizeof(CK_OBJECT_CLASS_32);

   memcpy( buf + offset, &count, sizeof(CK_ULONG_32) );
   offset += sizeof(CK_ULONG_32);

   memcpy( buf + offset, &obj->name,  sizeof(CK_BYTE) * 8 );
   offset += 8;
   rc = template_flatten( obj->template, buf + offset );
   if (rc != CKR_OK) {
      free( buf );
      return rc;
   }

   *data = buf;
   *len  = total_len;

   return CKR_OK;
}



// object_free()
//
// does what it says...
//
void object_free(OBJECT *obj)
{
	/* refactorization here to do actual free - fix from coverity scan */
	if (obj) {
		if (obj->template)
			template_free(obj->template);
		free(obj);
	}
}

//call_free()
//This function is added to silence the compiler during implicit void (*)(void*) function pointer casting in call back functions.
//
void call_free(void *ptr)
{
	if (ptr)
		object_free ((OBJECT*) ptr);
}

// object_is_modifiable()
//
CK_BBOOL
object_is_modifiable( OBJECT *obj )
{
   CK_ATTRIBUTE  * attr = NULL;
   CK_BBOOL        modifiable;
   CK_BBOOL        found;

   found = template_attribute_find( obj->template, CKA_MODIFIABLE, &attr );
   if (found == FALSE)
      return TRUE;      // should always be found but we default to TRUE

   //axelrh: prevent dereferencing NULL from bad parse
   if (attr->pValue == NULL)
	return TRUE; //default to TRUE

   modifiable = *(CK_BBOOL *)attr->pValue;

   return modifiable;
}


// object_is_private()
//
// an is_private member should probably be added to OBJECT
//
CK_BBOOL
object_is_private( OBJECT *obj )
{
   CK_ATTRIBUTE  * attr = NULL;
   CK_BBOOL        priv;
   CK_BBOOL        found;

   found = template_attribute_find( obj->template, CKA_PRIVATE, &attr );
   if (found == FALSE){
      return TRUE;      // should always be found but we default to TRUE
   }
   if ( attr == NULL)  return  TRUE;
  

   //axelrh: prevent segfault caused by 0-len attribute
   //that has a null pValue
   CK_BBOOL *bboolPtr = (CK_BBOOL *)attr->pValue;
   if (bboolPtr == NULL)
       return TRUE; //default

   priv = *(bboolPtr);

   return priv;
}


// object_is_public()
//
CK_BBOOL
object_is_public( OBJECT *obj )
{
   CK_BBOOL rc;

   rc = object_is_private( obj );

   if (rc)
      return FALSE;

   return TRUE;
}


// object_is_token_object()
//
CK_BBOOL
object_is_token_object( OBJECT *obj )
{
   CK_ATTRIBUTE  * attr = NULL;
   CK_BBOOL        is_token;
   CK_BBOOL        found;

   found = template_attribute_find( obj->template, CKA_TOKEN, &attr );
   if (found == FALSE)
      return FALSE;

   //axelrh: prevent dereferencing NULL from bad parse
   if (attr->pValue == NULL)
	return FALSE;

   is_token = *(CK_BBOOL *)attr->pValue;
   return is_token;
}


// object_is_session_object()
//
CK_BBOOL
object_is_session_object( OBJECT *obj )
{
   CK_BBOOL rc;

   rc = object_is_token_object( obj );

   if (rc)
      return FALSE;
   else
      return TRUE;
}


// object_get_size()
//
CK_ULONG
object_get_size( OBJECT *obj )
{
   CK_ULONG  size;

   size = sizeof(OBJECT) + template_get_size(obj->template);

   return size;
}


//
//
CK_RV
object_get_attribute_values( OBJECT        * obj,
                             CK_ATTRIBUTE  * pTemplate,
                             CK_ULONG        ulCount )
{
   TEMPLATE          *obj_tmpl = NULL;
   CK_ATTRIBUTE      *attr     = NULL;
   CK_ULONG           i;
   CK_BBOOL           flag;
   CK_RV              rc;

   rc = CKR_OK;

   obj_tmpl = obj->template;

   for (i=0; i < ulCount; i++) {
      flag = template_check_exportability( obj_tmpl, pTemplate[i].type);
      if (flag == FALSE) {
         TRACE_ERROR("%s\n", ock_err(ERR_ATTRIBUTE_SENSITIVE));
         rc = CKR_ATTRIBUTE_SENSITIVE;
         pTemplate[i].ulValueLen = (CK_ULONG)-1;
         continue;
      }

      flag = template_attribute_find( obj_tmpl, pTemplate[i].type, &attr );
      if (flag == FALSE) {
         TRACE_ERROR("%s\n", ock_err(ERR_ATTRIBUTE_TYPE_INVALID));
         rc = CKR_ATTRIBUTE_TYPE_INVALID;
         pTemplate[i].ulValueLen = (CK_ULONG)-1;
         continue;
      }

      if (pTemplate[i].pValue == NULL) {
         pTemplate[i].ulValueLen = attr->ulValueLen;
      }
      else if (pTemplate[i].ulValueLen >= attr->ulValueLen) {
         memcpy( pTemplate[i].pValue, attr->pValue, attr->ulValueLen );
         pTemplate[i].ulValueLen = attr->ulValueLen;
      }
      else {
         TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
         rc = CKR_BUFFER_TOO_SMALL;
         pTemplate[i].ulValueLen = (CK_ULONG)-1;
      }
   }

   return rc;
}

// object_set_attribute_values()
//
CK_RV
object_set_attribute_values( OBJECT        * obj,
                             CK_ATTRIBUTE  * pTemplate,
                             CK_ULONG        ulCount )
{
   TEMPLATE * new_tmpl = NULL;
   CK_BBOOL   found;
   CK_ULONG   class, subclass;
   CK_RV      rc;


   if (!obj || !pTemplate){
      TRACE_ERROR("Invalid function arguments.\n");
      return CKR_FUNCTION_FAILED;
   }

   found = template_get_class( obj->template, &class, &subclass );
   if (found == FALSE) {
      TRACE_ERROR("Failed to find CKA_CLASS in object template.\n");
      rc = CKR_FUNCTION_FAILED;
      goto error;
   }

   new_tmpl = (TEMPLATE *)malloc(sizeof(TEMPLATE));
   if (!new_tmpl){
      TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
      return CKR_HOST_MEMORY;
   }
   memset( new_tmpl, 0x0, sizeof(TEMPLATE) );

   rc = template_add_attributes( new_tmpl, pTemplate, ulCount );
   if (rc != CKR_OK){
      TRACE_DEVEL("template_add_attributes failed.\n");
      goto error;
   }

   // the user cannot change object classes so we assume the existing
   // object attributes are valid.  we still need to check the new attributes.
   // we cannot merge the new attributes in with the old ones and then check
   // for validity because some attributes are added internally and are not
   // allowed to be specified by the user (ie. CKA_LOCAL for key types) but
   // may still be part of the old template.
   //
   rc = template_validate_attributes( new_tmpl, class, subclass, MODE_MODIFY );
   if (rc != CKR_OK){
      TRACE_DEVEL("template_validate_attributes failed.\n");
      goto error;
   }

   // merge in the new attributes
   //
   rc = template_merge( obj->template, &new_tmpl );
   if (rc != CKR_OK){
      TRACE_DEVEL("template_merge failed.\n");
      return rc;
   }
   return CKR_OK;

error:
   // we only free the template if there was an error...otherwise the
   // object "owns" the template
   //
   if (new_tmpl)  template_free( new_tmpl );
   return rc;
}


//
//
CK_RV
object_restore( CK_BYTE *data, OBJECT **new_obj, CK_BBOOL replace )
{
   return object_restore_withSize(data, new_obj, replace, -1);
}

//
//Modified object_restore to prevent buffer overflow
//If data_size=-1, won't do bounds checking
CK_RV
object_restore_withSize( CK_BYTE *data, OBJECT **new_obj, CK_BBOOL replace, int data_size )
{
   TEMPLATE  * tmpl = NULL;
   OBJECT    * obj  = NULL;
   CK_ULONG    offset = 0;
   CK_ULONG_32    count = 0;
   CK_RV       rc;

   if (!data || !new_obj){
      TRACE_ERROR("Invalid function arguments.\n");
      return CKR_FUNCTION_FAILED;
   }
   obj = (OBJECT *)malloc(sizeof(OBJECT));
   if (!obj) {
      TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
      rc = CKR_HOST_MEMORY;
      goto error;
   }

  
   memset( obj, 0x0, sizeof(OBJECT) );

   memcpy( &obj->class, data + offset, sizeof(CK_OBJECT_CLASS_32) );
   offset += sizeof(CK_OBJECT_CLASS_32);

   memcpy( &count, data + offset, sizeof(CK_ULONG_32) );
   offset += sizeof(CK_ULONG_32);


   memcpy( &obj->name, data + offset, 8 );
   offset += 8;

   rc = template_unflatten_withSize( &tmpl, data + offset, count, data_size );
   if (rc != CKR_OK){
      TRACE_DEVEL("template_unflatten_withSize failed.\n");
      goto error;
   }
   obj->template = tmpl;

   if (replace == FALSE) {
      *new_obj = obj;
   }
   else {
      template_free( (*new_obj)->template );
      memcpy( *new_obj, obj, sizeof(OBJECT) );

      free( obj );  // don't want to do object_free() here!
   }

   return CKR_OK;

error:
   if (obj)  object_free( obj );
   if (tmpl) template_free( tmpl );

   return rc;
}


//
//
CK_RV
object_create_skel( CK_ATTRIBUTE  * pTemplate,
                    CK_ULONG        ulCount,
                    CK_ULONG        mode,
                    CK_ULONG        class,
                    CK_ULONG        subclass,
                    OBJECT       ** obj )
{
   TEMPLATE  * tmpl  = NULL;
   TEMPLATE  * tmpl2 = NULL;
   OBJECT   * o     = NULL;
   CK_RV      rc;


   if (!obj){
      TRACE_ERROR("Invalid function arguments.\n");
      return CKR_FUNCTION_FAILED;
   }
   if (!pTemplate && (ulCount != 0)){
      TRACE_ERROR("Invalid function arguments.\n");
      return CKR_FUNCTION_FAILED;
   }
   o     = (OBJECT *)malloc(sizeof(OBJECT));
   tmpl  = (TEMPLATE *)malloc(sizeof(TEMPLATE));
   tmpl2 = (TEMPLATE *)malloc(sizeof(TEMPLATE));

   if (!o || !tmpl || !tmpl2) {
      TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
      rc = CKR_HOST_MEMORY;
      goto done;
   }
   memset( o,     0x0, sizeof(OBJECT)   );
   memset( tmpl,  0x0, sizeof(TEMPLATE) );
   memset( tmpl2, 0x0, sizeof(TEMPLATE) );


   rc = template_add_attributes( tmpl2, pTemplate, ulCount );
   if (rc != CKR_OK)
      goto done;


   // at this point, the new template has the list of attributes.  we need
   // to do some more checking now:
   //    1) invalid attribute values
   //    2) missing required attributes
   //    3) attributes inappropriate for the object class
   //    4) conflicting attributes/values
   //

   rc = template_validate_attributes( tmpl2, class, subclass, mode );
   if (rc != CKR_OK){
      TRACE_DEVEL("template_validate_attributes failed.\n");
      goto done;
   }

   rc = template_check_required_attributes( tmpl2, class, subclass, mode );
   if (rc != CKR_OK){
      TRACE_DEVEL("template_check_required_attributes failed.\n");
      goto done;
   }

   rc = template_add_default_attributes( tmpl, tmpl2, class, subclass, mode );
   if (rc != CKR_OK)
      goto done;


   rc = template_merge( tmpl, &tmpl2 );
   if (rc != CKR_OK){
      TRACE_DEVEL("template_merge failed.\n");
      goto done;
   }
   // at this point, we should have a valid object with correct attributes
   //
   o->template = tmpl;
   *obj = o;

   return CKR_OK;

done:
   if (o)      free( o );
   if (tmpl)   template_free( tmpl );
   if (tmpl2)  template_free( tmpl2 );

   return rc;
}
