/*
 * fade.cc --
 *
 *      A fade effect
 *
 * Copyright (c) 1993-2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) $Header: /usr/mash/src/repository/mash/mash-1/fx/effects/fade.cc,v 1.1 2002/02/07 04:18:19 chema Exp $ */


#include <dvmbasic.h>
#include "fade.h"



FadeEffect::FadeEffect(int decimation) 
	: EffectModule(2, decimation)
{
	factor_.value = 0.0;
	factor_.from = 0.0;
	factor_.to = 1.0;
	factor_.incr = 1;

	// create the temporal frames
	temp1_ = new Uncompressed;
	temp2_ = new Uncompressed;

	return;
}



int FadeEffect::command(int argc, const char*const* argv)
{
	return (EffectModule::command(argc, argv));
}



void FadeEffect::trigger_effect (Uncompressed* input)
{
	//printf ("w(%dx%d) ", input->w_, input->h_); fflush (stdout); // XXX

	// temp1 has to be the same size than the input
	if (!temp1_->samesize(input)) {
		temp1_->init (input, csss_, VIDREP_ALLOC_MEMORY);
	}
	// temp2 has to be the same size than the output
	if ((output_->w_ != temp2_->w_) || (output_->h_ != temp2_->h_)) {
		temp2_->init (output_, csss_, VIDREP_ALLOC_MEMORY);
	}

	// calculate the fading factor
	if (factor_.incr == 1) {
		factor_.value = factor_.value + 0.05;
	} else {
		factor_.value = factor_.value - 0.05;
	}
	if (factor_.value > factor_.to) {
		factor_.value = factor_.to;
		factor_.incr = 0;
	} else if (factor_.value < factor_.from) {
		factor_.value = factor_.from;
		factor_.incr = 1;
	}

	// step 1: temp1 = input * factor
	ByteImage *in_l = input->lum_;
	ByteImage *in_cr = input->cr_;
	ByteImage *in_cb = input->cb_;
	ByteImage *temp1_l = temp1_->lum_;
	ByteImage *temp1_cr = temp1_->cr_;
	ByteImage *temp1_cb = temp1_->cb_;
	ByteMultiply(in_l, factor_.value, temp1_l);
	ByteMultiply(in_cr, factor_.value, temp1_cr);
	ByteMultiply(in_cb, factor_.value, temp1_cb);

	// step 2: temp2 = buffer * (1 - factor)
	ByteImage *buf_l = buffer_[0]->lum_;
	ByteImage *buf_cr = buffer_[0]->cr_;
	ByteImage *buf_cb = buffer_[0]->cb_;
	ByteImage *temp2_l = temp2_->lum_;
	ByteImage *temp2_cr = temp2_->cr_;
	ByteImage *temp2_cb = temp2_->cb_;
	ByteMultiply(buf_l, (1.0 - factor_.value), temp2_l);
	ByteMultiply(buf_cr, (1.0 - factor_.value), temp2_cr);
	ByteMultiply(buf_cb, (1.0 - factor_.value), temp2_cb);

	// step 3: output = temp1 + temp2
	ByteImage *out_l = output_->lum_;
	ByteImage *out_cr = output_->cr_;
	ByteImage *out_cb = output_->cb_;
	ByteAdd(temp1_l, temp2_l, out_l);
	ByteAdd(temp1_cr, temp2_cr, out_cr);
	ByteAdd(temp1_cb, temp2_cb, out_cb);

	// deliver the output frame
	deliver_frame();

	return;
}

