/*
 * mb-file.cc --
 *
 *      fragementation / assembly routines for mb
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef MB_FILE_CC
#define MB_FILE_CC

#include "mb.h"

#ifndef lint
static const char rcsid[] = "$Header: /usr/mash/src/repository/mash/mash-1/mb/mb-file.cc,v 1.15 2002/02/03 03:16:30 lim Exp $";
#endif

#include "mb-file.h"

ulong MBFile::nextTmpNo=0;

Bool MBFile::getNextFrag(int maxFragLen, MBFrag* pFrag)
{
    return (pFrag->Read(channel_, maxFragLen));
}

// takes either -file <filename> or <filename>
// returns FALSE if it encounters any error
Bool MBFile::Init(int argc, const char*const* argv) {
    const char* szFileName=NULL;
    if (argc==1)
        szFileName = argv[0];
    else if ((argc==2) && !strcmp(argv[0],"-file"))
        szFileName = argv[1];

    if (!szFileName || (*szFileName==cchNull)) {
        Tcl_AddErrorInfo(MB_Interp, "No file name specified!\n");
        return FALSE;
    }

    channel_ = Tcl_OpenFileChannel(MB_Interp, (char*)szFileName, "r", 0);
    if (0==channel_) {
        int errnum=Tcl_GetErrno();
        char szTmp[150];
        sprintf(szTmp,"File %s could not be opened, POSIX err:%d\n",
                szFileName, errnum);
        Tcl_AddErrorInfo(MB_Interp, szTmp);
        return FALSE;
    }
    Tcl_SetChannelOption(MB_Interp, channel_, "-translation", "binary");
    return TRUE;
}

//
// return the name of temporary file the fragments are saved into
//
/* static */
const char* MBFile::Assemble(int nFrag, MBFrag** ppFragArg,
                             const char* szFileName/*=NULL*/)
{
#ifndef WIN32
    /* REVIEW: consider changing to tempnam, which seems quite standard */
    static char s_szTmpFN[MAX_N_LONG_CHAR+4];

    const char sztmpFFmt[] = "/tmp/mb%lu.tmp";
    MB_DefTcl(tcl);
    if (NULL==szFileName) {
        for(;;) {                    // finds the next tmp file name to use
            sprintf(s_szTmpFN, sztmpFFmt, nextTmpNo++);
            szFileName = s_szTmpFN;
            tcl.evalf("file exists %s", szFileName);
            if (!strcmp(tcl.result(),"0")) {
                    break;
            }
        }
    }
#else // WIN32
    static char s_szTmpFN[_MAX_PATH];
    // we use tempnam rather them tmp because tmpnam defaults to '\' which
    // is not overwritable by the user, plus we want to distinguish
    // mb temp files
    if (NULL==szFileName) {
            char* szTmp = _tempnam(NULL, "mb");
            strcpy(s_szTmpFN, szTmp);
            free(szTmp);                // _tempnam uses malloc, so use free
            szFileName = s_szTmpFN;
    }
#endif // WIN32

    Tcl_Channel tmpChannel = Tcl_OpenFileChannel(MB_Interp, (char*)szFileName,
                                                 "w+", 0600);
    Tcl_CallWhenDeleted(MB_Interp, deleteTemp ,
                         (ClientData)strdup(szFileName));
    if (NULL==tmpChannel) {
        int errnum=Tcl_GetErrno();
        SignalError(("Error: file %s could not be opened, POSIX err:%d\n",
                     szFileName, errnum));
        return NULL;
    }
    Tcl_SetChannelOption(MB_Interp, tmpChannel, "-translation", "binary");

    for (MBFrag** ppFrag=ppFragArg; ppFrag<ppFragArg+nFrag; ppFrag++) {
        if (-1==(*ppFrag)->Write(tmpChannel)) {
            SignalError( ("Error writing to temp file %s, errnum=%d",
                          szFileName, Tcl_GetErrno()) );
            goto err;
        }
    }
    Tcl_Close(MB_Interp, tmpChannel);
    return szFileName;

err:
    Tcl_Close(MB_Interp, tmpChannel);
    return NULL;
}

/* Delete the temporary file using during a session, callback from
   the tcl interpreter when it exits */
/*static*/
void MBFile::deleteTemp(ClientData cd, Tcl_Interp* /*interp*/)
{
        char *szFN = (char*) cd;
        if (szFN && -1 == unlink(szFN)) {
                fprintf(stderr, "unlink of %s failed", szFN);
        }
	if (szFN) free(szFN);
}

#endif /* #ifdef MB_FILE_CC */
