# sds_service.tcl --
#
#       Base SDS service object
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import SRMv2_Session
import Timer/Periodic

Class CSdsService -superclass {SRMv2_Session Timer/Periodic}

#
# CRVCSession constructor
#
# input: inetAddr - a string with the internet address
#        iSPort - sending port
#        iRPort - receiving port
#


##############################################################################
#
# CSdsService instproc init { agent inetAddr iSPort iRPort iTTL} {
#
# Input:
# agent - the agent for this object, will be used to get more information
# inetAddr - the inet addr for sds broadcasts eg. "224.23.23.23"
# iSPort, iRPort - the send and receive ports for the sds broadcasts
# iTTL - the ttl for the broadcasts
#
# Output:
# none
#
# Description:
# Constructor.  Sets up the system
#
##############################################################################
CSdsService instproc init { agent inetAddr iSPort iRPort iTTL} {
    $self next $inetAddr $iSPort $iRPort $iTTL

    # create the one and only source object
    $self instvar m_agent
    $self instvar m_source
    $self instvar m_cidUpdate

    set m_agent $agent

    set m_source [new SRMv2_Source $self ]

    # set the source info
    $m_source app_info { "Service Source" }

    # create some default containers
    set m_cidUpdate [$m_source calloc 0 "Query Update"]

    # Ok now let's set the timer to go off periodically
    # when the timer goes off it will call timeout
    $self start 500
}

##############################################################################
#
# CSdsService instproc srm_recv { src cid seqno data } {
#
# Input:
# src, cid, seqno, data - same as SRMv2_Session object
#
# Output:
# none
#
# Description:
# Overloaded function of SRMv2_Session.  It should handle data that comes in,
# but for this function, we don't care about incomming data.
#
##############################################################################
CSdsService instproc srm_recv { src cid seqno data } {
}

##############################################################################
#
# CSdsService instproc srm_read_adu { src cid seqno } {
#
# Input:
# src, cid, seqno - same as SRMv2_Session object
#
# Output:
# none
#
# Description:
# Since we don't care about incomming data - we don't care about recovery
#
##############################################################################
CSdsService instproc srm_read_adu { src cid seqno } {
    # no recovery mechanism
}

##############################################################################
#
# CSdsService instproc srm_should_recover { src cid sseq eseq } {
#
# Input:
# src, cid, seqno, eseq - same as SRMv2_Session object
#
# Output:
# 0 = no data recovered
#
# Description:
# We're just retransmitting the same data so no recovery system is needed
#
##############################################################################
CSdsService instproc srm_should_recover { src cid sseq eseq } {
    return 0
}

##############################################################################
#
# CSdsService instproc timeout {} {
#
# Input:
# none
#
# Output:
# none
#
# Description:
# This is a callback function for the timer portion of this class.  It will
# be called periodically and will serve to periodically send the description
# of this service out to the service discovery service.
#
##############################################################################
CSdsService instproc timeout {} {
    $self instvar m_source
    $self instvar m_cidUpdate
    $self instvar m_agent

    set data [$m_agent GetSdsServiceData]

    $m_source send $m_cidUpdate $data
}
