# synch-switch.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1999-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import Observer
import SynchSrcVideoAgent
import VideoAgent

Class SynchSwitch;

SynchSwitch instproc init {app} {
    $self next;

    $self instvar sources_ synch_transmitters_ source_sessions_
    $self instvar master_lts_;
    $self instvar app_;
    $self instvar new_source_cmd_;

    set app_ $app;

    set sources_ "";
    set source_sessions_ "";
    set synch_transmitters_ "";

    set master_lts_ [new SynchLTS];
    $master_lts_ speed 1.0;
    $master_lts_ set_reference 0 0;

    set new_source_cmd_ "";

    $self instvar expansion_speed_ compression_speed_;

    set expansion_speed_ 0.9;
    set compression_speed_ 1.1;
}

SynchSwitch instproc register_source {sess src} {
    $self instvar sources_ master_lts_

    set new_src [new SynchSource $sess $src $master_lts_];
    lappend sources_ $new_src;

    $self instvar new_source_cmd_;

    if {$new_source_cmd_ != ""} {
	eval $new_source_cmd_ $new_src
    }

    return $new_src
}

SynchSwitch instproc unregister_source {src} {
    $self instvar sources_;

    set k [lsearch -exact $sources_ $src];
    set sources_ [lreplace $sources_ $k $k];

    delete $src;
}

Class SynchTransmitAgent -superclass VideoAgent

SynchTransmitAgent instproc create_decoder {src} {
    set decoder [new Module/VideoDecoder/Null];
    $decoder set src_ $src;
    $decoder set agent_ $self;
    return $decoder;
}

SynchTransmitter instproc init {switch spec} {
    $self instvar sess_ switch_ src_ srcid_ cur_src_ spec_;

    set spec_ $spec;
    set sess_ [new SynchTransmitAgent [$switch set app_] $spec];
    [$sess_ get_transmitter] set loopback_ 0;

    set src_ [$sess_ set local_];
    set srcid_ [$sess_ get_local_srcid];
    set cur_src_ "";

    $self target [$sess_ get_transmitter];
    $self reset;
}

Class SynchSource -superclass Observer

SynchSource instproc init {sess src master} {
    $self next;

    $self instvar sess_ src_ master_;

    set sess_ $sess;
    set src_ $src;
    set master_ $master;

    $self instvar synch_buffer_ lts_;

    set synch_buffer_ [new SynchBuffer];
    $synch_buffer_ set synch_src_ $self;

    set lts_ [new SynchLTS];

    $synch_buffer_ lts $lts_;
    $lts_ lts $master;
    $lts_ speed 1.0;
    $synch_buffer_ reset_reference;
    $sess_ attach $self
}

SynchSource public trigger_sr {src} {
    $self instvar src_;

    if {$src == $src_} {
	$self instvar synch_buffer_;

	$synch_buffer_ synch_to_source $src_;
    }
}

SynchSource instproc synch_buffer {} {
    $self instvar synch_buffer_;

    return $synch_buffer_;
}

SynchSource instproc buffer_latency {} {
    $self instvar synch_buffer_;
    return [$synch_buffer_ buffer_latency];
}

SynchSource instproc num_frames {} {
    $self instvar synch_buffer_;
    return [$synch_buffer_ num_frames];
}

SynchSource instproc destroy {} {
    $self instvar synch_buffer_ lts_;
    $lts_ lts "";
    $synch_buffer_ lts "";
    delete $lts_;
    delete $synch_buffer_;

    $self instvar sess_;
    $sess_ detach $self;

    $self next;
}

#################
# Below is API for Synch Switch although
# this is not strictly enforced (i.e.,
# procedures defined above are accessible).
################

# add_source_session
#    Add session for possible sources

SynchSwitch instproc add_source_session {spec} {
    $self instvar app_ source_sessions_;

    set new_sess [new SynchSrcVideoAgent $app_ $self $spec];
    lappend source_sessions_ $new_sess;
}

# new_source_command
#    Set callback for when new source is heard

SynchSwitch instproc new_source_command {args} {
    $self instvar new_source_cmd_;

    if {[llength $args] == 0} {
	return $new_source_cmd_;
    } else {
	set new_source_cmd_ [lindex $args 0];
    }
}

# sources
#    Return list of sources we know about.

SynchSwitch instproc sources {} {
    $self instvar sources_;

    return $sources_;
}

# create_synch_transmitter
#    Create a new source for transmission

SynchSwitch instproc create_synch_transmitter {spec} {
    $self instvar synch_transmitters_

    set tsrc [new SynchTransmitter $self $spec];

    lappend synch_transmitters_ $tsrc;
    return $tsrc;
}

# buffer_latency
#    Query current estimated buffer latency for a source.
#    If <src> is "", returns a list of source/latency pairs
#    for all known sources.

SynchSwitch instproc buffer_latency {{src ""}} {
    if {$src != ""} {
	return [$src buffer_latency];
    } else {
	$self instvar sources_;
	set res "";
	foreach s $sources_ {
	    lappend res [list $s [$s buffer_latency]];
	}
	return $res;
    }
}

# master_speed
#    Set/query master LTS speed

SynchSwitch instproc master_speed {{speed ""}} {
    $self instvar master_lts_;

    if {$speed == ""} {
	return [$master_lts_ speed];
    } else {
	$master_lts_ speed $speed;
    }
}

# master_offset_adjust
#    Adjust master logical offset by <delta>

SynchSwitch instproc master_offset_adjust {delta} {
    $self instvar master_lts_;

    $master_lts_ adjust $delta;
}

# master_time_expansion_speed
#    Set/query speed used for time expansion

SynchSwitch instproc master_time_expansion_speed {{speed ""}} {
    $self instvar expansion_speed_;

    if {$speed == ""} {
	return $expansion_speed_;
    } else {
	set expansion_speed_ $speed;
    }
}

# master_time_compression_speed
#    Set/query speed used for time compression

SynchSwitch instproc master_time_compression_speed {{speed ""}} {
    $self instvar compression_speed_;

    if {$speed == ""} {
	return $compression_speed_;
    } else {
	set compression_speed_ $speed;
    }
}

# prep_switch
#    Switch transmission to specified source after
#    eliminating negative latency. Positive latencies
#    eliminated after switch only if <no_positive> flag
#    set.

SynchSwitch instproc prep_switch {src transmitter {no_positive 0}} {
    $self instvar pending_action_ expansion_speed_ compression_speed_;

    if {[info exists pending_action_]} {
	after cancel $pending_action_;
	unset pending_action_;
	$self master_speed 1.0;
    }

    set sb [$src set synch_buffer_];
    set lat [$sb buffer_latency];
    if {$lat < -0.030} {
	$self master_speed $expansion_speed_;
	set delta [expr int((($lat * -1.0)/(1.0 - $expansion_speed_)) * 1000.0)];

	set pending_action_ [after $delta "$self straight_switch $src $transmitter"];
    } else {
	$self straight_switch $src $transmitter;
	if {($lat > 0.030) && $no_positive} {
	    $self master_speed $compression_speed_;
	    set delta [expr int(($lat / ($compression_speed_ - 1.0)) * 1000.0)];
	    set pending_action_ [after $delta "$self straight_switch $src $transmitter"];
	}
    }
}

# switch_prep
#    Switch transmission to specified source immediately,
#    eliminate negative latency after switch. Positive
#    latencies eliminated after switch only if <no_positive>
#    flag set.

SynchSwitch instproc switch_prep {src transmitter {no_positive 0}} {
    $self instvar pending_action_ expansion_speed_ compression_speed_;

    if {[info exists pending_action_]} {
	after cancel $pending_action_;
	unset pending_action_;
	$self master_speed 1.0;
    }

    $self straight_switch $src $transmitter;

    set sb [$src set synch_buffer_];
    set lat [$sb buffer_latency];
    if {$lat < -0.030} {
	$self master_speed $expansion_speed_;
	set delta [expr int((($lat * -1.0)/(1.0 - $expansion_speed_)) * 1000.0)];
	set pending_action_ [after $delta "$self straight_switch $src $transmitter"];
    } else {
	if {($lat > 0.030) && $no_positive} {
	    $self master_speed $compression_speed_;
	    set delta [expr int(($lat / ($compression_speed_ - 1.0)) * 1000.0)];
	    set pending_action_ [after $delta "$self straight_switch $src $transmitter"];
	}
    }
}

# straight_switch
#    Switch transmission without adjusting for latencies

SynchSwitch instproc straight_switch {src transmitter} {
    $self instvar sources_ pending_action_;

    if {[info exists pending_action_]} {
	after cancel $pending_action_;
	unset pending_action_;
	$self master_speed 1.0;
    }

    if {[$transmitter set cur_src_] == $src} {
	return;
    }

    set sb [$src set synch_buffer_];

    if {[lsearch -exact $sources_ [$transmitter set cur_src_]] != -1} {
	set old_sb [[$transmitter set cur_src_] set synch_buffer_];
	if {[$old_sb target] == $transmitter} {
	    $old_sb target "";
	}
	$transmitter set cur_src_ "";
    }
    $transmitter reset;
    $transmitter set cur_src_ $src;
    $sb target $transmitter;
}

