;;;-*- Mode: Lisp; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;


(eval-when (:compile-toplevel :load-toplevel :execute)
  (require "SPARC-ARCH")
  (require "DLL-NODE")
  (require "SPARC-ASM")
  (require "SPARC-LAP"))


(defun sparc-gpr (r)
  (svref sparc::*gpr-register-names* r))

(defun sparc-fpr (r)
  (svref sparc::*fpr-register-names* r))

(defun xsimm13 (val)
  (setq val (logand val (1- (ash 1 13))))
  (if (logbitp 12 val)
    (- val (ash 1 13))
    val))

(defun sparc-disassemble-op2 (val)
  (if (logbitp 13 val)
    (xsimm13 val)
    (sparc-gpr (ldb (byte 5 0) val))))

(defun sparc-disassemble-address (val)
  (let* ((low (logand val (1- (ash 1 14))))
	 (rs1 (sparc-gpr (ldb (byte 5 14) val))))
    (if (= low 0)
      `(,rs1)
      `(,rs1 ,(sparc-disassemble-op2 val)))))
    

; To "unmacroexpand" something is to undo the effects of
; some sort of macroexpansion, returning some presumably
; more meaningful equivalent form. I'm not sure how meaningful
; this is on the sparc.

;

(defparameter *sparc-unmacroexpanders* (make-hash-table :test #'equalp))

(defun sparc-unmacroexpand-function (name)
  (let* ((pname (string name))
         (opnum (gethash pname sparc::*sparc-opcode-numbers*)))
    (unless opnum (error "Unknown sparc opcode name ~s." name))
    (values (gethash pname *sparc-unmacroexpanders*))))

(defun (setf sparc-unmacroexpand-function) (def name)
  (let* ((pname (string name))
         (opnum (gethash pname sparc::*sparc-opcode-numbers*)))
    (unless opnum (error "Unknown sparc opcode name ~s." name))
    (setf (gethash pname *sparc-unmacroexpanders*) def)))

(defmacro def-sparc-unmacroexpand (name insn-var lambda-list &body body)
  `(setf (sparc-unmacroexpand-function ',name)
         #'(lambda (,insn-var)
             (destructuring-bind ,lambda-list (lap-instruction-parsed-operands ,insn-var)
               ,@body))))


(defun sparc-unmacroexpand (insn)
  (let* ((expander (sparc-unmacroexpand-function (arch::opcode-name (lap-instruction-opcode insn))))
         (expansion (if expander (funcall expander insn))))
    (when expansion
      (setf (lap-instruction-opcode insn) (car expansion)
            (lap-instruction-parsed-operands insn) (cdr expansion))
      expansion)))


(defun find-sparc-opcode (op)
  (let* ((v sparc::*sparc-opcodes*)
	 (high (ldb (byte 16 16) op))
	 (low (ldb (byte 16 0) op)))
    (dotimes (i (length v))
      (let* ((code (svref sparc::*sparc-opcodes* i)))
	(when (and (= (logand (arch::opcode-mask-high code) high)
		      (arch::opcode-op-high code))
		   (= (logand (arch::opcode-mask-low code) low)
		      (arch::opcode-op-low code)))
	  (return code))))))

(defun sparc-disasm-1 (i pc header)
  (let* ((opcode (find-sparc-opcode i)))
    (if (null opcode)
      (error "Unknown SPARC instruction : #x~8,'0x" i)    ; should handle somehow
      (let* ((vals ()))
        (dolist (operand (arch::opcode-operands opcode))
	  (let* ((extract-fn (arch::operand-extract-function operand)))
	    (push (if extract-fn
                      (funcall extract-fn i)
                      (sparc::extract-default operand i))
		  vals)))
        (let* ((insn (%make-lap-instruction opcode)))
          (setf (lap-instruction-parsed-operands insn)
                (nreverse vals))
          (setf (lap-instruction-address insn)
                pc)
          (append-dll-node insn header))))))
                

(defvar *disassembled-sparc-instructions* ())
(defvar *disassembled-sparc-labels* ())



(defun sparc-label-at-address (address)
  (dolist (l *disassembled-sparc-labels* 
             (let* ((label (%make-lap-label (intern (format nil "L~d" address)))))
               (setf (lap-label-address label) address)
               (push label *disassembled-sparc-labels*)
               label))
    (when (= address (lap-label-address l))
      (return l))))

(defun insert-sparc-label (l instructions)
  (let* ((labaddr (lap-label-address l)))
   (do-dll-nodes (insn instructions (append-dll-node l instructions))
     (when (>= (lap-instruction-address insn) labaddr)
       (return (insert-dll-node-after l (lap-instruction-pred insn)))))))



(defun sparc-analyze-operands (instructions constants)

  (let* ((pc 0))
    (declare (fixnum pc))    
    (do-dll-nodes (insn instructions)
      (unless (sparc-unmacroexpand insn)
        (let* ((opcode (lap-instruction-opcode insn))
               (opvalues (lap-instruction-parsed-operands insn)))
          (do* ((operands (arch::opcode-operands opcode) (cdr operands))
                (operand (car operands) (car operands))
                (header (cons nil opvalues))
                (tail header))
               ((null operands) (setf (lap-instruction-parsed-operands insn) (cdr header)))
            (declare (dynamic-extent header))
            (let* ((flags (arch::operand-flags operand))
                   (val (cadr tail)))
              (declare (fixnum flags))
	      (if (eq (arch::operand-index operand) :subprim)
		  (let* ((info (find val *subprims* :key #'subprimitive-info-offset)))
		    (when info (setf (cadr tail) (subprimitive-info-name info))))
		  (cond ((logbitp sparc::sparc-operand-relative flags)
			 (let* ((label (sparc-label-at-address (+ pc val))))
			   (setf (cadr tail) (lap-label-name label))))
			((logbitp sparc::sparc-operand-fpr flags)
			 (setf (cadr tail) (sparc-fpr val)))
			((logbitp sparc::sparc-operand-gpr flags)
			 (setf (cadr tail) (sparc-gpr val))
			 (when (eq val sparc::%fn)
			   (let* ((disp (car tail)))
			     (when (and disp (typep disp 'fixnum))
			       (let* ((unscaled (+ (- sparc::misc-data-offset) disp)))
				 (unless (logtest 3 unscaled)
				   (let* ((idx (ash unscaled -2)))
				     (if (< idx (uvsize constants))
					 (rplaca tail (list 'quote (uvref constants idx)))))))))))
			((logbitp sparc::sparc-operand-address flags)
			 (setf (cadr tail) (sparc-disassemble-address val)))
                        ((eq (arch::operand-index operand) :trap-number)
                         (setf (cadr tail) (sparc-disassemble-op2 val)))
			((logbitp sparc::sparc-operand-op2 flags)
			 (setf (cadr tail) (sparc-disassemble-op2 val)))))
	      (setq tail (cdr tail))))))
      (incf pc 4))
    (dolist (l *disassembled-sparc-labels*) (insert-sparc-label l instructions))))
      
; This returns a doubly-linked list of INSTRUCTION-ELEMENTs; the caller (disassemble, INSPECT)
; can format the contents however it wants.
(defun disassemble-sparc-function (code-vector constants-vector &optional (header (make-dll-header)))
  (let* ((*disassembled-sparc-labels* nil))
    (let* ((n (uvsize code-vector)))
      (declare (fixnum n))
      (do* ((i 0 (1+ i))
            (pc 0 (+ pc 4)))
           ((= i n))
        (declare (fixnum i))
        (let* ((opcode (uvref code-vector i)))
          (declare (integer opcode))
          (if (= opcode 0)
            (return)
            (sparc-disasm-1 opcode pc header))))
      (sparc-analyze-operands header constants-vector)))
  header)

(defun print-sparc-instruction (stream tabcount opcode parsed-operands)
  (let* ((name (if (symbolp opcode) opcode (arch::opcode-name opcode)))
	 (delay-next (if (symbolp opcode)
			 nil
			 (logbitp #.sparc::f-delayed
				  (arch::opcode-flags opcode)))))
				  
    (if (keywordp name)
      (format stream "~&~V,t(~s" tabcount name)
      (format stream "~&~V,t(~a" tabcount name))
    (dolist (op parsed-operands (format stream ")"))
      (format stream (if (and (consp op) (eq (car op) 'quote)) " ~s" " ~a") op))
    delay-next))

(defun print-sparc-instructions (stream instructions &optional for-lap)
  (let* ((tab (if for-lap 6 2))
	 (delayed nil))
    (when for-lap 
      (let* ((lap-function-name (car for-lap)))
        (format stream "~&(~S ~S ~&  (~S (~s) ~&    (~s ~s ()" 
                'nfunction lap-function-name 'lambda '&lap 'lap-function lap-function-name)))
    (do-dll-nodes (i instructions)
      (etypecase i
        (lap-label (format stream "~&~a " (lap-label-name i)))
        (lap-instruction
	 (setq delayed
         (print-sparc-instruction stream (+ tab (if delayed 2 0)) (lap-instruction-opcode i) (lap-instruction-parsed-operands i))))))
    (when for-lap (format stream ")))~&"))))


(defun sparc-Xdisassemble (fn-vector &key (for-lap nil) (stream *debug-io*))
  (print-sparc-instructions stream (sparc-function-to-dll-header fn-vector) 
                          (if for-lap (list (uvref fn-vector (- (uvsize fn-vector) 2)))))
  (values))

(defun sparc-function-to-dll-header (fn-vector)
  (let* ((codev (uvref fn-vector 0)))
    (disassemble-sparc-function codev fn-vector)))

#+sparc-target
(defun disassemble-list (thing)
  (let ((dll (sparc-function-to-dll-header (function-for-disassembly thing)))
        (address 0)
        (label-p nil)
        (res nil))
    (do-dll-nodes (i dll)
      (setq address (instruction-element-address i))
      (etypecase i
        (lap-label
         (setq label-p (lap-label-name i)))
        (lap-instruction
         (let ((opcode (lap-instruction-opcode i))
               (operands (lap-instruction-parsed-operands i)))
           (push (list* (if label-p `(label ,address) address)
                        (if (symbolp opcode) opcode (arch::opcode-name opcode))
                        operands)
                 res)
           (setq label-p nil)))))
    (nreverse res)))

#+sparc-target
(defun disasm-prin1 (thing stream)
  (if (and (consp thing) (consp (cdr thing)) (null (cddr thing)))
    (cond ((eq (%car thing) 'quote)
           (prin1 thing stream))
          ((eq (%car thing) 'function)
           (format stream "#'~S" (cadr thing)))
          ((eq (%car thing) 16)
             (format stream "#x~X" (cadr thing)))
          ((eq (%car thing) 'label)
           (let ((*print-radix* nil))
             (princ (cadr thing) stream)))
          (t (princ thing stream)))
    (princ thing stream)))

; Might want to have some other entry for, e.g., the inspector
; and to let it get its hands on the list header returned by 
; disassemble-sparc-function.  Maybe disassemble-sparc-function
; should take care of "normalizing" the code-vector ?
#+sparc-target
(defun disassemble (thing)
  (sparc-xdisassemble (require-type (function-for-disassembly thing) 'compiled-function)))

#+sparc-target
(defun function-for-disassembly (thing)
  (let* ((fun thing))
    (when (typep fun 'standard-method) (setq fun (%method-function fun)))
    (when (or (symbolp fun)
              (and (consp fun) (neq (%car fun) 'lambda)))
      (setq fun (fboundp thing))
      (when (and (symbolp thing) (not (functionp fun)))
        (setq fun (macro-function thing))))
    (if (or (typep fun 'interpreted-function)
            (typep fun 'interpreted-lexical-closure))
      (setq fun (function-lambda-expression fun))
      (if (typep fun 'compiled-lexical-closure)
        (setq fun (closure-function fun))))
    (when (lambda-expression-p fun)
      (setq fun (compile-named-function fun nil)))
    fun))

#+sparc-target
(%fhave 'df #'disassemble)

(provide "SPARC-DISASSEMBLE")
