# - Find CxxTest
# Find the CxxTest suite and declare a helper macro for creating unit tests
# and integrating them with CTest.
# For more details on CxxTest see http://cxxtest.tigris.org
#
# INPUT Variables
#
#   CXXTEST_USE_PYTHON
#       If true, the CXXTEST_ADD_TEST macro will use
#       the Python test generator instead of Perl.
#
# OUTPUT Variables
#
#   CXXTEST_FOUND
#       True if the CxxTest framework was found
#   CXXTEST_INCLUDE_DIR
#       Where to find the CxxTest include directory
#   CXXTEST_PERL_TESTGEN_EXECUTABLE
#       The perl-based test generator.
#   CXXTEST_PYTHON_TESTGEN_EXECUTABLE
#       The python-based test generator.
#
# MACROS for optional use by CMake users:
#
#    CXXTEST_ADD_TEST(<test_name> <gen_source_file> <input_files_to_testgen...>)
#       Creates a CxxTest runner and adds it to the CTest testing suite
#       Parameters:
#           test_name               The name of the test
#           gen_source_file         The generated source filename to be generated by CxxTest
#           input_files_to_testgen  The list of header files containing the
#                                   CxxTest::TestSuite's to be included in this runner
#
#       #==============
#       Example Usage:
#
#           find_package(CxxTest)
#           if(CXXTEST_FOUND)
#               include_directories(${CXXTEST_INCLUDE_DIR})
#               enable_testing()
#
#               CXXTEST_ADD_TEST(unittest_foo foo_test.cc
#                                 ${CMAKE_CURRENT_SOURCE_DIR}/foo_test.h other.cpp)
#               target_link_libraries(unittest_foo foo) # as needed
#           endif()
#
#              This will (if CxxTest is found):
#              1. Invoke the testgen executable to autogenerate foo_test.cc in the
#                 binary tree from "foo_test.h" in the current source directory.
#              2. Create an executable and test called unittest_foo.
#              3. Link additionally against other.cpp
#
#      #=============
#      Example foo_test.h:
#
#          #include <cxxtest/TestSuite.h>
#
#          class MyTestSuite : public CxxTest::TestSuite
#          {
#          public:
#             void testAddition( void )
#             {
#                TS_ASSERT( 1 + 1 > 1 );
#                TS_ASSERT_EQUALS( 1 + 1, 2 );
#             }
#          };
#
#
# Version 1.2 (3/2/08)
#     Included patch from Tyler Roscoe to have the perl & python binaries
#     detected based on CXXTEST_INCLUDE_DIR
# Version 1.1 (2/9/08)
#     Clarified example to illustrate need to call target_link_libraries()
#     Changed commands to lowercase
#     Added licensing info
# Version 1.0 (1/8/08)
#     Fixed CXXTEST_INCLUDE_DIRS so it will work properly
#     Eliminated superfluous CXXTEST_FOUND assignment
#     Cleaned up and added more documentation
#
# FindCxxTest.cmake
# Copyright (c) 2008-2009
#     Philip Lowman <philip@yhbt.com>
#
#  Redistribution AND use is allowed according to the terms of the New
#  BSD license.
#  For details see the accompanying COPYING-CMAKE-SCRIPTS file.

#=============================================================
# CXXTEST_ADD_TEST (public macro)
#=============================================================
macro( CXXTEST_ADD_TEST _cxxtest_testname _cxxtest_outfname _cxxtest_testsuite )
    set(_cxxtest_real_outfname ${CMAKE_CURRENT_BINARY_DIR}/${_cxxtest_outfname})
    if(CXXTEST_USE_PYTHON)
        set(_cxxtest_executable ${CXXTEST_PYTHON_TESTGEN_EXECUTABLE})
    else()
        set(_cxxtest_executable ${CXXTEST_PERL_TESTGEN_EXECUTABLE})
    endif()

    add_custom_command(
        OUTPUT  ${_cxxtest_real_outfname}
        DEPENDS ${_cxxtest_testsuite} ${ARGN}
        COMMAND ${_cxxtest_executable}
        --error-printer -o ${_cxxtest_real_outfname} ${_cxxtest_testsuite}
    )

    set_source_files_properties(${_cxxtest_real_outfname} PROPERTIES GENERATED true)
    add_executable(${_cxxtest_testname} ${_cxxtest_real_outfname} ${ARGN})

    if(CMAKE_RUNTIME_OUTPUT_DIRECTORY)
        add_test(${_cxxtest_testname} ${CMAKE_RUNTIME_OUTPUT_DIRECTORY}/${_cxxtest_testname})
    elseif(EXECUTABLE_OUTPUT_PATH)
        add_test(${_cxxtest_testname} ${EXECUTABLE_OUTPUT_PATH}/${_cxxtest_testname})
    else()
        add_test(${_cxxtest_testname} ${CMAKE_CURRENT_BINARY_DIR}/${_cxxtest_testname})
    endif()

endmacro(CXXTEST_ADD_TEST)

#=============================================================
# CXXTEST_ADD_TEST_FROM_LIST (public macro)
#=============================================================
#
# Usage:
#   CXXTEST_ADD_TESTS_FROM_LIST( A, B, ... )
#
#   1. A is a list of cpp and/or h files where the class name is stripped of
#   2. then out of the class name the test targets and testsuite names are generated
#   3. finally they are linked with all libs present in parameter B
#   4. If there are more arguments (ARGN) then they are exclude from A before 1-3 starts)
#
FUNCTION( CXXTEST_ADD_TESTS_FROM_LIST _SourceList _TestLibs )
    # remove unwanted tests
    IF( ${ARGC} GREATER 2 )
        FOREACH( fname ${ARGN} )
            LIST( REMOVE_ITEM _SourceList ${CMAKE_CURRENT_SOURCE_DIR}/${fname} )
        ENDFOREACH( fname )
    ENDIF()

    # extract class names from source files
    FOREACH( _File ${_SourceList} )
      STRING( REGEX REPLACE "^.*/" "" _StrippedPath "${_File}" )
      STRING( REGEX REPLACE "\\..*$" "" _StrippedExtension "${_StrippedPath}" )
      STRING( REGEX MATCH ".*\\.h" _IsHeader "${_File}" ) #Match Header Files
      IF( NOT _IsHeader ) # Do not generate test for header files
        LIST( APPEND _TestList ${_StrippedExtension} ) 
      ENDIF()
    ENDFOREACH( _File )

    # generate for each class a unit test if there is a testsiute for it
    FOREACH( _ClassName ${_TestList} )
        SET( _TestName ${_ClassName}_test )
        SET( _TestTarget unittest_${_ClassName} )
        SET( _TestSuitePath "${CMAKE_CURRENT_SOURCE_DIR}/test/${_TestName}.h" )

        # check if testsuite is present and generate code if true
        IF( EXISTS ${_TestSuitePath} )
            CXXTEST_ADD_TEST( ${_TestTarget}
                ${_TestName}.cc
                ${_TestSuitePath}
                )

            TARGET_LINK_LIBRARIES( ${_TestTarget} ${_TestLibs} )
        ELSE()
            MESSAGE( STATUS "WARNING: Skipping ${_ClassName}, no unit test available." )
        ENDIF()
    ENDFOREACH( _ClassName )
ENDFUNCTION( CXXTEST_ADD_TESTS_FROM_LIST )

#=============================================================
# main()
#=============================================================

find_path(CXXTEST_INCLUDE_DIR cxxtest/TestSuite.h)
find_program(CXXTEST_PERL_TESTGEN_EXECUTABLE cxxtestgen.pl
    PATHS ${CXXTEST_INCLUDE_DIR})
find_program(CXXTEST_PYTHON_TESTGEN_EXECUTABLE cxxtestgen.py
    PATHS ${CXXTEST_INCLUDE_DIR})

include(FindPackageHandleStandardArgs)
FIND_PACKAGE_HANDLE_STANDARD_ARGS(CxxTest DEFAULT_MSG CXXTEST_INCLUDE_DIR)

set(CXXTEST_INCLUDE_DIRS ${CXXTEST_INCLUDE_DIR})
