/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
#include <glib.h>
#include <string>
#include <fstream>
#include "everything-cpp-stubs.hh"
#include "constants.hh"

using namespace std;

void testConst() {
	std::cout << "Testing constants...\n";
	g_assert(test::CONST_SHORT==0x1234);
	g_assert(test::CONST_LONG==0x12345678);
	g_assert(test::CONST_LONGLONG==0x12345678);
	g_assert(strcmp(test::CONST_STRING,"ConstString")==0);

	// I can never get constant floats to compare properly. These print out as the same
	// number, but fail the assert
	cout << test::CONST_FLOAT << "==" << 1234.56 << endl;
	//g_assert(test::CONST_FLOAT==1234.56);
	g_assert(test::CONST_DOUBLE==1234.5678);
	g_assert(test::FAVORITE_COLOUR==test::red);
	
}


void testFixedLengthArray(test::TestFactory_var factory) {
	std::cout << "Testing Fixed length element array...\n";
	test::BasicServer_var target = factory->getBasicServer();

	test::LongArray_var inArg;
	test::LongArray_var inoutArg;
	test::LongArray_var outArg;	
	test::LongArray_var retn;

	test::LongArray_copy(inArg,constants::SEQ_LONG_IN);
	test::LongArray_copy(inoutArg,constants::SEQ_LONG_INOUT_IN);
	
	retn = target->opLongArray(inArg,inoutArg,outArg);

	for(int i=0;i<constants::SEQLEN;i++)
		g_assert(inoutArg[i]==constants::SEQ_LONG_INOUT_OUT[i]);
	for(int i=0;i<constants::SEQLEN;i++)
		g_assert(outArg[i]==constants::SEQ_LONG_OUT[i]);
	for(int i=0;i<constants::SEQLEN;i++)
		g_assert(retn[i]==constants::SEQ_LONG_RETN[i]);

}

void testVariableLengthArray(test::TestFactory_var factory) {
	std::cout << "Testing Variable length element array...\n";

	test::StrArray inArg;
	test::StrArray inoutArg;
	test::StrArray_var outArg;	
	test::StrArray_var retn;
	
	test::BasicServer_var target = factory->getBasicServer();
	
	// CORBA 2.3 requires that strings in arrays are initialised to ""
	g_assert(strcmp(inArg[0],"")==0);

	// check initialisation of multidim array
	test::StrArrayMultiDimensional_slice *foo = test::StrArrayMultiDimensional_alloc();
	g_assert(strcmp(foo[0][0][0],"")==0);
	g_assert(strcmp(foo[1][2][4],"")==0);
	test::StrArrayMultiDimensional_free(foo);
	// check initialisation of nested struct
	test::VariableLengthStructArray_slice *vlsa = test::VariableLengthStructArray_alloc();
	g_assert(strcmp(vlsa[1].a,"")==0);
	test::VariableLengthStructArray_free(vlsa);

	test::StrArray_copy(inArg,reinterpret_cast<test::StrArray_slice const *>(constants::SEQ_STRING_IN));
	test::StrArray_copy(inoutArg,reinterpret_cast<test::StrArray_slice const *>(constants::SEQ_STRING_INOUT_IN));

	
	retn = target->opStrArray(inArg,inoutArg,outArg);
	
	for(int i=0;i<constants::SEQLEN;i++)
		g_assert(strcmp(inoutArg[i],constants::SEQ_STRING_INOUT_OUT[i])==0);	
	for(int i=0;i<constants::SEQLEN;i++)
		g_assert(strcmp(outArg[i],constants::SEQ_STRING_OUT[i])==0);
	for(int i=0;i<constants::SEQLEN;i++)
		g_assert(strcmp(retn[i],constants::SEQ_STRING_RETN[i])==0);
	
}


void testAttribute(test::TestFactory_var factory) {
	std::cout << "Testing attribute...\n";
	test::BasicServer_var target = factory->getBasicServer();

	target->foo(constants::STRING_IN);
	CORBA::String_var sv_retval = target->foo();
	g_assert(strcmp(sv_retval,constants::STRING_RETN)==0);
	g_assert(target->bah() == constants::LONG_RETN);
	
}

void testString(test::TestFactory_var factory) {
	std::cout << "Testing string...\n";
	test::BasicServer_var target = factory->getBasicServer();

	CORBA::String_var sv_in(constants::STRING_IN);
	CORBA::String_var sv_inout(constants::STRING_INOUT_IN);
	CORBA::String_var sv_out;

	CORBA::String_var sv_retval = target->opString(sv_in,
												   sv_inout.inout(),
												   sv_out);
	
	g_assert(strcmp(sv_in,constants::STRING_IN)==0);
	g_assert(strcmp(sv_inout,constants::STRING_INOUT_OUT)==0);
	g_assert(strcmp(sv_out,constants::STRING_OUT)==0);
	g_assert(strcmp(sv_retval,constants::STRING_RETN)==0);	
}

void testLong(test::TestFactory_var factory) {
	std::cout << "Testing long...\n";
	test::BasicServer_var target = factory->getBasicServer();
	CORBA::Long inArg = constants::LONG_IN;
	CORBA::Long inoutArg = constants::LONG_INOUT_IN;
	CORBA::Long outArg;
	CORBA::Long retn;
	retn = target->opLong(inArg,inoutArg,outArg);
	g_assert(inArg == constants::LONG_IN);
	g_assert(inoutArg == constants::LONG_INOUT_OUT);
	g_assert(outArg == constants::LONG_OUT);
	g_assert(retn == constants::LONG_RETN);
}

void testException(test::TestFactory_var factory){
	
	std::cout << "Testing exception...\n";
	test::BasicServer_var target = factory->getBasicServer();

	bool caught = false;
	try {
		target->opException();
	}
	catch(test::TestException& ex) {
		caught = true;
		g_assert(strcmp(ex.reason,constants::STRING_IN) == 0);
		g_assert(ex.number == constants::LONG_IN);
		g_assert(ex.aseq.length() == 1);
		g_assert(ex.aseq[0] == constants::LONG_IN);
		testLong(ex.factory);
	}
	g_assert(caught == true); // assert that the exception was caught
}

void testEnum(test::TestFactory_var factory) {
	std::cout << "Testing enum...\n";
	test::BasicServer_var target = factory->getBasicServer();
	test::AnEnum inArg = test::ENUM_IN;
	test::AnEnum inoutArg = test::ENUM_INOUT_IN;
	test::AnEnum outArg;
	test::AnEnum retn;
	retn = target->opEnum(inArg,inoutArg,outArg);
	g_assert(inArg == test::ENUM_IN);
	g_assert(inoutArg == test::ENUM_INOUT_OUT);
	g_assert(outArg == test::ENUM_OUT);
	g_assert(retn == test::ENUM_RETN);
}


void
testFixedLengthStruct(test::TestFactory_var factory){
	std::cout << "Testing fixed length structs...\n";
	test::StructServer_var target = factory->getStructServer();

	test::FixedLengthStruct inArg,inoutArg,outArg;
	inArg.a = constants::SHORT_IN;
	inoutArg.a = constants::SHORT_INOUT_IN;

	test::FixedLengthStruct retn = target->opFixed(inArg,inoutArg,outArg);

	g_assert(inArg.a == constants::SHORT_IN);
	g_assert(inoutArg.a == constants::SHORT_INOUT_OUT);
	g_assert(outArg.a == constants::SHORT_OUT);
	g_assert(retn.a == constants::SHORT_RETN);
}

void
testVariableLengthStruct(test::TestFactory_var factory){
	std::cout << "Testing variable length structs...\n";
	test::StructServer_var target = factory->getStructServer();

	test::VariableLengthStruct inArg, inoutArg;
	test::VariableLengthStruct_var outArg, retn;


	// CORBA 2.3 requires that strings in structs are initialised to ""
	g_assert(strcmp(inArg.a,"")==0); 

	inArg.a = CORBA::string_dup(constants::STRING_IN);
	inoutArg.a = CORBA::string_dup(constants::STRING_INOUT_IN);
	
	retn = target->opVariable(inArg,inoutArg,outArg);
	
	g_assert(strcmp(inArg.a,constants::STRING_IN)==0);
	g_assert(strcmp(inoutArg.a,constants::STRING_INOUT_OUT)==0);
	g_assert(strcmp(outArg->a,constants::STRING_OUT)==0);
	g_assert(strcmp(retn->a,constants::STRING_RETN)==0);	
}


void
testCompoundStruct(test::TestFactory_var factory){
	std::cout << "Testing compound structs...\n";
	test::StructServer_var target = factory->getStructServer();

	test::CompoundStruct inArg, inoutArg;
	test::CompoundStruct_var outArg, retn;
	inArg.a.a = CORBA::string_dup(constants::STRING_IN);
	inoutArg.a.a = CORBA::string_dup(constants::STRING_INOUT_IN);
	
	retn = target->opCompound(inArg,inoutArg,outArg);
	
	g_assert(strcmp(inArg.a.a,constants::STRING_IN)==0);
	g_assert(strcmp(inoutArg.a.a,constants::STRING_INOUT_OUT)==0);
	g_assert(strcmp(outArg->a.a,constants::STRING_OUT)==0);
	g_assert(strcmp(retn->a.a,constants::STRING_RETN)==0);	
}

void
testUnboundedSequence(test::TestFactory_var factory) {
	std::cout << "Testing unbounded sequences...\n";
	test::SequenceServer_var target = factory->getSequenceServer();
	
	test::StrSeq inSeq, inoutSeq;
	inSeq.length(2);
	// CORBA 2.3 requires that strings in sequences are initialised to ""
	g_assert(strcmp(inSeq[0],"")==0); 	
	inSeq.length(3);
	
	inoutSeq.length(2);

	test::StrSeq_var retnSeq;
	test::StrSeq_var outSeq;
	
	for (guint i=0;i<inSeq.length();i++)
		inSeq[i] = constants::SEQ_STRING_IN[i];
	for (guint i=0;i<inoutSeq.length();i++)
		inoutSeq[i] = constants::SEQ_STRING_INOUT_IN[i];
	
	retnSeq = target->opStrSeq(inSeq,inoutSeq,outSeq);
	
	for (guint i=0;i<inoutSeq.length();i++){
		g_assert(strcmp(inoutSeq[i],constants::SEQ_STRING_INOUT_OUT[i]) == 0);
	}

	for (guint i=0;i<outSeq->length();i++){
		g_assert(strcmp(outSeq[i],constants::SEQ_STRING_OUT[i]) == 0);
	}

	for (guint i=0;i<retnSeq->length();i++){
		g_assert(strcmp(retnSeq[i],constants::SEQ_STRING_RETN[i]) == 0);
	}	
}

void
testBoundedSequence(test::TestFactory_var factory) {
	std::cout << "Testing bounded sequences...\n";
	test::SequenceServer_var target = factory->getSequenceServer();
	
	test::BoundedStructSeq inSeq, inoutSeq;
	inSeq.length(2);
	
	// CORBA 2.3 requires that strings in structs are initialised to ""
	g_assert(strcmp(inSeq[0].a.a,"")==0); 
	
	inoutSeq.length(2);
	
	test::BoundedStructSeq_var retnSeq;
	test::BoundedStructSeq_var outSeq;
	
	for (guint i=0;i<inSeq.length();i++)
		inSeq[i].a.a = constants::SEQ_STRING_IN[i];
	
	for (guint i=0;i<inoutSeq.length();i++)
		inoutSeq[i].a.a = constants::SEQ_STRING_INOUT_IN[i];
	
	
	retnSeq = target->opBoundedStructSeq(inSeq,inoutSeq,outSeq);
	
	for (guint i=0;i<inoutSeq.length();i++){
		g_assert(strcmp(inoutSeq[i].a.a,constants::SEQ_STRING_INOUT_OUT[i]) == 0);
	}

	for (guint i=0;i<outSeq->length();i++){
		g_assert(strcmp(outSeq[i].a.a,constants::SEQ_STRING_OUT[i]) == 0);
	}

	for (guint i=0;i<retnSeq->length();i++){
		g_assert(strcmp(retnSeq[i].a.a,constants::SEQ_STRING_RETN[i]) == 0);
	}	
}

void
testPolymorphism(test::TestFactory_var factory) {
	std::cout << "Testing Multiple inheritance polymorphism...\n";
	test::BaseServer_var base = factory->getBaseServer();
	test::BaseServer_var derived = factory->getDerivedServer();

	test::B2_var b2 = factory->getDerivedServerAsB2();

	
	CORBA::String_var retval;

	g_assert(base->opPolymorphic() == constants::LONG_BASE);
	retval = base->attribPolymorphic();
	g_assert(strcmp(retval,constants::STRING_BASE)==0);
	base->attribPolymorphic(constants::STRING_BASE);

	g_assert(derived->opPolymorphic() == constants::LONG_DERIVED);
	retval = derived->attribPolymorphic();
	g_assert(strcmp(retval,constants::STRING_DERIVED)==0);
	derived->attribPolymorphic(constants::STRING_DERIVED);

	
	g_assert(b2->opPolymorphic() == constants::LONG_DERIVED);
	retval = b2->attribPolymorphic();
	g_assert(strcmp(retval,constants::STRING_DERIVED)==0);
	derived->attribPolymorphic(constants::STRING_DERIVED);

}

void
testTransientObj(test::TestFactory_var factory) {
	std::cout << "Testing transient object destruction...\n";
	test::TransientObj_var t = factory->createTransientObj();
	t->remove();

	// a second call to the object should result in a
	// CORBA::OBJECT_NOT_EXIST exception
	bool caught=false;
	try {
		t->remove();
	} catch (CORBA::OBJECT_NOT_EXIST) {
		caught = true;
	}
	g_assert(caught);
}

void
testFixedLengthUnion(test::TestFactory_var factory){
	std::cout << "Testing fixed length unions...\n";
	test::UnionServer_var target = factory->getUnionServer();

	test::FixedLengthUnion inArg,inoutArg,outArg;
	inArg.x(constants::LONG_IN);
	inoutArg.y('t');

	test::FixedLengthUnion retn = target->opFixed(inArg,inoutArg,outArg);

	g_assert(inArg._d() == 'a');
	g_assert(inArg.x() == constants::LONG_IN);
	g_assert(inoutArg._d() == 'c');
	g_assert(inoutArg.z() == true);
	g_assert(outArg._d() == 'a');
	g_assert(outArg.x() == constants::LONG_OUT);
	g_assert(retn._d() == 'd');
	g_assert(retn.y() == false);
}

void
testVariableLengthUnion(test::TestFactory_var factory){
	std::cout << "Testing variable length unions...\n";
	test::UnionServer_var target = factory->getUnionServer();

	test::VariableLengthUnion inArg,inoutArg;
	test::VariableLengthUnion_var outArg,retn;

	test::VariableLengthStruct vstruct;
	vstruct.a = "test string1";
	inArg.a(vstruct); // deep copy
	g_assert(inArg._d() == 5);
	g_assert(strcmp(inArg.a().a,"test string1") == 0);

    
	inArg.a().a="test string2";

	g_assert(strcmp(vstruct.a,"test string1") == 0);  // confirm deep copy, not shallow

	inoutArg = inArg;  // operator= deep copies the union
	g_assert(inoutArg._d() == 5);
	g_assert(strcmp(inoutArg.a().a,"test string2") == 0);

	inArg.x(constants::LONG_IN);  // deep frees the struct
	inoutArg.y(CORBA::string_dup(constants::STRING_INOUT_IN)); // deep frees the struct

	retn = target->opVariable(inArg,inoutArg,outArg);

	g_assert(inArg._d() == 1);
	g_assert(inArg.x() == constants::LONG_IN);
	g_assert(inoutArg._d() == 3);
	g_assert(inoutArg.z() == true);
	g_assert(outArg->_d() == 1);
	g_assert(outArg->x() == constants::LONG_OUT);
	g_assert(retn->_d() == 4);
	g_assert(retn->y() == false);

	test::BooleanUnion *bu = new test::BooleanUnion;
	delete bu;
}


void
testAnyLong(test::TestFactory_var factory){
	std::cout << "Testing any with longs...\n";
	test::AnyServer_var target = factory->getAnyServer();

	CORBA::Any a;
	a <<= constants::LONG_IN;
	CORBA::Long l;
	CORBA::ULong ul;
	assert(!(a >>= ul));   // should return false, since value is a long
	assert(a >>= l);

	CORBA::Any inArg, inoutArg;

	inArg <<= constants::LONG_IN;
	inoutArg <<= constants::LONG_INOUT_IN;

	CORBA::Any *outArg;
	CORBA::Any *retval = target->opAnyLong(inArg,inoutArg,outArg);

	assert(inoutArg >>= l);
	assert(l == constants::LONG_INOUT_OUT);

	assert((*retval) >>= l);
	assert(l == constants::LONG_RETN);

	assert((*outArg) >>= l);
	assert(l == constants::LONG_OUT);
	
	delete outArg;
	delete retval;
}


void
testAnyString(test::TestFactory_var factory){
	std::cout << "Testing any with strings...\n";
	test::AnyServer_var target = factory->getAnyServer();

	CORBA::Any inArg, inoutArg;

	inArg <<= constants::STRING_IN;
	inoutArg <<= constants::STRING_INOUT_IN;

	CORBA::Any *outArg;
	CORBA::Any *retval = target->opAnyString(inArg,inoutArg,outArg);

	const char *s;
	assert(inoutArg >>= s);
	assert(strcmp(s,constants::STRING_INOUT_OUT)==0);

	assert((*retval) >>= s);
	assert(strcmp(s,constants::STRING_RETN)==0);

	assert((*outArg) >>= s);
	assert(strcmp(s,constants::STRING_OUT)==0);
	
	delete outArg;
	delete retval;
}

void
testAnyStruct(test::TestFactory_var factory){
	std::cout << "Testing any with structs...\n";
	test::AnyServer_var target = factory->getAnyServer();
	
	test::VariableLengthStruct inArgStruct, inoutArgStruct;
	CORBA::Any inArg, inoutArg;

	inArgStruct.a = constants::STRING_IN;
	inoutArgStruct.a = constants::STRING_INOUT_IN;
	
	inArg <<= inArgStruct;
	inoutArg <<= inoutArgStruct;

	CORBA::Any *outArg;
	CORBA::Any *retval = target->opAnyStruct(inArg,inoutArg,outArg);

	const test::VariableLengthStruct *s;
	g_assert(inoutArg >>= s);
	g_assert(strcmp(s->a,constants::STRING_INOUT_OUT) == 0);

	g_assert((*retval) >>= s);
	g_assert(strcmp(s->a,constants::STRING_RETN) == 0);

	g_assert((*outArg) >>= s);
	g_assert(strcmp(s->a,constants::STRING_OUT) == 0);
	
	delete outArg;
	delete retval;
}



void testTypeCode(test::TestFactory_var factory) {
	std::cout << "Testing TypeCodes...\n";
	test::AnyServer_var target = factory->getAnyServer();

	CORBA::TypeCode_var inoutArg, outArg;
	inoutArg=CORBA::TypeCode::_duplicate(test::_tc_AnyServer);
	CORBA::TypeCode_var retval = target->opTypeCode(test::_tc_ArrayUnion,inoutArg,outArg);
	g_assert(inoutArg->equal(test::_tc_TestException));
	g_assert(outArg->equal(test::_tc_AnEnum));
	g_assert(retval->equal(test::_tc_VariableLengthStruct));
}

int
main(int argc, char *argv[]) {
	CORBA::ORB_var orb = CORBA::ORB_init(argc, argv, "orbit-local-orb");
	ifstream in("iorfile");
	string ior;
	in >> ior;
	CORBA::Object_var obj = orb->string_to_object(ior.c_str());
	test::TestFactory_var ptr = test::TestFactory::_narrow(obj);

	try {
        for(int i=0;i<20;i++){
			testConst();
			testString(ptr);             
			testLong(ptr);
			testAttribute(ptr);
			testException(ptr);
			testEnum(ptr);
			testFixedLengthStruct(ptr);
			testVariableLengthStruct(ptr);
			testCompoundStruct(ptr);
			testUnboundedSequence(ptr);
			testBoundedSequence(ptr);
			testPolymorphism(ptr);
            testFixedLengthArray(ptr);
			testVariableLengthArray(ptr);
            //testTransientObj(ptr);
			testFixedLengthUnion(ptr);
            testVariableLengthUnion(ptr);
			testAnyLong(ptr);
			testAnyString(ptr);
			testAnyStruct(ptr);
			testTypeCode(ptr);
        }
	} catch (CORBA::SystemException& ex){
		cout << "Uncaught corba exception: " << ex._orbitcpp_get_repoid() << endl;
		return 1;
	}
	catch (...){
		cout << "Uncaught c++ exception!\n";
		return 1;
	}
	return 0;  // pass

}
