
/******************************************************************************
**
**  Copyright (C) 2006 Brian Wotring.
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, view a current copy of the license
**  file here:
**
**      http://www.hostintegrity.com/osiris/LICENSE
**
******************************************************************************/

/*****************************************************************************
**
**  File:    rootpriv.c
**  Date:    February 17, 2002
**
**  Author:  Brian Wotring
**  Purpose: scanning daemon main implementation
**
******************************************************************************/

#include "libosiris.h"
#include "libfileapi.h"
#include "common.h"

#include "logging.h"
#include "rootpriv.h"

#ifdef USE_PRIVSEP

/* this is defined in regex, and sometimes conflicts with stuff */
/* in sys/param.h so we whack it here.                          */

#ifdef RE_DUP_MAX
#undef RE_DUP_MAX
#endif

#include <sys/uio.h>
#include <sys/param.h>
#include <sys/wait.h>

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#endif

#include <signal.h>

#ifndef HAVE_DIRFD
#ifndef dirfd
#define dirfd(dirp)   ((dirp)->dd_fd)
#endif
#endif

/* some systems don't have these macros, use default behavior. */

#ifndef CMSG_LEN
#define CMSG_LEN(size)  (sizeof(struct cmsghdr) + (size))
#endif
#ifndef CMSG_SPACE
#define CMSG_SPACE(size) (sizeof(struct cmsghdr) + (size))
#endif

/* used to detect root op requests. */

static fd_set root_read_set;

static volatile osi_atomic_t received_sigchld = 0;
static volatile osi_atomic_t received_sigterm = 0;
static volatile osi_atomic_t received_sigint = 0;

extern pid_t child_pid;
extern int rootpriv_pipe[2];      /* socketpair used for rootpriv comm.  */

#ifndef WIN32
extern char pid_file[MAX_PATH_LENGTH];
#endif

#define INCOMING_ROOT_REQUEST() ( FD_ISSET( rootpriv_pipe[1], &root_read_set ) )


/* THE FOLLOWING FUNCTIONS RUN AS ROOT!! */

void handle_root_requests()
{
    setup_signals();

    for(;;)
    {
        handle_signals();
        wait_for_root_request();

        if( INCOMING_ROOT_REQUEST() )
        {
            process_root_request();
        }
    }
}

void wait_for_root_request()
{
    int result;

wait_start:

    FD_ZERO( &root_read_set );

    /* always add the server socket to the set. */

    FD_SET( rootpriv_pipe[1], &root_read_set );

    result = select( ( rootpriv_pipe[1] + 1 ), &root_read_set,NULL,NULL,NULL );

    if( result < 0 )
    {
        if( errno == EINTR )
        {
            handle_signals();
        }

        goto wait_start;
    }
}

void process_root_request()
{
    char result;

    char op; 
    char path[MAX_PATH_LENGTH] = "";

    /* read op command. */

    if( ( osi_read( rootpriv_pipe[1], &op, sizeof( op ) ) ) <= 0 )
    {
        log_error( "error reading root privileged operation." );
        return;
    }

    /* now process the request. */

    if( op == ROOT_OP_STAT )
    {
        struct stat stats;

        /* read path. */

        if( ( osi_read( rootpriv_pipe[1], path, sizeof( path ) ) ) <= 0 )
        {
            log_error( "error reading root privileged message." );
            return;
        }
        
        if( stat( path, &stats ) == 0 )
        {
            result = ROOT_OP_SUCCESS;
            osi_write( rootpriv_pipe[1], &result, sizeof( result ) );
            osi_write( rootpriv_pipe[1], (char *)&stats, sizeof( stats ) );   
        }

        else
        {
            result = ROOT_OP_FAIL;
            osi_write( rootpriv_pipe[1], &result, sizeof( result ) );
        }
    }

    else if( op == ROOT_OP_LSTAT )
    {
        struct stat stats;

        /* read path. */

        if( ( osi_read( rootpriv_pipe[1], path, sizeof( path ) ) ) <= 0 )
        {
            log_error( "error reading root privileged message." );
            return;
        }

#ifdef WIN32
        result = ROOT_OP_FAIL;
        osi_write( rootpriv_pipe[1], &result, sizeof( result ) );
#else
        if( lstat( path, &stats ) == 0 )
        {
            result = ROOT_OP_SUCCESS;
            osi_write( rootpriv_pipe[1], &result, sizeof( result ) );
            osi_write( rootpriv_pipe[1], (char *)&stats, sizeof( stats ) );
        }

        else
        {
            result = ROOT_OP_FAIL;
            osi_write( rootpriv_pipe[1], &result, sizeof( result ) );
        }
#endif
    }

    else if( op == ROOT_OP_OPENDIR )
    {
        DIR *directory;

        /* read path. */

        if( ( osi_read( rootpriv_pipe[1], path, sizeof( path ) ) ) <= 0 )
        {
            log_error( "error reading root privileged message." );
            return;
        }

        if( ( directory = opendir( path ) ) == NULL )
        {
            result = ROOT_OP_FAIL;  
            osi_write( rootpriv_pipe[1], &result, sizeof( result ) );
        }

        else
        {
            result = ROOT_OP_SUCCESS;

            osi_write( rootpriv_pipe[1], &result, sizeof( result ) );
            send_fd( rootpriv_pipe[1], (int)dirfd( directory ) );

            closedir(directory);
        }
    }

    else if( op == ROOT_OP_FOPEN )
    {
        int fd;
        FILE *file;

        /* read path. */

        if( ( osi_read( rootpriv_pipe[1], path, sizeof( path ) ) ) <= 0 )
        {
            log_error( "error reading root privileged message." );
            return;
        }
        
        file = osi_fopen( path, "r", 0 );

        if( ( file == NULL ) || ( ( fd = fileno( file ) ) < 0 ) )
        {
            result = ROOT_OP_FAIL;
            osi_write( rootpriv_pipe[1], &result, sizeof( result ) );
        }

        /* send this file descriptor to the child. */

        else
        {
            result = ROOT_OP_SUCCESS;
            osi_write( rootpriv_pipe[1], &result, sizeof( result ) );
            send_fd( rootpriv_pipe[1], fd );
            fclose( file );
        }
    }

    else if( op == ROOT_OP_PIDFILE )
    {
        setup_pidfile();

        result = ROOT_OP_SUCCESS;
        osi_write( rootpriv_pipe[1], &result, sizeof( result ) );
    }

    else
    {
        log_error( "unrecognized root privileged request." );
    }
}

void setup_signals()
{
#ifdef HAVE_SIGSET
    sigset( SIGCHLD, root_handle_sigchld );
    sigset( SIGTERM, root_handle_sigterm );
    sigset( SIGINT, root_handle_sigint );
#else
    signal( SIGCHLD, root_handle_sigchld );
    signal( SIGTERM, root_handle_sigterm );
    signal( SIGINT, root_handle_sigint );
#endif
}

void setup_pidfile()
{
    FILE *pidfile = fopen( pid_file, "wb" );

    if( pidfile != NULL )
    {
        fprintf( pidfile, "%ld\n", (long)getpid() );
        fclose( pidfile );
    }
}

void handle_signals()
{
    pid_t pid;
    int status;

    if( received_sigchld )
    {
       while( (pid = waitpid( -1, &status, WNOHANG ) ) > 0 ||
              ( ( pid < 0 ) && ( errno == EINTR ) ) )
        ;

        /* child process has died, we die. */
    
        exit(0);
    }

    if( received_sigterm || received_sigint )
    {
        unlink( pid_file);

        /* kill child daemon. */

        kill( child_pid, SIGTERM );
    }
}

void root_handle_sigchld( int signal )
{
    received_sigchld = 1;
}

void root_handle_sigterm( int signal )
{
    received_sigterm = 1;
}

void root_handle_sigint( int signal )
{
    received_sigint = 1;
}


/* THE FOLLOWING FUNCTIONS DO >NOT< RUN AS ROOT. */

void rootpriv_setup_pidfile()
{
    int result;

    char op = ROOT_OP_PIDFILE;
    char response;

    /* send root operation to root process. */

    osi_write( rootpriv_pipe[0], &op, sizeof( op ) );

    /* read response. */

    result = osi_read( rootpriv_pipe[0], &response, sizeof( response ) );
}


FILE * rootpriv_fopen( const char *path )
{
    int fd;
    int result;

    char op = ROOT_OP_FOPEN;
    char response;

    if( path == NULL )
    {
        return NULL;
    }

    /* send root operation to root process. */

    if( osi_write( rootpriv_pipe[0], &op, sizeof( op ) ) <= 0 )
    {
        return NULL;
    }

    if( osi_write( rootpriv_pipe[0], path, strlen( path ) ) <= 0 )
    {
        return NULL;
    }

    /* read response. */

    result = osi_read( rootpriv_pipe[0], &response, sizeof( response ) );

    if( ( result <= 0 ) || ( response != ROOT_OP_SUCCESS ) )
    {
        return NULL;
    }

    /* receive the file descriptor. */

    fd = receive_fd( rootpriv_pipe[0] );    

    if( fd <= 0 )
    {
        return NULL;
    }

    return fdopen( fd, "r" );
}

int rootpriv_stat( const char *path, struct stat *file_stats )
{
    int result;

    char op = ROOT_OP_STAT;
    char response;

    if( ( path == NULL ) || ( file_stats == NULL ) )
    {
        return -1;
    }

    /* send root operation to root process. */
 
    if( osi_write( rootpriv_pipe[0], &op, sizeof( op ) ) <= 0 )
    {
        return -1;
    }

    if( osi_write( rootpriv_pipe[0], path, strlen( path ) ) <= 0 )
    {
        return -1;
    }

    /* read response. */

    result = osi_read( rootpriv_pipe[0], &response, sizeof( response ) );

    if( ( result <= 0 ) || ( response != ROOT_OP_SUCCESS ) )
    {
        return -1;
    }

    result = osi_read( rootpriv_pipe[0], (char *)file_stats,
                       sizeof( struct stat ) );

    if( result <= 0 )
    {
        return -1;
    }

    return 0;
}

int rootpriv_lstat( const char *path, struct stat *file_stats )
{
    int result;

    char op = ROOT_OP_LSTAT;
    char response;

    if( ( path == NULL ) || ( file_stats == NULL ) )
    {
        return -1;
    }

    /* send root operation to root process. */

    if( osi_write( rootpriv_pipe[0], &op, sizeof( op ) ) <= 0 )
    {
        return -1;
    }

    if( osi_write( rootpriv_pipe[0], path, strlen( path ) ) <= 0 )
    {
        return -1;
    }

    /* read response. */

    result = osi_read( rootpriv_pipe[0], &response, sizeof( response ) );

    if( ( result <= 0 ) || ( response != ROOT_OP_SUCCESS ) )
    {
        return -1;
    }

    result = osi_read( rootpriv_pipe[0], (char *)file_stats,
                       sizeof( struct stat ) );

    if( result <= 0 )
    {
        return -1;
    }

    return 0;
}

osi_bool rootpriv_opendir( const char *path, OSI_DIRECTORY *dir )
{
    int result;
    int fd;

    char op = ROOT_OP_OPENDIR;
    char response;

    if( path == NULL || dir == NULL )
    {
        return FALSE;
    }

    /* send root operation to root process. */

    if( osi_write( rootpriv_pipe[0], &op, sizeof( op ) ) <= 0 )
    {
        return FALSE;
    }

    if( osi_write( rootpriv_pipe[0], path, strlen( path ) ) <= 0 )
    {
        return FALSE;
    }

    /* read response. */

    result = osi_read( rootpriv_pipe[0], &response, sizeof( response ) );

    if( ( result <= 0 ) || ( response != ROOT_OP_SUCCESS ) )
    {
        return FALSE;
    }

    fd = receive_fd( rootpriv_pipe[0] );

    if( fd <= 0 )
    {
        return FALSE;
    }

    osi_directory_set_fd( dir, fd );

    /* for compatibility with the file api, we read in the first */
    /* filename, if there is one.                                */

    return osi_get_next_file( dir );
}


/*
 * Copyright 2001 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

void send_fd( int socket, int fd )
{
#if defined(HAVE_SENDMSG) && (defined(HAVE_ACCRIGHTS_IN_MSGHDR) || defined(HAVE_CONTROL_IN_MSGHDR))
    struct msghdr msg;
    struct iovec vec;
    char ch = '\0';
    ssize_t n;
#ifndef HAVE_ACCRIGHTS_IN_MSGHDR
    char tmp[CMSG_SPACE(sizeof(int))];
    struct cmsghdr *cmsg;
#endif

    memset(&msg, 0, sizeof(msg));
#ifdef HAVE_ACCRIGHTS_IN_MSGHDR
    msg.msg_accrights = (caddr_t)&fd;
    msg.msg_accrightslen = sizeof(fd);
#else
    msg.msg_control = (caddr_t)tmp;
    msg.msg_controllen = CMSG_LEN(sizeof(int));
    cmsg = CMSG_FIRSTHDR(&msg);
    cmsg->cmsg_len = CMSG_LEN(sizeof(int));
    cmsg->cmsg_level = SOL_SOCKET;
    cmsg->cmsg_type = SCM_RIGHTS;
    *(int *)CMSG_DATA(cmsg) = fd;
#endif

    vec.iov_base = &ch;
    vec.iov_len = 1;
    msg.msg_iov = &vec;
    msg.msg_iovlen = 1;

    if ((n = sendmsg(socket, &msg, 0)) == -1)
        fprintf(stderr,"unable to send message! (%d)\n",errno );
    if (n != 1)
        fprintf( stderr,"expected different byte count!" );
#else
    fprintf(stderr,"privilege seperation not supported!\n" );
#endif
}


int receive_fd( int socket )
{
#if defined(HAVE_RECVMSG) && (defined(HAVE_ACCRIGHTS_IN_MSGHDR) || defined(HAVE_CONTROL_IN_MSGHDR))
    struct msghdr msg;
    struct iovec vec;
    ssize_t n;
    char ch;
    int fd;
#ifndef HAVE_ACCRIGHTS_IN_MSGHDR
    char tmp[CMSG_SPACE(sizeof(int))];
    struct cmsghdr *cmsg;
#endif

    memset(&msg, 0, sizeof(msg));
    vec.iov_base = &ch;    vec.iov_len = 1;
    msg.msg_iov = &vec;
    msg.msg_iovlen = 1;
#ifdef HAVE_ACCRIGHTS_IN_MSGHDR
    msg.msg_accrights = (caddr_t)&fd;
    msg.msg_accrightslen = sizeof(fd);
#else
    msg.msg_control = tmp;
    msg.msg_controllen = sizeof(tmp);
#endif
    if ((n = recvmsg(socket, &msg, 0)) == -1)
    fprintf(stderr,"error receiving message!\n" );
    if (n != 1)
    fprintf(stderr," expected different byte receive.\n" );

#ifdef HAVE_ACCRIGHTS_IN_MSGHDR
    if (msg.msg_accrightslen != sizeof(fd))
    fprintf(stderr,"NO FD!\n" );
#else
    cmsg = CMSG_FIRSTHDR(&msg);
    if (cmsg->cmsg_type != SCM_RIGHTS)
        fprintf(stderr,"got wrong receive type!\n" );
    fd = (*(int *)CMSG_DATA(cmsg));
#endif
    return fd;
#else
    fprintf(stderr,"privilege seperation not supported!\n" );
#endif
}

#if 0
static void send_fd( int socket, int fd )
{
#if defined(HAVE_SENDMSG) && (defined(HAVE_ACCRIGHTS_IN_MSGHDR) || defined(HAVE_CONTROL_IN_MSGHDR))
    struct msghdr msg;
    struct iovec iov[1];

    char unused = '\0';
#ifdef HAVE_CONTROL_IN_MSGHDR
    union
    {
        struct cmsghdr cm;
        char control[CMSG_SPACE(sizeof(int))];
    } control_un;

    struct cmsghdr *cmptr;

    msg.msg_control = control_un.control;
    msg.msg_controllen = sizeof( control_un.control );

    cmptr = CMSG_FIRSTHDR(&msg);
    cmptr->cmsg_len = CMSG_LEN( sizeof(int) );
    cmptr->cmsg_level = SOL_SOCKET;
    cmptr->cmsg_type = SCM_RIGHTS;
    *((int *) CMSG_DATA(cmptr)) = fd;
#else
    msg.msg_accrights = (caddr_t) & fd;
    msg.msg_accrightslen = sizeof(int);
#endif

    msg.msg_name = NULL;
    msg.msg_namelen = 0;

    iov[0].iov_base = &unused;
    iov[0].iov_len = 1;
    msg.msg_iov = iov;
    msg.msg_iovlen = 1;

    sendmsg( socket, &msg, 0 );

#else
    #error "privsep problem: look at the send_fd method in __FUNCTION__!"
#endif
}

static int receive_fd( int socket )
{
#if defined(HAVE_RECVMSG) && (defined(HAVE_ACCRIGHTS_IN_MSGHDR) || defined(HAVE_CONTROL_IN_MSGHDR))

    struct msghdr msg;    struct iovec iov[1];

    char unused = '\0';
    int recv_socket = 0;

#ifdef HAVE_CONTROL_IN_MSGHDR

    union    {
        struct cmsghdr cm;
        char control[CMSG_SPACE(sizeof(int))];    } control_un;

    struct cmsghdr *cmptr;

    msg.msg_control = control_un.control;
    msg.msg_controllen = sizeof( control_un.control );

#else
    int newfd;

    msg.msg_accrights = (caddr_t) & newfd;
    msg.msg_accrightslen = sizeof(int);
#endif

    msg.msg_name = NULL;
    msg.msg_namelen = 0;

    iov[0].iov_base = &unused;
    iov[0].iov_len = 1;
    msg.msg_iov = iov;
    msg.msg_iovlen = 1;

    if( ( recv_socket = recvmsg( socket, &msg, 0 ) ) <= 0 )
    {
        return -1;
    }


#ifdef HAVE_CONTROL_IN_MSGHDR 
    if( ( ( cmptr = CMSG_FIRSTHDR(&msg)) != NULL ) &&
        ( cmptr->cmsg_len == CMSG_LEN(sizeof(int)) ) )
    {
        if( cmptr->cmsg_level != SOL_SOCKET )
        {
            return -1;
        }

        if( cmptr->cmsg_type != SCM_RIGHTS )
        {
            return -1;
        }

        recv_socket = *((int *) CMSG_DATA(cmptr));
    }

    else
    {
        recv_socket = -1;
    }
#else
    if( msg.msg_accrightslen == sizeof(int) )
    {
        recv_socket = newfd;
    }

    else
    {
        recv_socket = -1;
    }
#endif

    return recv_socket;

#else
    #error "privsep problem: look at the receive_fd method in __FUNCTION__!"
#endif
}


#endif

#endif /* USE_PRIVSEP */

