///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file AtomType.h
 * \brief Contains the definition of the AtomViz::AtomType class.
 */

#ifndef __ATOM_TYPE_H
#define __ATOM_TYPE_H

#include <atomviz/AtomViz.h>

#include <core/reference/RefTarget.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/gui/properties/PropertiesEditor.h>

namespace AtomViz {

class AtomsObject;	// defined in AtomsObject.h

/**
 * \brief Stores information about a class of atoms.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT AtomType : public RefTarget
{
public:

	/// \brief Constructs a new AtomType.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	AtomType(bool isLoading = false);

	/// \brief Gets the types's display name.
	/// \return The human-readable name of the atom type.
	/// \sa setName()
	const QString& name() const { return _name; }

	/// \brief Sets the types's display name.
	/// \param name The new uman-readable name for this atom type.
	/// \undoable
	/// \sa name()
	void setName(const QString& name) { _name = name; }

	/// \brief Returns the color that is assigned to the atoms of this type.
	/// \return The color used for these kinds of atoms at the current animation time.
	/// \sa setColor()
	Color color() const { return colorCtrl ? (Color)colorCtrl->getCurrentValue() : Color(0,0,0); }

	/// \brief Sets the color that is assigned to the atoms of this type.
	/// \param color The new color to be used to display atoms of this kind at the current animation time.
	/// \undoable
	/// \sa color()
	void setColor(const Color& color) { if(colorCtrl) colorCtrl->setCurrentValue(color); }

	/// \brief Returns the controller that controls the color of the atoms that are of this type.
	/// \return The color controller.
	VectorController* colorController() const { return colorCtrl; }

	/// \brief Sets the controller that controls the color of the atoms that are of this type.
	/// \param ctrl The new color controller.
	/// \undoable
	void setColorController(const VectorController::SmartPtr& ctrl) { colorCtrl = ctrl; }

	/// \brief Returns the radius of the atoms.
	/// \return The radius in world units used to display the atoms of this kind at the current animation time.
	/// \sa setAtomRadius()
	FloatType atomRadius() const { return radiusCtrl ? radiusCtrl->getCurrentValue() : (FloatType)1.0; }

	/// \brief Sets the radius of the atoms.
	/// \param newRadius The radius in world units to be used to display this kind of atoms at the current animation time.
	/// \undoable
	/// \sa atomRadius()
	void setAtomRadius(FloatType newRadius) { if(radiusCtrl) radiusCtrl->setCurrentValue(newRadius); }

	/// \brief Returns the controller that controls the radius of the atoms that are of this type.
	/// \return The radius controller
	FloatController* radiusController() const { return radiusCtrl; }

	/// \brief Sets the controller for the radius of the atoms.
	/// \param ctrl The new radius controller
	/// \undoable
	void setAtomRadiusController(const FloatController::SmartPtr& ctrl) { radiusCtrl = ctrl; }

	// From RefTarget class:

	/// Returns the title of this object.
	virtual QString schematicTitle() { return name(); }

public:

	Q_PROPERTY(QString name READ name WRITE setName)
	Q_PROPERTY(Color color READ color WRITE setColor)
	Q_PROPERTY(FloatType atomRadius READ atomRadius WRITE setAtomRadius)

protected:

	/// The name of this atom type.
	PropertyField<QString, QString, SCHEMATIC_TITLE_CHANGED> _name;

	/// This controller stores the color of the atom type.
	ReferenceField<VectorController> colorCtrl;

	/// This controller stores the radius of the atom type.
	ReferenceField<FloatController> radiusCtrl;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(AtomType)
	DECLARE_REFERENCE_FIELD(colorCtrl)
	DECLARE_REFERENCE_FIELD(radiusCtrl)
	DECLARE_PROPERTY_FIELD(_name)
};

/// A list of atom type objects.
typedef QVector<AtomType*> AtomTypeList;

/**
 * \brief A properties editor for the AtomType class.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT AtomTypeEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:
	Q_OBJECT
	DECLARE_PLUGIN_CLASS(AtomTypeEditor)
};

};	// End of namespace AtomViz

#endif // __ATOM_TYPE_H
