///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include "UtilityCommandPage.h"
#include <core/plugins/PluginManager.h>
#include <core/plugins/utility/UtilityPlugin.h>

namespace Core {

/******************************************************************************
* Initializes the utility panel.
******************************************************************************/
UtilityCommandPage::UtilityCommandPage() : CommandPanelPage(),
	currentUtility(NULL), utilitiesButtonGroup(NULL)
{
	scanInstalledPlugins();

	QVBoxLayout* layout = new QVBoxLayout();
	layout->setContentsMargins(2,2,2,2);

	// Create the rollout container.
	rolloutContainer = new RolloutContainer(this);
	rolloutContainer->setFrameStyle(QFrame::NoFrame | QFrame::Plain);
	layout->addWidget(rolloutContainer, 1);

	// Create rollout that displays the list of installed utility plugins.
	utilityListPanel = new QWidget();
	QGridLayout* gridLayout = new QGridLayout();
#ifndef Q_WS_MAC
	gridLayout->setContentsMargins(4,4,4,4);
#endif
	utilityListPanel->setLayout(gridLayout);
	rolloutContainer->addRollout(utilityListPanel, tr("Utilities"));

#ifndef Q_WS_MAC
	utilityListPanel->setStyleSheet("QPushButton:checked { "
							   		"background-color: moccasin; "
							   		"}");
#endif

	setLayout(layout);
	rebuildUtilityList();
}


/******************************************************************************
* Resets the panel to the initial state.
******************************************************************************/
void UtilityCommandPage::reset()
{
	CommandPanelPage::reset();
}

/******************************************************************************
* Is called when the user selects another page.
******************************************************************************/
void UtilityCommandPage::onLeave()
{
	CommandPanelPage::onLeave();
	closeUtility();
}

/******************************************************************************
* Finds all utility classes provided by the installed plugins.
******************************************************************************/
void UtilityCommandPage::scanInstalledPlugins()
{
	// Create an iterator that retrieves all UtilityPlugin derived classes.
	Q_FOREACH(PluginClassDescriptor* clazz, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(UtilityPlugin))) {
		classes.push_back(clazz);
	}
}

/******************************************************************************
*Updates the displayed button in the utility selection panel.
******************************************************************************/
void UtilityCommandPage::rebuildUtilityList()
{
	Q_FOREACH(QObject* w, utilityListPanel->children())
		if(w->isWidgetType()) delete w;
	if(utilitiesButtonGroup) {
		delete utilitiesButtonGroup;
		utilitiesButtonGroup = NULL;
	}

	utilitiesButtonGroup = new QButtonGroup(utilityListPanel);
	utilitiesButtonGroup->setExclusive(false);

	Q_FOREACH(PluginClassDescriptor* descriptor, classes) {

        // Extract display name of the utility from manifest metadata.
        QDomElement descriptionElement = descriptor->getMetaData("Utility-Plugin-Description");

		QString displayName;
		if(!descriptionElement.isNull())
			displayName = descriptionElement.attribute("Display-Name");
		else
			displayName = descriptor->name();

		// Create a button that activates the utility.
		QPushButton* btn = new QPushButton(displayName, utilityListPanel);
		btn->setCheckable(true);
		utilitiesButtonGroup->addButton(btn);
		utilityListPanel->layout()->addWidget(btn);

		// Associate button with the utility plugin class.
		btn->setProperty("ClassDescriptor", qVariantFromValue((void*)descriptor));
	}

	// Listen for events.
	connect(utilitiesButtonGroup, SIGNAL(buttonClicked(QAbstractButton*)), this, SLOT(onUtilityButton(QAbstractButton*)));
	// Resize the rollout panel.
	utilityListPanel->layout()->update();
}

/******************************************************************************
* Is called when the user invokes one of the utility plugins.
******************************************************************************/
void UtilityCommandPage::onUtilityButton(QAbstractButton* button)
{
	PluginClassDescriptor* descriptor = (PluginClassDescriptor*)button->property("ClassDescriptor").value<void*>();
	CHECK_POINTER(descriptor);

	if(button->isChecked() && currentUtility && currentUtility->pluginClassDescriptor() == descriptor) {
		closeUtility();
		currentButton->setChecked(false);
		return;
	}

	// Close previous utility.
	closeUtility();

	try {
		// Create an instance of the utility plugin.
		currentUtility = static_object_cast<UtilityPlugin>(descriptor->createInstance());
		currentButton = button;
		currentButton->setChecked(true);

		currentUtility->openUtility(rolloutContainer, RolloutInsertionParameters().animate());
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
* Closes the current utility.
******************************************************************************/
void UtilityCommandPage::closeUtility()
{
	if(!currentUtility) return;
	CHECK_OBJECT_POINTER(currentUtility.get());
	CHECK_POINTER(currentButton);
	OVITO_ASSERT(currentButton->property("ClassDescriptor").value<void*>() == currentUtility->pluginClassDescriptor());

	// Close the utility.
	currentUtility->closeUtility(rolloutContainer);
	// Deactivate the button.
	currentButton->setChecked(false);

	currentButton = NULL;
	CHECK_OBJECT_POINTER(currentUtility.get());
	currentUtility = NULL;
}

};
