///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/reference/CustomAttributesContainer.h>
#include <core/data/ObjectSaveStream.h>
#include <core/data/ObjectLoadStream.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(CustomAttributesContainer, RefMaker)
DEFINE_VECTOR_REFERENCE_FIELD(CustomAttributesContainer, RefTarget, "Attributes", _attributes)
SET_PROPERTY_FIELD_LABEL(CustomAttributesContainer, _attributes, "Attributes")

/******************************************************************************
* The default constructor. 
******************************************************************************/
CustomAttributesContainer::CustomAttributesContainer(bool isLoading, RefMaker* owner) 
	: RefMaker(isLoading), _owner(owner) 
{
	INIT_PROPERTY_FIELD(CustomAttributesContainer, _attributes);
}

/******************************************************************************
* Inserts the given object into this container. 
******************************************************************************/
void CustomAttributesContainer::addAttribute(RefTarget* data)
{
	CHECK_OBJECT_POINTER(data);
	OVITO_ASSERT_MSG(data->pluginClassDescriptor()->isSerializable(), "CustomAttributesContainer::addAttribute", "Only serializable objects may be used as custom attribute data.");

	if(_attributes.contains(data))
		return;	// It's already registered.
	
	// Create a reference to the attribute data.
	_attributes.push_back(data);
}

/******************************************************************************
* Removes the given object from this container.
******************************************************************************/
void CustomAttributesContainer::removeAttribute(RefTarget* data)
{
	int index = _attributes.indexOf(data);
	if(index == -1) 
		return;
	_attributes.remove(index);
}

/******************************************************************************
* Returns the data that is an instance of the given class
* or any derived class. If there is no such data stored
* in this container then NULL is returned. 
* The returned object can safely be cast to the given type.
******************************************************************************/
RefTarget* CustomAttributesContainer::attributeOfType(PluginClassDescriptor* type) const
{
	CHECK_POINTER(type);
	Q_FOREACH(RefTarget* data, attributes()) {
		if(data->pluginClassDescriptor()->isKindOf(type))
			return data;
	}
	return NULL;
}

/******************************************************************************
* Returns all attributes that are an instance of the given class
* or any derived class. 
* The returned objects can safely be cast to the given type.
******************************************************************************/
QVector<RefTarget*> CustomAttributesContainer::attributesOfType(PluginClassDescriptor* type) const
{
	CHECK_POINTER(type);
	QVector<RefTarget*> list;
	Q_FOREACH(RefTarget* data, attributes()) {
		if(data->pluginClassDescriptor()->isKindOf(type))
			list.push_back(data);
	}
	return list;
}

/******************************************************************************
* This method is called when a reference target changes. To be implemented by sub-classes.
******************************************************************************/
bool CustomAttributesContainer::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(msg->type() == REFTARGET_CHANGED && owner()) {
        owner()->onCustomAttributesChanged();
	}
	return RefMaker::onRefTargetMessage(source, msg);
}

/******************************************************************************
* Is called when a reference target has been added to a list reference field of this RefMaker.
******************************************************************************/
void CustomAttributesContainer::onRefTargetInserted(const PropertyFieldDescriptor& field, RefTarget* newTarget, int listIndex)
{
	RefMaker::onRefTargetInserted(field, newTarget, listIndex);
	if(owner()) owner()->onCustomAttributesChanged();
}

/******************************************************************************
* Is called when a reference target has been removed from a list reference field of this RefMaker.
******************************************************************************/
void CustomAttributesContainer::onRefTargetRemoved(const PropertyFieldDescriptor& field, RefTarget* oldTarget, int listIndex)
{
	RefMaker::onRefTargetRemoved(field, oldTarget, listIndex);
	if(owner()) owner()->onCustomAttributesChanged();
}

/******************************************************************************
* Saves the class' contents to the given stream. 
******************************************************************************/
void CustomAttributesContainer::saveToStream(ObjectSaveStream& stream)
{
	RefMaker::saveToStream(stream);
	stream.saveObject(owner());
}

/******************************************************************************
* Loads the class' contents from the given stream. 
******************************************************************************/
void CustomAttributesContainer::loadFromStream(ObjectLoadStream& stream)
{
	RefMaker::loadFromStream(stream);
	
	// What we do here is a little bit dangerous.
	// The smart pointer to the loaded owner object is thrown away and only a plain pointer is kept.
	// This of course decreases the reference counter of the owner object by one
	// but it isn't deleted because the ObjectLoadStream still holds an internal
	// reference to the loaded owner object.
	// But when the ObjectLoadStream is destructed a new reference to the owner
	// should have been created in the meantime by some third object. 
	_owner = stream.loadObject<RefMaker>().get();
}

};
