
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: event.c,v 1.63 2006/02/07 09:10:07 mschwerin Exp $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdlib.h>

#include "environment.h"
#include "event.h"
#include "gui_utils.h"
#include "filelist_menu.h"
#include "heap.h"
#include "help_menu.h"
#include "i18n.h"
#include "logger.h"
#include "main_menu.h"
#include "oxine.h"
#include "playing_menu.h"
#include "scheduler.h"


static void
do_not_restart_playlist (oxine_t * oxine)
{
    /* Do not restart the playlist. */
    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine,
                              "gui.playlist.current_track", &centry);
    centry.num_value = -1;
    xine_config_update_entry (oxine->xine, &centry);
    xine_config_save (oxine->xine, get_file_config ());
}

/* 
 * ***************************************************************************
 * Name:            oxine_stream_key_handler
 * Access:          private
 *
 * Description:     Handles keypress events.
 * ***************************************************************************
 */
static void
oxine_stream_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    if (!odk_current_is_normal_mode (oxine->odk))
        return;

    /* First we handle all the keys related to stream speed. */
    int old_speed = odk_get_stream_param (oxine->odk, ODK_PARAM_SPEED);
    int new_speed = old_speed;
    switch (ev->source.key) {
        case OXINE_KEY_PPLAY:
        case OXINE_KEY_SELECT:
            if (old_speed == ODK_SPEED_PAUSE)
                new_speed = ODK_SPEED_NORMAL;
            else
                new_speed = ODK_SPEED_PAUSE;
            break;
        case OXINE_KEY_PLAY:
            new_speed = ODK_SPEED_NORMAL;
            break;
        case OXINE_KEY_PAUSE:
            new_speed = ODK_SPEED_PAUSE;
            break;
        case OXINE_KEY_SPEED_UP:
            if (old_speed < ODK_SPEED_FAST_4)
                new_speed <<= 1;
            break;
        case OXINE_KEY_SPEED_DOWN:
            if (old_speed > ODK_SPEED_SLOW_4)
                new_speed >>= 1;
            break;
        default:
            break;
    }
    if (old_speed != new_speed) {
        odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED, new_speed);
        show_stream_parameter (oxine, ODK_PARAM_SPEED);

        if (oxine->control_menu_is_visible) {
            show_control_menu_cb (oxine);
        } else if (oxine->user_interface_is_visible) {
            assert (oxine->repaint_menu);
            oxine->repaint_menu (oxine);
        }
    }

    /* Now we handle all the other stream related keys. */
    switch (ev->source.key) {
        case OXINE_KEY_STOP:
            odk_stop (oxine->odk);
            break;
        case OXINE_KEY_REWIND:
            odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED,
                                  ODK_SPEED_NORMAL);
            odk_change_stream_param (oxine->odk, ODK_PARAM_POSITION,
                                     ev->data.how, 0, 100);
            show_stream_parameter (oxine, ODK_PARAM_POSITION);
            break;
        case OXINE_KEY_FFORWARD:
            odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED,
                                  ODK_SPEED_NORMAL);
            odk_change_stream_param (oxine->odk, ODK_PARAM_POSITION,
                                     ev->data.how, 0, 100);
            show_stream_parameter (oxine, ODK_PARAM_POSITION);
            break;
        case OXINE_KEY_SEEK:
            odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED,
                                  ODK_SPEED_NORMAL);
            odk_set_stream_param (oxine->odk, ODK_PARAM_POSITION,
                                  ev->data.how);
            show_stream_parameter (oxine, ODK_PARAM_POSITION);
            break;
        case OXINE_KEY_PREV:
            if (odk_current_is_dvd (oxine->odk)
                && odk_current_has_chapters (oxine->odk)) {
                odk_xine_event_send (oxine->odk, XINE_EVENT_INPUT_PREVIOUS);
            } else {
                playlist_play_prev (oxine);
            }
            break;
        case OXINE_KEY_NEXT:
            if (odk_current_is_dvd (oxine->odk)
                && odk_current_has_chapters (oxine->odk)) {
                odk_xine_event_send (oxine->odk, XINE_EVENT_INPUT_NEXT);
            } else {
                playlist_play_next (oxine);
            }
            break;
        case OXINE_KEY_VOLUME:
            odk_change_stream_param (oxine->odk, ODK_PARAM_AUDIO_VOLUME,
                                     ev->data.how, 0, 100);
            show_stream_parameter (oxine, ODK_PARAM_AUDIO_VOLUME);
            break;
        case OXINE_KEY_VOLMUTE:
            odk_change_stream_param (oxine->odk, ODK_PARAM_AUDIO_MUTE, 1,
                                     0, 1);
            show_stream_parameter (oxine, ODK_PARAM_AUDIO_MUTE);
            break;
        case OXINE_KEY_MENU_OSD:
            if (!odk_current_has_video (oxine->odk))
                return;
            if (!oxine->user_interface_is_visible) {
                show_playing_menu_cb (oxine);
            } else {
                oxine->backto_menu = clean_otk_stuff;
                clean_otk_stuff (oxine);
                show_stream_parameter (oxine, ODK_PARAM_SPEED);
            }
            break;
        default:
            break;
    }

    if (!odk_current_has_video (oxine->odk))
        return;

    /* Now we handle all keys that are only 
     * interesting if the stream has video. */
    switch (ev->source.key) {
        case OXINE_KEY_ZOOM:
            odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X,
                                     ev->data.how, 0, 400);
            odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y,
                                     ev->data.how, 0, 400);
            show_stream_parameter (oxine, ODK_PARAM_VO_ZOOM_X);
            break;
        case OXINE_KEY_ZOOM_X:
            odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X,
                                     ev->data.how, 0, 400);
            show_stream_parameter (oxine, ODK_PARAM_VO_ZOOM_X);
            break;
        case OXINE_KEY_ZOOM_Y:
            odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y,
                                     ev->data.how, 0, 400);
            show_stream_parameter (oxine, ODK_PARAM_VO_ZOOM_X);
            break;
        case OXINE_KEY_ZOOM_RESET:
            odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X, 100);
            odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y, 100);
            show_stream_parameter (oxine, ODK_PARAM_VO_ZOOM_X);
            break;
        case OXINE_KEY_SATURATION:
            odk_change_stream_param (oxine->odk, ODK_PARAM_VO_SATURATION,
                                     ev->data.how, 0, 65535);
            show_stream_parameter (oxine, ODK_PARAM_VO_SATURATION);
            break;
        case OXINE_KEY_BRIGHTNESS:
            odk_change_stream_param (oxine->odk, ODK_PARAM_VO_BRIGHTNESS,
                                     ev->data.how, 0, 65535);
            show_stream_parameter (oxine, ODK_PARAM_VO_BRIGHTNESS);
            break;
        case OXINE_KEY_CONTRAST:
            odk_change_stream_param (oxine->odk, ODK_PARAM_VO_CONTRAST,
                                     ev->data.how, 0, 65535);
            show_stream_parameter (oxine, ODK_PARAM_VO_CONTRAST);
            break;
        case OXINE_KEY_HUE:
            odk_change_stream_param (oxine->odk, ODK_PARAM_VO_HUE,
                                     ev->data.how, 0, 65535);
            show_stream_parameter (oxine, ODK_PARAM_VO_HUE);
            break;
        case OXINE_KEY_SPU_CHANNEL:
            {
                int max_channels = odk_get_stream_info (oxine->odk,
                                                        ODK_STREAM_INFO_MAX_SPU_CHANNEL);
                if (max_channels > 1) {
                    odk_change_stream_param (oxine->odk,
                                             ODK_PARAM_SPU_CHANNEL,
                                             ev->data.how, -2,
                                             max_channels - 1);
                    show_stream_parameter (oxine, ODK_PARAM_SPU_CHANNEL);
                }
            }
            break;
        case OXINE_KEY_SPU_OFFSET:
            odk_change_stream_param (oxine->odk, ODK_PARAM_SPU_OFFSET,
                                     ev->data.how, -90000, 90000);
            show_stream_parameter (oxine, ODK_PARAM_SPU_OFFSET);
            break;
        case OXINE_KEY_AUDIO_CHANNEL:
            {
                int max_channels = odk_get_stream_info (oxine->odk,
                                                        ODK_STREAM_INFO_MAX_AUDIO_CHANNEL);
                if (max_channels > 1) {
                    odk_change_stream_param (oxine->odk,
                                             ODK_PARAM_AUDIO_CHANNEL,
                                             ev->data.how, -2,
                                             max_channels - 1);
                    show_stream_parameter (oxine, ODK_PARAM_AUDIO_CHANNEL);
                }
            }
            break;
        case OXINE_KEY_AUDIO_OFFSET:
            odk_change_stream_param (oxine->odk,
                                     ODK_PARAM_AUDIO_OFFSET,
                                     ev->data.how, -90000, 90000);
            show_stream_parameter (oxine, ODK_PARAM_AUDIO_OFFSET);
            break;
        case OXINE_KEY_TOGGLE_ASPECT_RATIO:
            odk_change_stream_param (oxine->odk,
                                     ODK_PARAM_VO_ASPECT_RATIO, 1,
                                     ODK_VO_ASPECT_AUTO, ODK_VO_ASPECT_DVB);
            show_stream_parameter (oxine, ODK_PARAM_VO_ASPECT_RATIO);
            break;
        case OXINE_KEY_VO_DEINTERLACE:
            odk_change_stream_param (oxine->odk, ODK_PARAM_VO_DEINTERLACE,
                                     1, 0, 1);
            show_stream_parameter (oxine, ODK_PARAM_VO_DEINTERLACE);
            break;
        default:
            break;
    }
}


/* 
 * ***************************************************************************
 * Name:            oxine_key_handler
 * Access:          private
 *
 * Description:     Handles keypress events.
 * ***************************************************************************
 */
static void
oxine_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    /* Evaluate all the keys related only to the stream. */
    oxine_stream_key_handler (oxine, ev);

    switch (ev->source.key) {
        case OXINE_KEY_BACK:
            if (filelist_menu_is_current_menu (oxine) &&
                (oxine->current_filelist != oxine->toplevel_filelist))
                filelist_menu_level_up (oxine);
            else {
                assert (oxine->backto_menu);
                oxine->backto_menu (oxine);
            }
            break;
        case OXINE_KEY_HELP:
            show_help_menu_cb (oxine);
            break;
        case OXINE_KEY_EJECT:
            eject_cb (oxine);
            break;
        case OXINE_KEY_MENU_MAIN:
            do_not_restart_playlist (oxine);
            show_main_menu_cb (oxine);
            break;
#ifdef HAVE_DVB
        case OXINE_KEY_MENU_DVB:
            do_not_restart_playlist (oxine);
            play_dvb_cb (oxine);
            break;
#endif
#ifdef HAVE_V4L
        case OXINE_KEY_MENU_V4L:
            do_not_restart_playlist (oxine);
            play_v4l_cb (oxine);
            break;
#endif
        case OXINE_KEY_MENU_VIDEO:
            do_not_restart_playlist (oxine);
            show_video_menu_cb (oxine);
            break;
        case OXINE_KEY_MENU_MUSIC:
            do_not_restart_playlist (oxine);
            show_music_menu_cb (oxine);
            break;
#ifdef HAVE_IMAGE_TOOLS
        case OXINE_KEY_MENU_IMAGE:
            do_not_restart_playlist (oxine);
            show_image_menu_cb (oxine);
            break;
#endif
        case OXINE_KEY_FULLSCREEN:
            if (odk_is_fullscreen (oxine->odk))
                odk_set_fullscreen (oxine->odk, 0);
            else
                odk_set_fullscreen (oxine->odk, 1);
            break;
        case OXINE_KEY_SHUTDOWN:
            shutdown_cb (oxine);
            break;
        default:
            break;
    }
}

/* 
 * ***************************************************************************
 * Name:            oxine_motion_handler
 * Access:          public
 *
 * Description:     Handles mouse movements
 * ***************************************************************************
 */
static void
oxine_motion_handler (oxine_t * oxine, oxine_event_t * event)
{
    if (odk_current_is_normal_mode (oxine->odk)
        && !oxine->user_interface_is_visible) {

        if (!oxine->control_menu_is_visible && (event->data.where.y > 580)) {
            show_control_menu_cb (oxine);
        }

        if (oxine->control_menu_is_visible && (event->data.where.y < 450)) {
            clean_otk_stuff (oxine);
        }

        if (oxine->control_menu_is_visible && (event->data.where.y > 450)) {
            cancel_job (oxine->otk_clear_job);
            oxine->otk_clear_job = schedule_job (6000, (void (*)(void *))
                                                 clean_otk_stuff, oxine);
        }
    }
}

/* 
 * ***************************************************************************
 * Name:            oxine_event_handler
 * Access:          public
 *
 * Description:     This is the main event handler.
 * ***************************************************************************
 */
void
oxine_event_handler (void *oxine_p, oxine_event_t * event)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

#ifdef DEBUG_EVENTS
    switch (event->type) {
        case OXINE_EVENT_KEY:
            debug ("got OXINE_EVENT_KEY %d", event->source.key);
            break;
        case OXINE_EVENT_BUTTON:
            debug ("got OXINE_EVENT_BUTTON %d +%d+%d", event->source.button,
                   event->data.where.x, event->data.where.y);
            break;
        case OXINE_EVENT_MOTION:
            debug ("got OXINE_EVENT_MOTION +%d+%d", event->data.where.x,
                   event->data.where.y);
            break;
        case OXINE_EVENT_PLAYBACK_STARTED:
            debug ("got OXINE_EVENT_PLAYBACK_STARTED");
            break;
        case OXINE_EVENT_PLAYBACK_STOPPED:
            debug ("got OXINE_EVENT_PLAYBACK_STOPPED");
            break;
        case OXINE_EVENT_PLAYBACK_FINISHED:
            debug ("got OXINE_EVENT_PLAYBACK_FINISHED");
            break;
        default:
            break;
    }
#endif

    switch (event->type) {
        case OXINE_EVENT_KEY:
            oxine_key_handler (oxine, event);
            break;
        case OXINE_EVENT_MOTION:
            oxine_motion_handler (oxine, event);
            break;
        case OXINE_EVENT_PLAYBACK_STARTED:
            if (odk_current_is_normal_mode (oxine->odk)) {
                oxine->backto_menu = clean_otk_stuff;

                if (!odk_current_has_video (oxine->odk)) {
                    show_playing_menu_cb (oxine);
                } else {
                    clean_otk_stuff (oxine);

                    if (!odk_current_is_dvd (oxine->odk)
                        && !odk_current_is_vcd (oxine->odk)
                        && !odk_current_is_dvb (oxine->odk)) {
                        show_stream_info_cb (oxine);
                        cancel_job (oxine->otk_clear_job);
                        oxine->otk_clear_job
                            = schedule_job (10000, (void (*)(void *))
                                            clean_otk_stuff, oxine);
                    }
                }
            }
            break;
        case OXINE_EVENT_PLAYBACK_STOPPED:
            {
                do_not_restart_playlist (oxine);

                assert (oxine->playback_ended_menu);
                oxine->playback_ended_menu (oxine);
            }
            break;
        case OXINE_EVENT_PLAYBACK_FINISHED:
            if (odk_current_is_normal_mode (oxine->odk)) {
                playlist_play_next (oxine);
            }
            break;
        case OXINE_EVENT_PLAYBACK_ERROR:
            {
                could_not_play (oxine, (otk_button_cb_t) playlist_play_next,
                                oxine->playback_ended_menu);
            }
            break;
        case OXINE_EVENT_WAITING_FOR_ALTERNATIVE:
            {
                show_message (oxine, _("Waiting for alternative stream..."));
            }
            break;
        default:
            break;
    }
}
