
/*
 * Copyright (C) 2002-2003 Stefan Holst
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: lirc.c,v 1.50 2006/01/17 14:11:27 mschwerin Exp $
 *
 */
#include "config.h"

#ifdef HAVE_LIRC
#include <lirc/lirc_client.h>
#endif

#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <sys/types.h>
#include <regex.h>

#include "heap.h"
#include "i18n.h"
#include "environment.h"
#include "lirc.h"
#include "logger.h"
#include "odk.h"

typedef struct {
    odk_input_t class;

    odk_t *odk;
    void (*event_handler) (void *data, oxine_event_t * ev);

    int lirc_fd;
    struct lirc_config *lircconfig;
} lirc_input_t;

#ifdef DEBUG
#define LIRC_VERBOSE 1
#else
#define LIRC_VERBOSE 0
#endif

#ifdef HAVE_LIRC
static int
extract_how (char *config, const char *regex)
{
    int how = 0;

    char str[255];
    memset (str, 0, 255);

    regex_t reg;
    regmatch_t match[3];
    if ((regcomp (&reg, regex, REG_EXTENDED | REG_ICASE) == 0) &&
        (regexec (&reg, config, 3, match, 0) == 0)) {

        strncpy (str, config + match[1].rm_so,
                 match[1].rm_eo - match[1].rm_so);

        if ((str[0] == '+') && (str[1] == '\0')) {
            how = 1;
        }

        else if ((str[0] == '+') && (str[1] == '\0')) {
            how = -1;
        }

        else {
            how = atoi (str);
        }
    }

    regfree (&reg);

    return how;
}

static void *
lirc_thread (void *this)
{
    lirc_input_t *li = (lirc_input_t *) this;

    while (1) {
        pthread_testcancel ();

        char *code;
        while (lirc_nextcode (&code) == 0) {
            pthread_testcancel ();

            if (code == NULL)
                break;

            int ret;
            char *config;
            while ((ret = lirc_code2char (li->lircconfig, code, &config) == 0)
                   && (config != NULL)) {
                pthread_testcancel ();

                debug ("received command: '%s'", config);

                oxine_event_t ev;
                ev.type = OXINE_EVENT_KEY;
                ev.source.key = 0;

                if (!strcasecmp (config, "up"))
                    ev.source.key = OXINE_KEY_UP;
                else if (!strcasecmp (config, "down"))
                    ev.source.key = OXINE_KEY_DOWN;
                else if (!strcasecmp (config, "left"))
                    ev.source.key = OXINE_KEY_LEFT;
                else if (!strcasecmp (config, "right"))
                    ev.source.key = OXINE_KEY_RIGHT;
                else if (!strcasecmp (config, "select"))
                    ev.source.key = OXINE_KEY_SELECT;
                else if (!strcasecmp (config, "back"))
                    ev.source.key = OXINE_KEY_BACK;
                else if (!strcasecmp (config, "activate"))
                    ev.source.key = OXINE_KEY_ACTIVATE;
                else if (!strcasecmp (config, "stop"))
                    ev.source.key = OXINE_KEY_STOP;
                else if (!strcasecmp (config, "pplay"))
                    ev.source.key = OXINE_KEY_PPLAY;
                else if (!strcasecmp (config, "play"))
                    ev.source.key = OXINE_KEY_PLAY;
                else if (!strcasecmp (config, "pause"))
                    ev.source.key = OXINE_KEY_PAUSE;
                else if (!strcasecmp (config, "next"))
                    ev.source.key = OXINE_KEY_NEXT;
                else if (!strcasecmp (config, "prev"))
                    ev.source.key = OXINE_KEY_PREV;
                else if (!strcasecmp (config, "eject"))
                    ev.source.key = OXINE_KEY_EJECT;
                else if (!strcasecmp (config, "toggle_aspect_ratio"))
                    ev.source.key = OXINE_KEY_TOGGLE_ASPECT_RATIO;
                else if (!strcasecmp (config, "toggle_deinterlace"))
                    ev.source.key = OXINE_KEY_VO_DEINTERLACE;
                else if (!strcasecmp (config, "volmute"))
                    ev.source.key = OXINE_KEY_VOLMUTE;
                else if (!strcasecmp (config, "shutdown"))
                    ev.source.key = OXINE_KEY_SHUTDOWN;
                else if (!strcasecmp (config, "help"))
                    ev.source.key = OXINE_KEY_HELP;
                else if (!strcasecmp (config, "osdmenu"))
                    ev.source.key = OXINE_KEY_MENU_OSD;
                else if (!strcasecmp (config, "mainmenu"))
                    ev.source.key = OXINE_KEY_MENU_MAIN;
                else if (!strcasecmp (config, "v4l")
                         || !strcasecmp (config, "tv"))
                    ev.source.key = OXINE_KEY_MENU_V4L;
                else if (!strcasecmp (config, "dvb"))
                    ev.source.key = OXINE_KEY_MENU_DVB;
                else if (!strcasecmp (config, "music"))
                    ev.source.key = OXINE_KEY_MENU_MUSIC;
                else if (!strcasecmp (config, "video"))
                    ev.source.key = OXINE_KEY_MENU_VIDEO;
                else if (!strcasecmp (config, "image"))
                    ev.source.key = OXINE_KEY_MENU_IMAGE;
                else if (!strcasecmp (config, "menu1"))
                    ev.source.key = OXINE_KEY_MENU1;
                else if (!strcasecmp (config, "menu2"))
                    ev.source.key = OXINE_KEY_MENU2;
                else if (!strcasecmp (config, "menu3"))
                    ev.source.key = OXINE_KEY_MENU3;
                else if (!strcasecmp (config, "menu4"))
                    ev.source.key = OXINE_KEY_MENU4;
                else if (!strcasecmp (config, "menu5"))
                    ev.source.key = OXINE_KEY_MENU5;
                else if (!strcasecmp (config, "menu6"))
                    ev.source.key = OXINE_KEY_MENU6;
                else if (!strcasecmp (config, "menu7"))
                    ev.source.key = OXINE_KEY_MENU7;

                else if (!strncasecmp (config, "volume", 6)) {
                    ev.source.key = OXINE_KEY_VOLUME;
                    ev.data.how = extract_how (config, "volume ([+-][0-9]*)");
                }

                else if (!strcasecmp (config, "fforward")) {
                    ev.source.key = OXINE_KEY_FFORWARD;
                    ev.data.how = +30;
                }

                else if (!strcasecmp (config, "rewind")) {
                    ev.source.key = OXINE_KEY_REWIND;
                    ev.data.how = -30;
                }

                else if (!strcasecmp (config, "speedup")) {
                    ev.source.key = OXINE_KEY_SPEED_UP;
                    ev.data.how = +1;
                }

                else if (!strcasecmp (config, "speeddown")) {
                    ev.source.key = OXINE_KEY_SPEED_DOWN;
                    ev.data.how = -1;
                }

                else if (!strncasecmp (config, "audio_channel", 13)) {
                    ev.data.how = extract_how (config,
                                               "audio_channel ([+-][0-9]*)");
                    ev.source.key = OXINE_KEY_AUDIO_CHANNEL;
                }

                else if (!strncasecmp (config, "audio_offset", 9)) {
                    ev.data.how = extract_how (config,
                                               "audio_offset ([+-][0-9]*)");
                    ev.source.key = OXINE_KEY_AUDIO_OFFSET;
                }

                else if (!strncasecmp (config, "spu_channel", 11)) {
                    ev.data.how = extract_how (config,
                                               "spu_channel ([+-][0-9]*)");
                    ev.source.key = OXINE_KEY_SPU_CHANNEL;
                }

                else if (!strncasecmp (config, "spu_offset", 10)) {
                    ev.data.how = extract_how (config,
                                               "spu_offset ([+-][0-9]*)");
                    ev.source.key = OXINE_KEY_SPU_OFFSET;
                }

                else if (!strncasecmp (config, "hue", 3)) {
                    ev.data.how = extract_how (config, "hue ([+-][0-9]*)");
                    ev.source.key = OXINE_KEY_HUE;
                }

                else if (!strncasecmp (config, "saturation", 10)) {
                    ev.data.how = extract_how (config,
                                               "saturation ([+-][0-9]*)");
                    ev.source.key = OXINE_KEY_SATURATION;
                }

                else if (!strncasecmp (config, "brightness", 10)) {
                    ev.data.how = extract_how (config,
                                               "brightness ([+-][0-9]*)");
                    ev.source.key = OXINE_KEY_BRIGHTNESS;
                }

                else if (!strncasecmp (config, "contrast", 8)) {
                    ev.data.how = extract_how (config,
                                               "contrast ([+-][0-9]*)");
                    ev.source.key = OXINE_KEY_CONTRAST;
                }

                else if (!strcasecmp (config, "volup")) {
                    // deprecated in version 0.4.5
                    warn (_("The LIRC command 'volup' has "
                            "been deprecated!"));
                    info (_("Please use 'volume' instead."));
                    ev.source.key = OXINE_KEY_VOLUME;
                    ev.data.how = +5;
                }

                else if (!strcasecmp (config, "voldown")) {
                    // deprecated in version 0.4.5
                    warn (_("The LIRC command 'voldown' has "
                            "been deprecated!"));
                    info (_("Please use 'volume' instead."));
                    ev.source.key = OXINE_KEY_VOLUME;
                    ev.data.how = -5;
                }

                else if (!strncasecmp (config, "audio_channel_logical", 21)) {
                    // deprecated in version 0.4.5
                    warn (_("The LIRC command 'audio_channel_logical' has "
                            "been deprecated!"));
                    info (_("Please use 'audio_channel' instead."));
                    ev.data.how = extract_how (config,
                                               "audio_channel_logical ([+-][0-9]*)");
                    ev.source.key = OXINE_KEY_AUDIO_CHANNEL;
                }

                if (ev.source.key)
                    li->event_handler (li->odk, &ev);

            }
            free (code);

            if (ret == -1)
                break;
        }
    }

    pthread_exit (NULL);
    return NULL;
}
#endif /* HAVE_LIRC */

odk_input_t *
start_lirc (odk_t * odk, odk_event_handler_t event_handler)
{
#ifdef HAVE_LIRC
    lirc_input_t *li = ho_new (lirc_input_t);
    li->odk = odk;
    li->event_handler = event_handler;

    /* Initialize LIRC client. */
    if ((li->lirc_fd = lirc_init ("oxine", LIRC_VERBOSE)) == -1) {
        error (_("Failed to initialize LIRC."));

        info (_("Make sure you have lircd running."));
        info (_("Make sure you have the permissions "
                "to connect to the LIRC socket."));

        ho_free (li);
        return NULL;
    }

    /* Try to read configuration. First we try the configuration in the users
     * oxine config directory and then we try the standard configuration.
     */
    const char *filename = get_file_lirc_config ();
    if (lirc_readconfig ((char *) filename, &li->lircconfig, NULL) != 0) {
        warn (_("Could not find %s, trying to use default "
                "configuration file %s/lircrc." ""), filename, OXINE_DATADIR);

        if (lirc_readconfig (OXINE_DATADIR "/lircrc",
                             &li->lircconfig, NULL) != 0) {
            error (_("Could not find a LIRC configuration file."));

            lirc_deinit ();
            ho_free (li);
            return NULL;
        }
    }

    /* Create the thread to handle LIRC commands. */
    if (pthread_create (&li->class.thread, NULL, lirc_thread, li) != 0) {
        error (_("Could not create LIRC thread: %s!"), strerror (errno));

        lirc_deinit ();
        ho_free (li);
        return NULL;
    }

    info (_("Successfully started the LIRC input plugin."));

    return (odk_input_t *) li;
#else
    debug ("LIRC support disabled at compile time!");
    return NULL;
#endif
}

void
stop_lirc (odk_input_t * oi)
{
#ifdef HAVE_LIRC
    lirc_input_t *li = (lirc_input_t *) oi;
    pthread_cancel (oi->thread);

    lirc_freeconfig (li->lircconfig);
    lirc_deinit ();
    ho_free (li);

    info (_("Successfully stopped the LIRC input plugin."));
#endif
}
