
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: mostlistened.c,v 1.8 2005/12/06 21:35:45 mschwerin Exp $
 *
 */
#include "config.h"

#include <limits.h>
#include <stdio.h>
#include <string.h>

#include "codeset.h"
#include "environment.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "mediamarks.h"
#include "mostlistened.h"
#include "oxine.h"

#define OXINE_MM_XML "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
#define OXINE_MM_DTD "<!DOCTYPE oxinemm SYSTEM \"http://oxine.sf.net/DTD/favorites.dtd\">\n\n"

static int
mostlistened_swap_cb (void *d1, void *d2)
{
    fileitem_t *f1 = (fileitem_t *) d1;
    fileitem_t *f2 = (fileitem_t *) d2;

    return (f2->type == FILE_TYPE_UPLINK)
        || ((f2->listened_count > f1->listened_count)
            && (f1->type != FILE_TYPE_UPLINK));
}

void
mostlistened_add_directory (oxine_t * oxine, const char *title,
                            const char *mrl)
{
    int found = 0;
    fileitem_t *current = filelist_first (oxine->mostlistened_filelist);
    while (current) {
        if (current->type == FILE_TYPE_DIRECTORY
            && strcmp (current->mrl, mrl) == 0) {
            current->listened_count++;
            found = 1;
            break;
        }
        current = filelist_next (oxine->mostlistened_filelist, current);
    }
    if (!found) {
        fileitem_t *new =
            filelist_add (oxine->mostlistened_filelist, title, mrl,
                          FILE_TYPE_DIRECTORY);
        new->listened_count++;
    }

    filelist_sort (oxine->mostlistened_filelist, mostlistened_swap_cb);

    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init (codeset, "utf8");

    FILE *f = fopen (get_file_favorites (), "w");
    if (f) {
        fprintf (f, OXINE_MM_XML);
        fprintf (f, OXINE_MM_DTD);
        fprintf (f, "<oxinemm>\n");
        fileitem_t *current = filelist_first (oxine->mostlistened_filelist);
        while (current) {

            if (current->type == FILE_TYPE_DIRECTORY) {
                char *utf8_mrl = recode (xr, current->mrl);
                char *utf8_title = recode (xr, current->title);
                fprintf (f, "    <entry>\n");
                fprintf (f, "        <mrl href=\"%s\" />\n", utf8_mrl);
                fprintf (f, "        <title>%s</title>\n", utf8_title);
                fprintf (f, "        <listened_count>%d</listened_count>\n",
                         current->listened_count);
                fprintf (f, "  </entry>\n");
                ho_free (utf8_mrl);
                ho_free (utf8_title);
            }

            current = filelist_next (oxine->mostlistened_filelist, current);
        }
        fprintf (f, "</oxinemm>\n");
        fclose (f);
    }

    recode_done (xr);
    if (codeset)
        ho_free (codeset);
}

void
mostlistened_new (oxine_t * oxine)
{
    fileitem_t *fileitem =
        filelist_add (oxine->toplevel_filelist, _("[Favourites]"),
                      get_file_favorites (),
                      FILE_TYPE_MEDIAMARKS);

    filelist_expand (oxine->toplevel_filelist, fileitem);
    oxine->mostlistened_filelist = fileitem->sublist;

    filelist_sort (oxine->mostlistened_filelist, mostlistened_swap_cb);
    filelist_sort (oxine->toplevel_filelist, NULL);
}
