
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_private.h,v 1.10 2006/01/18 18:01:18 mschwerin Exp $
 *
 * This file contains private types and methods used inside the odk system.
 * This file should not be included outside the odk system.
 *
 */

#ifndef HAVE_ODK_PRIVATE_H
#define HAVE_ODK_PRIVATE_H

#include "odk.h"
#include "playlist.h"
#include "odk_plugin.h"

/* 
 * ***************************************************************************
 * Description:     Width and height of the OSD.
 * ***************************************************************************
 */
#define V_WIDTH 800
#define V_HEIGHT 600

/* 
 * ***************************************************************************
 * Description:     Number of colors (defined by the xine engine).
 * ***************************************************************************
 */
#define NUM_COLORS 256

/* 
 * ***************************************************************************
 * Description:     A palette containing max. 256 colors
 * ***************************************************************************
 */
typedef struct {
    int num_colors;
    uint32_t colors[NUM_COLORS];
    uint8_t transparency[NUM_COLORS];
} odk_palette_t;

/* 
 * ***************************************************************************
 * Description:     An OSD image.
 * ***************************************************************************
 */
typedef struct {
    char *mrl;
    xine_osd_t *osd;
} odk_osd_image_t;

/*
 * ***************************************************************************
 * Description:     The ODK class structure.
 * ***************************************************************************
 */
struct odk_s {
    xine_t *xine;
    xine_osd_t *osd;

    odk_osd_image_t *image_cache;

    double vscale;
    double hscale;

    /* The output window. */
    odk_window_t *win;

    /* The event handler. */
    void (*event_handler) (void *data, oxine_event_t * ev);
    void *event_handler_data;

    /* The palette */
    odk_palette_t *palette;

    /* The input plugins for joystick and LIRC support. */
    odk_input_t *js;
    odk_input_t *lirc;

    /* Configuration */
    int image_change_delay;
    int visual_anim_style;
    int use_unscaled_osd;

    /* Some status variables. */
    int is_seeking_in_stream;
    int is_osd_visible;
    int is_current_font_freetype;
    int is_unscaled_osd_available;
    int forward_events_to_xine;

    /* Info about current stream */
    odk_mode_t current_mode;
    char *current_mrl;
    char *current_sub_mrl;
    char *current_title;
    playlist_t *current_alternatives;

    /* Post plugin and background/ animation streams for streams that don't
     * have video. */
    xine_post_t *novideo_post;

    xine_stream_t *background_stream;
    const char *background_stream_mrl;

    xine_stream_t *animation_stream;
    const char *animation_stream_mrl;

    /* Subtitle stream */
    xine_stream_t *subtitle_stream;

    /* The ID of the image finish job. */
    int image_finish_job;

    /* A mutex we use to allow only one instance of mutexed_play to be run at
     * a time. */
#ifdef DEBUG_THREADS
    pthread_mutexattr_t play_mutex_attr;
#endif
    pthread_mutex_t play_mutex;
};

int is_image (const char *mrl);

/* Post plugins */
void odk_post_init (odk_t * odk);
void odk_post_audio_rewire (odk_t * odk);
void odk_post_audio_unwire (odk_t * odk);

/* OSD */
void odk_osd_init (odk_t * odk);
void odk_osd_free (odk_t * odk);

void odk_get_output_size (odk_t * odk, int *width, int *height);
void odk_get_frame_size (odk_t * odk, int *w, int *h);

void odk_osd_adapt_size (odk_t * odk);
void odk_osd_free_image (odk_osd_image_t * osd_image);

/* Event handling */
void odk_event_init (odk_t * odk);
void odk_event_free (odk_t * odk);

#endif
